function md=solve(md,varargin)
%SOLVE - apply solution sequence for this model
%
%   Usage:
%      md=solve(md,varargin)
%      where varargin is a lit of paired arguments. 
%      arguments can be: 'analysis_type': 'diagnostic','thermal','prognostic','transient'
%      arguments can be: 'sub_analysis_type': 'transient','steady','' (default if empty = 'steady')
%      arguments can be: 'package': 'macayeal','ice','cielo' (default if not specified = 'cielo')
%
%   Examples:
%      md=solve(md,'analysis_type','diagnostic','package','cielo');
%      md=solve(md,'analysis_type','control','package','ice');
%      md=solve(md,'analysis_type','thermal','sub_analysis_type','transient','package','ice');
%      md=solve(md,'analysis_type','thermal','sub_analysis_type','steady','package','cielo');
%      md=solve(md,'analysis_type','thermal','package','cielo');

%some checks on list of arguments
global ISSM_DIR

%recover options
options=recover_options(varargin{:});

%add default options
options=process_solve_options(options);

%recover some fields
md.analysis_type=options.analysis_type;
md.sub_analysis_type=options.sub_analysis_type;
package=options.package;

%Use package to set solution namespace
usenamespace(package);

if ~ismodelselfconsistent(md,package),
	error(' '); %previous error messages should explain what is going on.
end

%preprocesses model before solving
md=presolve(md);

%if running qmu analysis, some preprocessing of dakota files using 
%models fields needs to be carried out. 
if md.qmu_analysis,
	md=preqmu(md,options);
end
displaystring(md.debug,'\n%s\n','launching solution sequence');

%If running in parallel, we have a different way of launching the solution
%sequences. 
if ~strcmpi(md.cluster,'none'),
	md=solveparallel(md,options);
	return;
end

%Launch correct solution sequence
if md.analysis_type==DiagnosticAnalysisEnum,
	md=diagnostic(md);

elseif md.analysis_type==TransientAnalysisEnum,
	md=transient(md);

elseif md.analysis_type==Mesh2gridAnalysisEnum,
	md=mesh2grid(md);;

elseif md.analysis_type==PrognosticAnalysisEnum,
	md=prognostic(md);;

elseif md.analysis_type==ControlAnalysisEnum,
	md=control(md);

elseif md.analysis_type==ThermalAnalysisEnum,
	md=thermal(md);

elseif md.analysis_type==ParametersAnalysisEnum,
	md=parameters(md);

else
	error('solution type not supported for this model configuration.');
end

%post processes qmu results if necessary
if md.qmu_analysis,
	md=postqmu(md);
	cd ..
end

%Check result is consistent
displaystring(md.debug,'%s\n','checking result consistency');
if ~isresultconsistent(md,options.analysis_type),
	disp('!! results not consistent correct the model !!') %it would be very cruel to put an error, it would kill the computed results (even if not consistent...)
end

%re-add the paths
addpath(genpath_ice([ISSM_DIR '/src/m/solutions/macayeal']));
addpath(genpath_ice([ISSM_DIR '/src/m/solutions/ice']));
addpath(genpath_ice([ISSM_DIR '/src/m/solutions/cielo']));
addpath(genpath_ice([ISSM_DIR '/bin']));
