/*
 * \file Elements.cpp
 * \brief: Implementation of Elements class, derived from DataSet class
 */

/*Headers: {{{*/
#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "./Element.h"
#include "./Elements.h"
#include "../Params/Parameters.h"
#include "../ExternalResults/Results.h"
#include "../ExternalResults/GenericExternalResult.h"
#include "../../toolkits/toolkits.h"
#include "../../shared/shared.h"

using namespace std;
/*}}}*/

/*Object constructors and destructor*/
/*FUNCTION Elements::Elements(){{{*/
Elements::Elements(){
	enum_type=MeshElementsEnum;
	return;
}
/*}}}*/
/*FUNCTION Elements::~Elements(){{{*/
Elements::~Elements(){
	return;
}
/*}}}*/

/*Object management*/
/*FUNCTION Elements::Configure{{{*/
void Elements::Configure(Elements* elements,Loads* loads, Nodes* nodes, Vertices* vertices, Materials* materials,Parameters* parameters){

	vector<Object*>::iterator object;
	Element* element=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		element=dynamic_cast<Element*>((*object));
		element->Configure(elements,loads,nodes,vertices,materials,parameters);

	}

}
/*}}}*/
/*FUNCTION Elements::DeleteResults{{{*/
void Elements::DeleteResults(void){

	for (int i=0;i<this->Size();i++){
		Element* element=dynamic_cast<Element*>(this->GetObjectByOffset(i));
		element->DeleteResults();
	}
}
/*}}}*/
/*FUNCTION Elements::SetCurrentConfiguration{{{*/
void Elements::SetCurrentConfiguration(Elements* elements,Loads* loads, Nodes* nodes, Vertices* vertices, Materials* materials,Parameters* parameters){

	vector<Object*>::iterator object;
	Element* element=NULL;

	for ( object=objects.begin() ; object < objects.end(); object++ ){

		element=dynamic_cast<Element*>((*object));
		element->SetCurrentConfiguration(elements,loads,nodes,materials,parameters);

	}

}
/*}}}*/
/*FUNCTION Elements::ToResults{{{*/
void Elements::ToResults(Results* results,Parameters* parameters){

	int my_rank;
	int num_procs;

	int                 *resultsenums       = NULL;
	int                 *resultssizes       = NULL;
	int                 *resultssteps       = NULL;
	IssmDouble          *resultstimes       = NULL;
	IssmDouble          *vector_serial      = NULL;
	Vector<IssmDouble> *vector = NULL;
	bool                io_gather;
	int                 numberofvertices,numberofelements;
	int                 numberofresults ,vectorsize;
	int                 rank;
	int                 minrank;

	/*recover my_rank:*/
	my_rank=IssmComm::GetRank();
	num_procs=IssmComm::GetSize();

	/*Recover parameters: */
	parameters->FindParam(&io_gather,SettingsIoGatherEnum);
	parameters->FindParam(&numberofvertices,MeshNumberofverticesEnum);
	parameters->FindParam(&numberofelements,MeshNumberofelementsEnum);

	/*Get rank of first cpu that has results*/
	if(this->Size()) rank=my_rank;
	else rank=num_procs;
	ISSM_MPI_Allreduce (&rank,&minrank,1,ISSM_MPI_INT,ISSM_MPI_MIN,IssmComm::GetComm());

	/*see what the first element of this partition has in stock (this is common to all partitions)*/
	if(my_rank==minrank){
		if(this->Size()==0) _error_("Cannot write results because there is no element??");
		Element* element=dynamic_cast<Element*>(this->GetObjectByOffset(0));
		element->ListResultsInfo(&resultsenums,&resultssizes,&resultstimes,&resultssteps,&numberofresults);
	}
	ISSM_MPI_Bcast(&numberofresults,1,ISSM_MPI_INT,minrank,IssmComm::GetComm());

	/*Get out if there is no results. Otherwise broadcast info*/
	if(!numberofresults) return;
	if(my_rank!=minrank){
		resultsenums=xNew<int>(numberofresults);
		resultssizes=xNew<int>(numberofresults);
		resultstimes=xNew<IssmDouble>(numberofresults);
		resultssteps=xNew<int>(numberofresults);
	}
	ISSM_MPI_Bcast(resultsenums,numberofresults,ISSM_MPI_INT,minrank,IssmComm::GetComm());
	ISSM_MPI_Bcast(resultssizes,numberofresults,ISSM_MPI_INT,minrank,IssmComm::GetComm());
	ISSM_MPI_Bcast(resultstimes,numberofresults,ISSM_MPI_DOUBLE,minrank,IssmComm::GetComm());
	ISSM_MPI_Bcast(resultssteps,numberofresults,ISSM_MPI_INT,minrank,IssmComm::GetComm());

	/*Loop over all results and get nodal vector*/
	for(int i=0;i<numberofresults;i++){

		/*Get vector for result number i*/
		if(resultssizes[i]==P1Enum)      vectorsize=numberofvertices;
		else if(resultssizes[i]==P0Enum) vectorsize=numberofelements;
		else _error_("Unkown result size: " << EnumToStringx(resultssizes[i]));
		vector=new Vector<IssmDouble>(vectorsize);

		for(int j=0;j<this->Size();j++){
			Element* element=dynamic_cast<Element*>(this->GetObjectByOffset(j));
			element->GetVectorFromResults(vector,i,resultsenums[i],resultssizes[i]);
		}
		vector->Assemble();

		/*Serialize and add to results*/
		vector_serial=vector->ToMPISerial();
		results->DeleteResult(resultsenums[i],resultssteps[i]);
		if(my_rank==0){
			/*No need to add this vector for all cpus*/
#ifdef _HAVE_ADOLC_
			IssmPDouble* vector_serial_passive=xNew<IssmPDouble>(vectorsize);
			for(int k=0;k<vectorsize;k++)vector_serial_passive[k]=reCast<IssmPDouble>(vector_serial[k]);
			results->AddResult(new GenericExternalResult<IssmPDouble*>(results->Size()+1,resultsenums[i],vector_serial_passive,vectorsize,1,resultssteps[i],resultstimes[i]));
			xDelete<IssmPDouble>(vector_serial_passive);
#else
			results->AddResult(new GenericExternalResult<IssmPDouble*>(results->Size()+1,resultsenums[i],vector_serial,vectorsize,1,resultssteps[i],resultstimes[i]));
#endif
		}

		/*clean up*/
		delete vector;
		xDelete<IssmDouble>(vector_serial);
	}

	/*Free ressources:*/
	xDelete<int>(resultsenums);
	xDelete<int>(resultssizes);
	xDelete<int>(resultssteps);
	xDelete<IssmDouble>(resultstimes);
}
/*}}}*/
/*FUNCTION Elements::MaxNumNodes{{{*/
int Elements::MaxNumNodes(void){

	int max=0;
	int allmax;
	int numnodes=0;

	/*Now go through all elements, and get how many nodes they own, unless they are clone nodes: */
	for(int i=0;i<this->Size();i++){

		Element* element=dynamic_cast<Element*>(this->GetObjectByOffset(i));
		numnodes=element->GetNumberOfNodes();
		if(numnodes>max)max=numnodes;
	}

	/*Grab max of all cpus: */
	ISSM_MPI_Allreduce((void*)&max,(void*)&allmax,1,ISSM_MPI_INT,ISSM_MPI_MAX,IssmComm::GetComm());
	max=allmax;

	return max;
}
/*}}}*/
/*FUNCTION Elements::NumberOfElements{{{*/
int Elements::NumberOfElements(void){

	int local_nelem;
	int numberofelements;

	local_nelem=this->Size();
	ISSM_MPI_Allreduce((void*)&local_nelem,(void*)&numberofelements,1,ISSM_MPI_INT,ISSM_MPI_SUM,IssmComm::GetComm());

	return numberofelements;
}
/*}}}*/
/*FUNCTION Elements::InputDuplicate{{{*/
void Elements::InputDuplicate(int input_enum,int output_enum){

	for(int i=0;i<this->Size();i++){
		Element* element=dynamic_cast<Element*>(this->GetObjectByOffset(i));
		element->InputDuplicate(input_enum,output_enum);
	}
}
/*}}}*/
