import numpy
from fielddisplay import fielddisplay
from EnumDefinitions import *
from checkfield import *
from MatlabFuncs import *

class mesh(object):
	"""
	MESH class definition

	   Usage:
	      mesh=mesh();
	"""

	def __init__(self): # {{{
		self.x                           = float('NaN');
		self.y                           = float('NaN');
		self.elements                    = float('NaN');
		self.numberofelements            = 0;
		self.numberofvertices            = 0;
		self.numberofedges               = 0;
		
		self.lat                         = float('NaN');
		self.long                        = float('NaN');
		self.hemisphere                  = float('NaN');

		self.vertexonboundary            = float('NaN');
		self.edges                       = float('NaN');
		self.segments                    = float('NaN');
		self.segmentmarkers              = float('NaN');
		self.vertexconnectivity          = float('NaN');
		self.elementconnectivity         = float('NaN');
		self.average_vertex_connectivity = 0;

		self.extractedvertices           = float('NaN');
		self.extractedelements           = float('NaN');

		#set defaults
		self.setdefaultparameters()

		#}}}
	def __repr__(self): # {{{
		string="   2d Mesh:" 

		string="%s\n%s"%(string,"\n      Elements and vertices:")
		string="%s\n%s"%(string,fielddisplay(self,"numberofelements","number of elements"))
		string="%s\n%s"%(string,fielddisplay(self,"numberofvertices","number of vertices"))
		string="%s\n%s"%(string,fielddisplay(self,"elements","vertex indices of the mesh elements"))
		string="%s\n%s"%(string,fielddisplay(self,"x","vertices x coordinate [m]"))
		string="%s\n%s"%(string,fielddisplay(self,"y","vertices y coordinate [m]"))
		string="%s\n%s"%(string,fielddisplay(self,"edges","edges of the 2d mesh (vertex1 vertex2 element1 element2)"))
		string="%s\n%s"%(string,fielddisplay(self,"numberofedges","number of edges of the 2d mesh"))

		string="%s%s"%(string,"\n\n      Properties:")
		string="%s\n%s"%(string,fielddisplay(self,"vertexonboundary","vertices on the boundary of the domain flag list"))
		string="%s\n%s"%(string,fielddisplay(self,"segments","edges on domain boundary (vertex1 vertex2 element)"))
		string="%s\n%s"%(string,fielddisplay(self,"segmentmarkers","number associated to each segment"))
		string="%s\n%s"%(string,fielddisplay(self,"vertexconnectivity","list of vertices connected to vertex_i"))
		string="%s\n%s"%(string,fielddisplay(self,"elementconnectivity","list of vertices connected to element_i"))
		string="%s\n%s"%(string,fielddisplay(self,"average_vertex_connectivity","average number of vertices connected to one vertex"))

		string="%s%s"%(string,"\n\n      Extracted model:")
		string="%s\n%s"%(string,fielddisplay(self,"extractedvertices","vertices extracted from the model"))
		string="%s\n%s"%(string,fielddisplay(self,"extractedelements","elements extracted from the model"))

		string="%s%s"%(string,"\n\n      Projection:")
		string="%s\n%s"%(string,fielddisplay(self,"lat","vertices latitude [degrees]"))
		string="%s\n%s"%(string,fielddisplay(self,"long","vertices longitude [degrees]"))
		string="%s\n%s"%(string,fielddisplay(self,"hemisphere","Indicate hemisphere 'n' or 's'"))
		return string
		#}}}
	def setdefaultparameters(self): # {{{
		
		#the connectivity is the averaged number of nodes linked to a
		#given node through an edge. This connectivity is used to initially
		#allocate memory to the stiffness matrix. A value of 16 seems to
		#give a good memory/time ration. This value can be checked in
		#trunk/test/Miscellaneous/runme.m
		self.average_vertex_connectivity=25

		return self
	#}}}
	def checkconsistency(self,md,solution,analyses):    # {{{

		md = checkfield(md,'mesh.x','NaN',1,'size',[md.mesh.numberofvertices])
		md = checkfield(md,'mesh.y','NaN',1,'size',[md.mesh.numberofvertices])
		md = checkfield(md,'mesh.z','NaN',1,'size',[md.mesh.numberofvertices])
		md = checkfield(md,'mesh.elements','NaN',1,'>',0,'values',numpy.arange(1,md.mesh.numberofvertices+1))
		md = checkfield(md,'mesh.elements','size',[md.mesh.numberofelements,3])
		if numpy.any(numpy.logical_not(ismember(numpy.arange(1,md.mesh.numberofvertices+1),md.mesh.elements))):
			md.checkmessage("orphan nodes have been found. Check the mesh outline")
		md = checkfield(md,'mesh.numberofelements','>',0)
		md = checkfield(md,'mesh.numberofvertices','>',0)
		md = checkfield(md,'mesh.average_vertex_connectivity','>=',9,'message',"'mesh.average_vertex_connectivity' should be at least 9 in 2d")
		if solution==ThermalSolutionEnum():
			md.checkmessage("thermal not supported for 2d mesh")

		return md
	# }}}
	def marshall(self,md,fid):    # {{{
		WriteData(fid,'object',self,'fieldname','x','format','DoubleMat','mattype',1)
		WriteData(fid,'object',self,'fieldname','y','format','DoubleMat','mattype',1)
		WriteData(fid,'object',self,'fieldname','elements','format','DoubleMat','mattype',2)
		WriteData(fid,'object',self,'fieldname','numberofelements','format','Integer')
		WriteData(fid,'object',self,'fieldname','numberofvertices','format','Integer')
		WriteData(fid,'object',self,'fieldname','average_vertex_connectivity','format','Integer')
	# }}}
