%MESH class definition
%
%   Usage:
%      mesh=mesh();

classdef mesh
	properties (SetAccess=public) 
		x                           = NaN;
		y                           = NaN;
		elements                    = NaN
		numberofelements            = 0;
		numberofvertices            = 0;
		numberofedges               = 0;

		lat                         = NaN
		long                        = NaN
		hemisphere                  = NaN

		vertexonboundary            = NaN

		edges                       = NaN
		segments                    = NaN
		segmentmarkers              = NaN
		vertexconnectivity          = NaN
		elementconnectivity         = NaN
		average_vertex_connectivity = 0;

		extractedvertices           = NaN
		extractedelements           = NaN
	end
	methods
		function obj = mesh(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%the connectivity is the averaged number of nodes linked to a
			%given node through an edge. This connectivity is used to initially
			%allocate memory to the stiffness matrix. A value of 16 seems to
			%give a good memory/time ration. This value can be checked in
			%trunk/test/Miscellaneous/runme.m
			obj.average_vertex_connectivity=25;
		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{

			md = checkfield(md,'mesh.x','NaN',1,'size',[md.mesh.numberofvertices 1]);
			md = checkfield(md,'mesh.y','NaN',1,'size',[md.mesh.numberofvertices 1]);
			md = checkfield(md,'mesh.elements','NaN',1,'>',0,'values',1:md.mesh.numberofvertices);
			md = checkfield(md,'mesh.elements','size',[md.mesh.numberofelements 3]);
			if any(~ismember(1:md.mesh.numberofvertices,sort(unique(md.mesh.elements(:)))));
				md = checkmessage(md,'orphan nodes have been found. Check the mesh outline');
			end
			md = checkfield(md,'mesh.numberofelements','>',0);
			md = checkfield(md,'mesh.numberofvertices','>',0);
			md = checkfield(md,'mesh.vertexonbed','size',[md.mesh.numberofvertices 1],'values',[0 1]);
			md = checkfield(md,'mesh.average_vertex_connectivity','>=',9,'message','''mesh.average_vertex_connectivity'' should be at least 9 in 2d');

			switch(solution),
				case ThermalSolutionEnum(),
					md = checkmessage(md,'thermal not supported for 2d mesh');
			end
		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   2d Mesh:')); 

			disp(sprintf('\n      Elements and vertices:'));
			fielddisplay(obj,'numberofelements','number of elements');
			fielddisplay(obj,'numberofvertices','number of vertices');
			fielddisplay(obj,'elements','vertex indices of the mesh elements');
			fielddisplay(obj,'x','vertices x coordinate [m]');
			fielddisplay(obj,'y','vertices y coordinate [m]');
			fielddisplay(obj,'edges','edges of the 2d mesh (vertex1 vertex2 element1 element2)');
			fielddisplay(obj,'numberofedges','number of edges of the 2d mesh');

			disp(sprintf('\n      Properties:'));
			fielddisplay(obj,'vertexonboundary','vertices on the boundary of the domain flag list');
			fielddisplay(obj,'segments','edges on domain boundary (vertex1 vertex2 element)');
			fielddisplay(obj,'segmentmarkers','number associated to each segment');
			fielddisplay(obj,'vertexconnectivity','list of vertices connected to vertex_i');
			fielddisplay(obj,'elementconnectivity','list of vertices connected to element_i');
			fielddisplay(obj,'average_vertex_connectivity','average number of vertices connected to one vertex');

			disp(sprintf('\n      Extracted model:'));
			fielddisplay(obj,'extractedvertices','vertices extracted from the model');
			fielddisplay(obj,'extractedelements','elements extracted from the model');

			disp(sprintf('\n      Projection:'));
			fielddisplay(obj,'lat','vertices latitude [degrees]');
			fielddisplay(obj,'long','vertices longitude [degrees]');
			fielddisplay(obj,'hemisphere','Indicate hemisphere ''n'' or ''s'' ');
		end % }}}
		function marshall(obj,md,fid) % {{{
			WriteData(fid,'object',obj,'fieldname','x','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'fieldname','y','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'fieldname','elements','format','DoubleMat','mattype',2);
			WriteData(fid,'object',obj,'fieldname','numberofelements','format','Integer');
			WriteData(fid,'object',obj,'fieldname','numberofvertices','format','Integer');
			WriteData(fid,'object',obj,'fieldname','average_vertex_connectivity','format','Integer');
		end % }}}
	end
end
