/*!\file: control_core.cpp
 * \brief: core of the control solution 
 */ 

#undef __FUNCT__ 
#define __FUNCT__ "control_core"

#include "./parallel.h"
#include "../issm.h"

void control_core(DataSet* results,FemModel* fems, ParameterInputs* inputs){

	extern int my_rank;

	/*fem models: */
	FemModel* fem_dh=NULL;
	FemModel* fem_dv=NULL;
	FemModel* fem_dhu=NULL;
	FemModel* fem_ds=NULL;
	FemModel* fem_sl=NULL;

	/*output: */
	Result* result=NULL;

	/*Intermediary: */
	DataSet* diagnostic_results=NULL;
	Vec     u_g=NULL;
	double  search_scalar;
	char*   control_type=NULL;
	double* fit=NULL;
	double* optscal=NULL;
	int     gsize;
	double* maxiter=NULL;
	double  tolx;
	double* param_g=NULL;
	Vec     grad_g=NULL;
	Vec     new_grad_g=NULL;
	Vec     grad_g_old=NULL;
	double* grad_g_double=NULL;
	double  mincontrolconstraint;
	double  maxcontrolconstraint;
	int     nsteps,n,i;
	double* J=NULL;
	OptArgs optargs;
	OptPars optpars;
	Param*  param=NULL;

	/*flags: */
	int   analysis_type;
	int   sub_analysis_type;
	int debug=0;
	int dim=-1;
	int ishutter=0;
	int ismacayealpattyn=0;
	int isstokes=0;
	int numberofdofspernode_sl;
	int numberofdofspernode_dh;
	int numberofdofspernode_ds;
	int numberofnodes;

	/*recover fem models: */
	fem_dh=fems+0;
	fem_dv=fems+1;
	fem_ds=fems+2;
	fem_dhu=fems+3;
	fem_sl=fems+4;

	//first recover parameters common to all solutions
	fem_dh->parameters->FindParam((void*)&debug,"debug");
	fem_dh->parameters->FindParam((void*)&dim,"dim");
	fem_dh->parameters->FindParam((void*)&ishutter,"ishutter");
	fem_dh->parameters->FindParam((void*)&ismacayealpattyn,"ismacayealpattyn");
	fem_dh->parameters->FindParam((void*)&numberofnodes,"numberofnodes");
	fem_dh->parameters->FindParam((void*)&isstokes,"isstokes");

	/*Recover parameters used throughout the solution:*/
	fem_dh->parameters->FindParam((void*)&nsteps,"nsteps");
	fem_dh->parameters->FindParam((void*)&control_type,"control_type");
	fem_dh->parameters->FindParam((void*)&fit,"fit");
	fem_dh->parameters->FindParam((void*)&optscal,"optscal");
	fem_dh->parameters->FindParam((void*)&maxiter,"maxiter");
	fem_dh->parameters->FindParam((void*)&tolx,"tolx");
	fem_dh->parameters->FindParam((void*)&mincontrolconstraint,"mincontrolconstraint");
	fem_dh->parameters->FindParam((void*)&maxcontrolconstraint,"maxcontrolconstraint");
	fem_dh->parameters->FindParam((void*)&param_g,"param_g");
	fem_dh->parameters->FindParam((void*)&analysis_type,"analysis_type");
	fem_dh->parameters->FindParam((void*)&sub_analysis_type,"sub_analysis_type");
	fem_dh->parameters->FindParam((void*)&numberofnodes,"numberofnodes");
	gsize=fem_dh->nodes->NumberOfDofs();

	/*Initialize misfit: */
	J=(double*)xmalloc(nsteps*sizeof(double));

	/*erase useless parameters: */
	param=(Param*)fem_dh->parameters->FindParamObject("param_g");
	fem_dh->parameters->DeleteObject((Object*)param);

	param=(Param*)fem_dh->parameters->FindParamObject("u_g");
	fem_dh->parameters->DeleteObject((Object*)param);

	/*take care of Stokes: compute slope and get spc once for all*/
	if (isstokes) ControlPrepareStokes(fems,inputs);

	/*Start looping: */
	for(n=0;n<nsteps;n++){

		_printf_("\n%s%i%s%i\n","   control method step ",n+1,"/",nsteps);
		inputs->Add(control_type,param_g,1,numberofnodes);
		inputs->Add("fit",fit[n]);

		/*Update parameters: */
		if (isstokes) UpdateFromInputsx(fem_ds->elements,fem_ds->nodes,fem_ds->loads, fem_ds->materials,inputs);
		else UpdateFromInputsx(fem_dh->elements,fem_dh->nodes,fem_dh->loads, fem_dh->materials,inputs);

		_printf_("%s\n","      computing gradJ...");
		if (isstokes) grad_g=GradJCompute(inputs,fem_ds);
		else grad_g=GradJCompute(inputs,fem_dh);
		_printf_("%s\n","      done.");

		_printf_("%s\n","      normalizing directions...");
		Orthx(&new_grad_g,grad_g,grad_g_old);
		VecFree(&grad_g); VecFree(&grad_g_old); 
		grad_g_old=new_grad_g;
		VecToMPISerial(&grad_g_double,new_grad_g);
		_printf_("%s\n","      done.");

		_printf_("%s\n","      optimizing along gradient direction...");
		if (isstokes) optargs.femmodel=fem_ds;
		else optargs.femmodel=fem_dh;
		optargs.param_g=param_g; optargs.grad_g=grad_g_double; optargs.inputs=inputs;optargs.n=n;
		optpars.xmin=0; optpars.xmax=1; optpars.tolerance=tolx; optpars.maxiter=(int)maxiter[n];
		BrentSearch(&search_scalar,J+n,&optpars,&objectivefunctionC,&optargs);
		_printf_("%s\n","      done.");

		_printf_("%s\n","      updating parameter using optimized search scalar...");
		for(i=0;i<numberofnodes;i++)param_g[i]=param_g[i]+search_scalar*optscal[n]*grad_g_double[i];
		_printf_("%s\n","      done.");

		_printf_("%s\n","      constraining the new distribution...");    
		ControlConstrainx(param_g,numberofnodes,mincontrolconstraint,maxcontrolconstraint,control_type);
		_printf_("%s\n","      done.");

		_printf_("%s%i%s%g\n","      value of misfit J after optimization #",n+1,": ",J[n]);

		/*some freeing:*/
		xfree((void**)&grad_g_double);

		//some temporary saving
		/*if (((n+1)%5)==0){
			_printf_("%s\n","      saving temporary results...");
			ControlTemporaryResults(fems,param_g,J,n,inputs);
			_printf_("%s\n","      done.");
		}*/
	}

	/*Write results to disk: */
	_printf_("%s\n","      preparing final velocity solution...");

	/*Launch diagnostic with the last parameter distribution*/
	inputs->Add(control_type,param_g,1,numberofnodes);
	diagnostic_results=new DataSet(ResultsEnum()); 
	diagnostic_core(diagnostic_results,fems, inputs);

	//extract u_g from diagnostic_results, and erase diagnostic_results;
	diagnostic_results->FindResult(&u_g,"u_g");
	delete diagnostic_results;

	/*Plug results into output dataset: */
	result=new Result(results->Size()+1,0,1,"u_g",u_g);
	results->AddObject(result);
	result=new Result(results->Size()+1,0,1,"param_g",param_g,gsize);
	results->AddObject(result);
	result=new Result(results->Size()+1,0,1,"J",J,nsteps);
	results->AddObject(result);
}
