%DAMAGEICE class definition
%
%   Usage:
%      damage=damage();

classdef damage
	properties (SetAccess=public)  
	% {{{
		%damage 
		D                   = NaN;
		law                 = '';
		spcdamage           = NaN; 
		stabilization       = NaN;
		
		%parameters for law 'initial': 
		stress_threshold    = NaN;
		c1                  = NaN;
		c2                  = NaN;
		c3                  = NaN;
		c4                  = NaN;
		healing             = NaN;

	end % }}}
	methods
		function obj = damage(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				case 1
					inputstruct=varargin{1};
					list1 = properties('damage');
					list2 = fieldnames(inputstruct);
					for i=1:length(list1)
						fieldname = list1{i};
						if ismember(fieldname,list2),
							obj.(fieldname) = inputstruct.(fieldname);
						end
					end
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%damage parameters: 
			obj.D=0;
			obj.law='undamaged';
		
			%Type of stabilization used
			obj.stabilization=2;
	
			%damage parameters for 'initial' law of damage propagation
			obj.stress_threshold=0;
			obj.healing=0;
			obj.c1=0;
			obj.c2=0;
			obj.c3=0;
			obj.c4=0;

		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{
			
			md = checkfield(md,'damage.D','>=0',0,'size',[md.mesh.numberofvertices 1]);
			md = checkfield(md,'damage.law','values',{'undamaged','pralong'});
			md = checkfield(md,'damage.spcdamage','forcing',1);
			md = checkfield(md,'damage.stabilization','numel',[1],'values',[0 1 2]);
			if strcmpi(obj.law,'pralong'),
				md = checkfield(md,'damage.healing','>=',0);
				md = checkfield(md,'damage.c1','>=',0);
				md = checkfield(md,'damage.c2','>=',0);
				md = checkfield(md,'damage.c3','>=',0);
				md = checkfield(md,'damage.c4','>=',0);
				md = checkfield(md,'damage.stress_threshold','>=',0);
			end

		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   Damage:\n'));

			fielddisplay(obj,'D','damage tensor (scalar)');
			fielddisplay(obj,'law','damage law (string) from {''undamaged'',''pralong''}');
			fielddisplay(obj,'spcdamage','damage constraints (NaN means no constraint) [m]');
			fielddisplay(obj,'stabilization','0: no, 1: artificial_diffusivity, 2: SUPG');
			if strcmpi(obj.law,'pralong'),
				fielddisplay(obj,'c1','damage parameter 1');
				fielddisplay(obj,'c2','damage parameter 2');
				fielddisplay(obj,'c3','damage parameter 3');
				fielddisplay(obj,'c4','damage parameter 4');
				fielddisplay(obj,'healing','damage healing parameter 1');
				fielddisplay(obj,'stress_threshold','damage stress threshold [Pa]');
			end

		end % }}}
		function marshall(obj,md,fid) % {{{
		
			WriteData(fid,'object',obj,'fieldname','D','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'fieldname','law','format','String');
			WriteData(fid,'object',obj,'fieldname','stabilization','format','Integer');
			WriteData(fid,'object',obj,'fieldname','spcdamage','format','DoubleMat','mattype',1,'forcinglength',md.mesh.numberofvertices+1);
			if strcmpi(obj.law,'pralong'),
				WriteData(fid,'object',obj,'fieldname','c1','format','Double');
				WriteData(fid,'object',obj,'fieldname','c2','format','Double');
				WriteData(fid,'object',obj,'fieldname','c3','format','Double');
				WriteData(fid,'object',obj,'fieldname','c4','format','Double');
				WriteData(fid,'object',obj,'fieldname','stress_threshold','format','Double');
				WriteData(fid,'object',obj,'fieldname','healing','format','Double');
			end

		end % }}}
	end
end
