%DAMAGEICE class definition
%
%   Usage:
%      damage=damage();

classdef damage
	properties (SetAccess=public)  
	% {{{
		%damage 
		isdamage                   = NaN;
		D                   = NaN;
		law                 = '';
		
		%parameters for law 'initial': 
		stress_threshold    = NaN;
		c1                  = NaN;
		c2                  = NaN;
		c3                  = NaN;
		c4                  = NaN;

	end % }}}
	methods
		function obj = damage(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				case 1
					inputstruct=varargin{1};
					list1 = properties('damage');
					list2 = fieldnames(inputstruct);
					for i=1:length(list1)
						fieldname = list1{i};
						if ismember(fieldname,list2),
							obj.(fieldname) = inputstruct.(fieldname);
						end
					end
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%damage parameters: 
			obj.isdamage=0;
			obj.D=0;
			obj.law='undamaged';
			
			%damage parameters for 'initial' law of damage propagation
			obj.stress_threshold=0;
			obj.c1=0;
			obj.c2=0;
			obj.c3=0;
			obj.c4=0;

		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{
			
			md = checkfield(md,'materials.D','>=0',0,'size',[md.mesh.numberofvertices 1]);
			md = checkfield(md,'materials.law','values',{'undamaged','pralong'});
			if strcmpi(obj.law,'pralong'),
				md = checkfield(md,'materials.c1','>=',0);
				md = checkfield(md,'materials.c2','>=',0);
				md = checkfield(md,'materials.c3','>=',0);
				md = checkfield(md,'materials.c4','>=',0);
				md = checkfield(md,'materials.stress_threshold','>=',0);
			end

		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   Damage:\n'));

			fielddisplay(obj,'D','damage tensor (scalar)');
			fielddisplay(obj,'law','damage law (string) from {''undamaged'',''pralong''}');
			if strcmpi(obj.law,'pralong'),
				fielddisplay(obj,'c1','damage parameter 1');
				fielddisplay(obj,'c2','damage parameter 2');
				fielddisplay(obj,'c3','damage parameter 3');
				fielddisplay(obj,'c4','damage parameter 4');
				fielddisplay(obj,'stress_threshold','damage stress threshold [Pa]');
			end

		end % }}}
		function marshall(obj,md,fid) % {{{
			
			WriteData(fid,'object',obj,'class','materials','fieldname','D','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'class','materials','fieldname','law','format','String');
			if strcmpi(obj.law,'pralong'),
				WriteData(fid,'object',obj,'class','materials','fieldname','c1','format','Double');
				WriteData(fid,'object',obj,'class','materials','fieldname','c2','format','Double');
				WriteData(fid,'object',obj,'class','materials','fieldname','c3','format','Double');
				WriteData(fid,'object',obj,'class','materials','fieldname','c4','format','Double');
				WriteData(fid,'object',obj,'class','materials','fieldname','stress_threshold','format','Double');
			end

		end % }}}
	end
end
