from fielddisplay import fielddisplay
from EnumDefinitions import *
from StringToEnum import StringToEnum
from checkfield import *
from WriteData import *

class matdamageice(object):
	"""
	MATDAMAGEICE class definition

	   Usage:
	      matdamageice=matdamageice();
	"""

	def __init__(self,*args):    # {{{
		self.rho_ice                    = 0.
		self.rho_water                  = 0.
		self.rho_freshwater             = 0.
		self.mu_water                   = 0.
		self.heatcapacity               = 0.
		self.latentheat                 = 0.
		self.thermalconductivity        = 0.
		self.meltingpoint               = 0.
		self.beta                       = 0.
		self.mixed_layer_capacity       = 0.
		self.thermal_exchange_velocity  = 0.
		self.rheology_B   = float('NaN')
		self.rheology_n   = float('NaN')
		self.rheology_law = ''
		
		#damage: 
		self.damage_D   = float('NaN')
		self.damage_law   = ''
		
		#parameters for law 'initial': 
		self.damage_c1                  = float('NaN')
		self.damage_c2                  = float('NaN')
		self.damage_c3                  = float('NaN')
		self.damage_c4                  = float('NaN')
		self.damage_stress_threshold    = float('NaN')

		#gia: 
		self.lithosphere_shear_modulus  = 0.
		self.lithosphere_density        = 0.
		self.mantle_shear_modulus       = 0.
		self.mantle_density             = 0.

		if not len(args):
			self.setdefaultparameters()
		else:
			raise RuntimeError("constructor not supported")

	# }}}
	def __repr__(self):    # {{{
		s ='   Materials:\n'

		s+="%s\n" % fielddisplay(self,"rho_ice","ice density [kg/m^3]")
		s+="%s\n" % fielddisplay(self,"rho_water","ocean water density [kg/m^3]")
		s+="%s\n" % fielddisplay(self,"rho_freshwater","fresh water density [kg/m^3]")
		s+="%s\n" % fielddisplay(self,"mu_water","water viscosity [N s/m^2]")
		s+="%s\n" % fielddisplay(self,"heatcapacity","heat capacity [J/kg/K]")
		s+="%s\n" % fielddisplay(self,"thermalconductivity","ice thermal conductivity [W/m/K]")
		s+="%s\n" % fielddisplay(self,"meltingpoint","melting point of ice at 1atm in K")
		s+="%s\n" % fielddisplay(self,"latentheat","latent heat of fusion [J/m^3]")
		s+="%s\n" % fielddisplay(self,"beta","rate of change of melting point with pressure [K/Pa]")
		s+="%s\n" % fielddisplay(self,"mixed_layer_capacity","mixed layer capacity [W/kg/K]")
		s+="%s\n" % fielddisplay(self,"thermal_exchange_velocity","thermal exchange velocity [m/s]")
		s+="%s\n" % fielddisplay(self,"rheology_B","flow law parameter [Pa/s^(1/n)]")
		s+="%s\n" % fielddisplay(self,"rheology_n","Glen's flow law exponent")
		s+="%s\n" % fielddisplay(self,"rheology_law","law for the temperature dependance of the rheology: 'None', 'Paterson', 'Arrhenius' or 'LliboutryDuval'")
		s+="%s\n" % fielddisplay(self,"lithosphere_shear_modulus","Lithosphere shear modulus [Pa]")
		s+="%s\n" % fielddisplay(self,"lithosphere_density","Lithosphere density [g/cm^-3]")
		s+="%s\n" % fielddisplay(self,"mantle_shear_modulus","Mantle shear modulus [Pa]")
		s+="%s\n" % fielddisplay(self,"mantle_density","Mantle density [g/cm^-3]")
		s+="%s\n" % fielddisplay(self,"damage_D","damage tensor (scalar for now)")

		if (obj.damage_law=='initial'):
			s+="%s\n" % fielddisplay(self,"damage_c1","damage parameter 1 ")
			s+="%s\n" % fielddisplay(self,"damage_c2","damage parameter 2 ")
			s+="%s\n" % fielddisplay(self,"damage_c3","damage parameter 3 ")
			s+="%s\n" % fielddisplay(self,"damage_c4","damage parameter 4 ")
			s+="%s\n" % fielddisplay(self,"damage_stress_threshold","damage stress threshold [Pa]")

		return s
	# }}}
	def setdefaultparameters(self):    # {{{

		#ice density (kg/m^3)
		self.rho_ice=917.

		#ocean water density (kg/m^3)
		self.rho_water=1023.

		#fresh water density (kg/m^3)
		self.rho_freshwater=1000.

		#water viscosity (N.s/m^2)
		self.mu_water=0.001787  

		#ice heat capacity cp (J/kg/K)
		self.heatcapacity=2093.

		#ice latent heat of fusion L (J/kg)
		self.latentheat=3.34*10**5

		#ice thermal conductivity (W/m/K)
		self.thermalconductivity=2.4

		#the melting point of ice at 1 atmosphere of pressure in K
		self.meltingpoint=273.15

		#rate of change of melting point with pressure (K/Pa)
		self.beta=9.8*10**-8

		#mixed layer (ice-water interface) heat capacity (J/kg/K)
		self.mixed_layer_capacity=3974.

		#thermal exchange velocity (ice-water interface) (m/s)
		self.thermal_exchange_velocity=1.00*10**-4

		#Rheology law: what is the temperature dependence of B with T
		#available: none, paterson and arrhenius
		self.rheology_law='Paterson'
		
		#damage parameters: 
		self.damage_law='initial'
		self.damage_stress_threshold=0
		self.damage_c1=0
		self.damage_c2=0
		self.damage_c3=0
		self.damage_c4=0

		# GIA:
		self.lithosphere_shear_modulus  = 6.7*10**10;  # (Pa)
		self.lithosphere_density        = 3.32;       # (g/cm^-3)
		self.mantle_shear_modulus       = 1.45*10**11; # (Pa)
		self.mantle_density             = 3.34;       # (g/cm^-3)
	# }}}
	def checkconsistency(self,md,solution,analyses):    # {{{
		md = checkfield(md,'materials.rho_ice','>',0)
		md = checkfield(md,'materials.rho_water','>',0)
		md = checkfield(md,'materials.rho_freshwater','>',0)
		md = checkfield(md,'materials.mu_water','>',0)
		md = checkfield(md,'materials.rheology_B','>',0,'size',[md.mesh.numberofvertices])
		md = checkfield(md,'materials.rheology_n','>',0,'size',[md.mesh.numberofelements])
		md = checkfield(md,'materials.rheology_law','values',['None','Paterson','Arrhenius','LliboutryDuval'])
		md = checkfield(md,'materials.lithosphere_shear_modulus','>',0,'numel',[1]);
		md = checkfield(md,'materials.lithosphere_density','>',0,'numel',[1]);
		md = checkfield(md,'materials.mantle_shear_modulus','>',0,'numel',[1]);
		md = checkfield(md,'materials.mantle_density','>',0,'numel',[1]);
		
		md = checkfield(md,'materials.damage_D','>',0,'size',[md.mesh.numberofvertices])
		md = checkfield(md,'materials.damage_law','values',['initial'])
		if self.damage_law == 'initial':
			md = checkfield(md,'materials.damage_c1','>=',0)
			md = checkfield(md,'materials.damage_c2','>=',0)
			md = checkfield(md,'materials.damage_c3','>=',0)
			md = checkfield(md,'materials.damage_c4','>=',0)
			md = checkfield(md,'materials.damage_stress_threshold','>=',0)

		return md
	# }}}
	def marshall(self,md,fid):    # {{{
		WriteData(fid,'enum',MaterialsEnum(),'data',MatdamageiceEnum(),'format','Integer')
		WriteData(fid,'object',self,'class','materials','fieldname','rho_ice','format','Double')
		WriteData(fid,'object',self,'class','materials','fieldname','rho_water','format','Double')
		WriteData(fid,'object',self,'class','materials','fieldname','rho_freshwater','format','Double')
		WriteData(fid,'object',self,'class','materials','fieldname','mu_water','format','Double')
		WriteData(fid,'object',self,'class','materials','fieldname','heatcapacity','format','Double')
		WriteData(fid,'object',self,'class','materials','fieldname','latentheat','format','Double')
		WriteData(fid,'object',self,'class','materials','fieldname','thermalconductivity','format','Double')
		WriteData(fid,'object',self,'class','materials','fieldname','meltingpoint','format','Double')
		WriteData(fid,'object',self,'class','materials','fieldname','beta','format','Double')
		WriteData(fid,'object',self,'class','materials','fieldname','mixed_layer_capacity','format','Double')
		WriteData(fid,'object',self,'class','materials','fieldname','thermal_exchange_velocity','format','Double')
		WriteData(fid,'object',self,'class','materials','fieldname','rheology_B','format','DoubleMat','mattype',1)
		WriteData(fid,'object',self,'class','materials','fieldname','rheology_n','format','DoubleMat','mattype',2)
		WriteData(fid,'data',StringToEnum(self.rheology_law)[0],'enum',MaterialsRheologyLawEnum(),'format','Integer')

		WriteData(fid,'object',self,'class','materials','fieldname','lithosphere_shear_modulus','format','Double');
		WriteData(fid,'object',self,'class','materials','fieldname','lithosphere_density','format','Double','scale',10.**3.);
		WriteData(fid,'object',self,'class','materials','fieldname','mantle_shear_modulus','format','Double');
		WriteData(fid,'object',self,'class','materials','fieldname','mantle_density','format','Double','scale',10.**3.);
		
		WriteData(fid,'object',self,'class','materials','fieldname','damage_D','format','DoubleMat','mattype',1)
		WriteData(fid,'object',self,'class','materials','fieldname','damage_law','format','String')
		if self.damage_law=='initial':
			WriteData(fid,'object',self,'class','materials','fieldname','damage_c1','format','Double')
			WriteData(fid,'object',self,'class','materials','fieldname','damage_c2','format','Double')
			WriteData(fid,'object',self,'class','materials','fieldname','damage_c3','format','Double')
			WriteData(fid,'object',self,'class','materials','fieldname','damage_c4','format','Double')
			WriteData(fid,'object',self,'class','materials','fieldname','damage_stress_threshold','format','Double')
	# }}}
