function manualcb(zmin,zmax,cmap,varargin)
%MANUALCB - custom colorbar
%
%   Usage:
%      manualcb(min,max,colormap,options)
%
%   Available options:
%      - 'fontsize'    : default is 12
%      - 'smallbars'   : bars next to each tick (default is false)
%      - 'position'    : colorbar position in normalized units
%      - 'orientation' : 'vertical' (default) or 'horizontal'
%      - 'title'       : colorbar title
%      - 'tick'        : specified values of tick labels
%      - 'ticksep'     : spacing between ticks
%      - 'inverttickposition' : put ticks on the left hand side for vertical cb

%check inputs
if nargin<3,
	help manualcb
	error('bad usage');
end
if zmin>zmax,
	error('zmin should be smaller than zmax');
end

%Get plot axes
mainaxes = gca;

%process options
options = pairoptions(varargin{:});
if exist(options,'tick') & exist(options,'ticksep'),
	error('only one of tick or ticksep can be specified');
end
fontsize  = getfieldvalue(options,'fontsize',12);
smallbars = getfieldvalue(options,'smallbars',false);

%Colorbar position
if ~exist(options,'position'),
	position = plotboxpos;
	xstart   = position(1)+position(3)+0.01;
	ystart   = position(2);
	width    = .02;
	height   = position(4);
else
	position = getfieldvalue(options,'position');
	xstart = position(1);
	ystart = position(2);
	width  = position(3);
	height = position(4);
end
axes('Units','normalized','Position',[xstart ystart width height],'XTickLabel','','YTickLabel','','Visible','on');
xlim([0 1]);
ylim([0 1]);

%Prepare ticks
if ~exist(options,'log'),
	deltaz = getfieldvalue(options,'ticksep',dtick(zmax-zmin));
	ztick  = getfieldvalue(options,'tick',(deltaz*ceil(zmin/deltaz)):deltaz:zmax);
	if (any(ztick>zmax) | any(ztick<zmin)),
		error('one or more specified tick values falls outside of [zmin,zmax]');
	end
	ytick  = (ztick-zmin)/(zmax-zmin);
else
	ztick = getfieldvalue(options,'tick',round(logspace(log(zmin)/log(10),log(zmax)/log(10),8)));
	ytick = linspace(0,1,numel(ztick));
end

%Display colorbar
hold on
if strcmpi(getfieldvalue(options,'orientation','vertical'),'vertical'),
	image_rgb = ind2rgb(repmat(uint16(1:length(cmap))',1,2),cmap);
else
	image_rgb = ind2rgb(repmat(uint16(1:length(cmap))',1,2)',cmap);
end
imagesc([0 1],[0 1],image_rgb);
patch([0,0,1,1],[0,1,1,0],'k','FaceColor','none','Clipping','off')

%Add ticks
if strcmpi(getfieldvalue(options,'orientation','vertical'),'vertical'),
	%Use FOR LOOP otherwise numbers are not correcly centered
	if getfieldvalue(options,'inverttickposition',0)==1,
		for i=1:length(ytick), text(-0.5,ytick(i),num2str(ztick(i)),'HorizontalAlignment','right','VerticalAlignment','middle','FontSize',fontsize); end
	else
		for i=1:length(ytick), text(1.5,ytick(i),num2str(ztick(i)),'HorizontalAlignment','left','VerticalAlignment','middle','FontSize',fontsize); end
	end
	if smallbars,
		for i=1:numel(ztick)
			patch([0.8 1.0],[ytick(i) ytick(i)],'k')
			patch([0.0 0.2],[ytick(i) ytick(i)],'k')
		end
	end
else
	%Use FOR LOOP otherwise numbers are not correcly centered
	for i=1:length(ytick), text(ytick(i),-0.5,num2str(ztick(i)),'HorizontalAlignment','center','VerticalAlignment','top','FontSize',fontsize); end
	if smallbars,
		for i=1:numel(ztick)
			patch([ytick(i) ytick(i)],[0.8 1.0],[ytick(i) ytick(i)],'k')
			patch([ytick(i) ytick(i)],[0.0 0.2],[ytick(i) ytick(i)],'k')
		end
	end
end

if exist(options,'title'),
	title(getfieldvalue(options,'title'),'FontSize',fontsize);
end
if exist(options,'ylabel'),
	if strcmpi(getfieldvalue(options,'orientation','vertical'),'horizontal'),
		th=title(getfieldvalue(options,'title'),'FontSize',fontsize);
		set(th,'Position',[ytick(end)+0.075,-0.3]);
	else
		ylabel(getfieldvalue(options,'ylabel'),'FontSize',fontsize);
	end
end
	
%Back to original axes
if getfieldvalue(options,'showregion',0)==0,
	axes(mainaxes);
end

function delta = dtick(range)
%Tick intervals
m = 10^floor(log10(range));
p = ceil(range/m);
if p <= 1,     delta = .1*m;
elseif p == 2, delta = .2*m;
elseif p <= 5, delta = .5*m;
else           delta = m;
end
