function md=SetMarineIceSheetBC(md,varargin)
%SETICEMARINESHEETBC - Create the boundary conditions for stressbalance and thermal models for a  Marine Ice Sheet with Ice Front
%
%   Neumann BC are used on the ice front (an ARGUS contour around the ice front
%   can be given in input, or it will be deduced as onfloatingice & onboundary)
%   Dirichlet BC are used elsewhere for stressbalance
%
%   Usage:
%      md=SetMarineIceSheetBC(md,icefrontfile)
%      md=SetMarineIceSheetBC(md)
%
%   Example:
%      md=SetMarineIceSheetBC(md,'Front.exp')
%      md=SetMarineIceSheetBC(md)
%
%   See also: SETICESHELFBC, SETMARINEICESHEETBC

%node on Dirichlet (boundary and ~icefront)
if nargin==2,
	%User provided Front.exp, use it
	icefrontfile=varargin{1};
	if ~exist(icefrontfile)
		error(['SetMarineIceSheetBC error message: ice front file ' icefrontfile ' not found']);
	end
	nodeinsideicefront=ContourToMesh(md.mesh.elements,md.mesh.x,md.mesh.y,icefrontfile,'node',2);
	vertexonicefront=double(md.mesh.vertexonboundary & nodeinsideicefront);
else
	%Guess where the ice front is
	vertexonfloatingice=zeros(md.mesh.numberofvertices,1);
	vertexonfloatingice(md.mesh.elements(find(md.mask.elementonfloatingice),:))=1;
	vertexonicefront=double(md.mesh.vertexonboundary & vertexonfloatingice);
end
pos=find(md.mesh.vertexonboundary & ~vertexonicefront);
if isempty(pos),
	warning('SetMarineIceSheetBC warning: ice front all around the glacier, no dirichlet found. Dirichlet must be added manually')
end
md.stressbalance.spcvx=NaN*ones(md.mesh.numberofvertices,1);
md.stressbalance.spcvy=NaN*ones(md.mesh.numberofvertices,1);
md.stressbalance.spcvz=NaN*ones(md.mesh.numberofvertices,1);
md.stressbalance.referential=NaN*ones(md.mesh.numberofvertices,6);
md.stressbalance.loadingforce=0*ones(md.mesh.numberofvertices,3);

%Position of ice front
md.mask.ice_levelset(find(vertexonicefront))=0;

%First find segments that are not completely on the front
if(md.mesh.dimension==2),
	numbernodesfront=2;
else 
	numbernodesfront=4;
end
segmentsfront=md.mask.ice_levelset(md.mesh.segments(:,1:numbernodesfront))==0;
segments=find(sum(segmentsfront,2)~=numbernodesfront);
%Find all nodes for these segments and spc them
pos=md.mesh.segments(segments,1:numbernodesfront);
md.stressbalance.spcvx(pos(:))=0;
md.stressbalance.spcvy(pos(:))=0;
md.stressbalance.spcvz(pos(:))=0;

%Dirichlet Values
if (length(md.inversion.vx_obs)==md.mesh.numberofvertices & length(md.inversion.vy_obs)==md.mesh.numberofvertices)
	disp('      boundary conditions for stressbalance model: spc set as observed velocities');
	md.stressbalance.spcvx(pos)=md.inversion.vx_obs(pos);
	md.stressbalance.spcvy(pos)=md.inversion.vy_obs(pos);
else
	disp('      boundary conditions for stressbalance model: spc set as zero');
end

md.hydrology.spcwatercolumn=zeros(md.mesh.numberofvertices,2);
pos=find(md.mesh.vertexonboundary);
md.hydrology.spcwatercolumn(pos,1)=1;

%Create zeros basalforcings and surfaceforcings
if (isnan(md.surfaceforcings.precipitation) & (md.surfaceforcings.ispdd==1)),
	md.surfaceforcings.precipitation=zeros(md.mesh.numberofvertices,1);
	disp('      no surfaceforcings.precipitation specified: values set as zero');
end
if (isnan(md.surfaceforcings.mass_balance) & (md.surfaceforcings.ispdd==0)),
	md.surfaceforcings.mass_balance=zeros(md.mesh.numberofvertices,1);
	disp('      no surfaceforcings.mass_balance specified: values set as zero');
end
if isnan(md.basalforcings.melting_rate),
	md.basalforcings.melting_rate=zeros(md.mesh.numberofvertices,1);
	disp('      no basalforcings.melting_rate specified: values set as zero');
end
if isnan(md.balancethickness.thickening_rate),
	md.balancethickness.thickening_rate=zeros(md.mesh.numberofvertices,1);
	disp('      no balancethickness.thickening_rate specified: values set as zero');
end

md.masstransport.spcthickness=NaN*ones(md.mesh.numberofvertices,1);
md.balancethickness.spcthickness=NaN*ones(md.mesh.numberofvertices,1);

if (length(md.initialization.temperature)==md.mesh.numberofvertices),
	md.thermal.spctemperature=NaN*ones(md.mesh.numberofvertices,1);
	pos=find(md.mesh.vertexonsurface); md.thermal.spctemperature(pos)=md.initialization.temperature(pos); %impose observed temperature on surface
	if (length(md.basalforcings.geothermalflux)~=md.mesh.numberofvertices),
		md.basalforcings.geothermalflux=zeros(md.mesh.numberofvertices,1);
		md.basalforcings.geothermalflux(find(md.mask.vertexongroundedice))=50.*10.^-3; %50mW/m2
	end
else
	disp('      no thermal boundary conditions created: no observed temperature found');
end
