/* \file issmmpi.cpp
 * \brief: implementation of all the mpi wrappers that ISSM requires. The goal is to control 
 * which MPI layer we are using at compile time: the standard mpi, the autodiff mpi or no mpi at all.
 */

#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <cassert>
#include <cstring> // for memcpy

#include "issmmpi.h"

// for now - needs to be replaced by issm's type header file >>>>---- from here
#ifdef _HAVE_ADOLC_
# include "adolc/adolc.h"
typedef adouble IssmDouble;
#else 
typedef double IssmDouble;
#endif
// ------<<<< until here

#ifndef _HAVE_MPI_
ISSMMPI_Status ourIssmMPIStatusIgnore=0;
size_t sizeHelper(ISSMMPI_Datatype type) { /*{{{*/

  switch(type) { 
  case ISSMMPI_CHAR: 
    return sizeof(char);
    break;
  case ISSMMPI_DOUBLE: 
    return sizeof(double);
    break;
  case ISSMMPI_INT: 
    return sizeof(int);
    break;
  default: 
    assert(0);
    break;
  }
  return 0;
}/*}}}*/
#endif

int ISSMMPI_Allgather(void *sendbuf, int sendcount, ISSMMPI_Datatype sendtype, void *recvbuf, int recvcount, ISSMMPI_Datatype recvtype, ISSMMPI_Comm comm) {  /*{{{*/
  int rc=0;
  assert(sendcount==recvcount || sendtype==recvtype); // we handle only identical representations
#ifdef _HAVE_MPI_
# ifdef _HAVE_AMPI_
  rc=AMPI_Allgather(sendbuf,
		    sendcount,
		    sendtype,
		    recvbuf,
		    recvcount,
		    recvtype,
		    comm);
# else
  rc=MPI_Allgather(sendbuf,
		   sendcount,
		   sendtype,
		   recvbuf,
		   recvcount,
		   recvtype,
		   comm);
# endif
#else
# ifdef _HAVE_ADOLC_
  if (sendtype==ISSMMPI_DOUBLE) { 
    IssmDouble* activeSendBuf=(IssmDouble*)sendbuf;
    IssmDouble* activeRecvBuf=(IssmDouble*)recvbuf;
    for(int i=0;i<sendcount;++i) activeRecvBuf[i]=activeSendBuf[i];
  }
  else 
# endif
    memcpy(recvbuf,sendbuf,sizeHelper(sendtype)*sendcount);
#endif
  return rc;
} /*}}}*/
int ISSMMPI_Allgatherv(void *sendbuf, int sendcount, ISSMMPI_Datatype sendtype, void *recvbuf, int *recvcounts, int *displs, ISSMMPI_Datatype recvtype, ISSMMPI_Comm comm) {  /*{{{*/
  int rc=0;
  assert(sendtype==recvtype); // we handle only identical representations
#ifdef _HAVE_MPI_
# ifdef _HAVE_AMPI_
  rc=AMPI_Allgatherv(sendbuf,
		     sendcount,
		     sendtype,
		     recvbuf,
		     recvcounts,
		     displs,
		     recvtype,
		     comm);
# else
  rc=MPI_Allgatherv(sendbuf,
		    sendcount,
		    sendtype,
		    recvbuf,
		    recvcounts,
		    displs,
		    recvtype,
		    comm);
# endif
#else
  assert(sendcount==recvcounts[0]); // we handle only identical representations 
# ifdef _HAVE_ADOLC_
  if (sendtype==ISSMMPI_DOUBLE) { 
    IssmDouble* activeSendBuf=(IssmDouble*)sendbuf;
    IssmDouble* activeRecvBuf=(IssmDouble*)(recvbuf)+displs[0];
    for(int i=0;i<sendcount;++i) activeRecvBuf[i]=activeSendBuf[i];
  }
  else 
# endif
    memcpy((char*)recvbuf+(sizeHelper(recvtype)*displs[0]),sendbuf,sizeHelper(sendtype)*sendcount);
#endif
  return rc;
}/*}}}*/
int ISSMMPI_Allreduce(void *sendbuf, void *recvbuf, int count, ISSMMPI_Datatype datatype, ISSMMPI_Op op, ISSMMPI_Comm comm){/*{{{*/
 
  int rc=0;
#ifdef _HAVE_MPI_
# ifdef _HAVE_AMPI_
  rc=AMPI_Allreduce(sendbuf,
		    recvbuf,
		    count,
		    datatype,
		    op,
		    comm);
# else
  rc=MPI_Allreduce(sendbuf,
		   recvbuf,
		   count,
		   datatype,
		   op,
		   comm);
# endif
#else
# ifdef _HAVE_ADOLC_
  if (datatype==ISSMMPI_DOUBLE) { 
    IssmDouble* activeSendBuf=(IssmDouble*)sendbuf;
    IssmDouble* activeRecvBuf=(IssmDouble*)recvbuf;
    for(int i=0;i<count;++i) activeRecvBuf[i]=activeSendBuf[i];
  }
  else 
# endif
    memcpy(recvbuf,sendbuf,sizeHelper(datatype)*count);
#endif
  return rc;
}/*}}}*/
int ISSMMPI_Barrier(ISSMMPI_Comm comm){  /*{{{*/

  int rc=0;
#ifdef _HAVE_MPI_
# ifdef _HAVE_AMPI_
  rc=AMPI_Barrier(comm);
# else 
  rc=MPI_Barrier(comm);
# endif
#else
// do nothing
#endif
  return rc;
}/*}}}*/
int ISSMMPI_Bcast(void *buffer, int count, ISSMMPI_Datatype datatype, int root, ISSMMPI_Comm comm){  /*{{{*/

  int rc=0;
#ifdef _HAVE_MPI_
# ifdef _HAVE_AMPI_
  rc=AMPI_Bcast(buffer, 
		count,
		datatype,
		root,
		comm);
# else
  rc=MPI_Bcast(buffer, 
	       count,
	       datatype,
	       root,
	       comm);
# endif
#else 
// nothing to be done here 
#endif
  return rc;
}/*}}}*/
int ISSMMPI_Comm_free(ISSMMPI_Comm *comm){ /*{{{*/
 
  int rc=0;
#ifdef _HAVE_MPI_
# ifdef _HAVE_AMPI_
  assert(0); // to be implemented
# else 
  rc=MPI_Comm_free(comm);
# endif
#else
// do nothing
#endif
  return rc;
}/*}}}*/
int ISSMMPI_Comm_rank(ISSMMPI_Comm comm, int *rank){  /*{{{*/

  int rc=0;
#ifdef _HAVE_MPI_
  rc=MPI_Comm_rank(comm,
		   rank);
#else
  *rank=0;
#endif
  return rc;
}/*}}}*/
int ISSMMPI_Comm_size( ISSMMPI_Comm comm, int *size){ /*{{{*/
 
  int rc=0;
#ifdef _HAVE_MPI_
  rc=MPI_Comm_size(comm,
		   size);
#else
  *size=1;
#endif
  return rc;
}/*}}}*/
int ISSMMPI_Finalize(void){  /*{{{*/

  int rc=0;
#ifdef _HAVE_MPI_
# ifdef _HAVE_AMPI_
  rc=AMPI_Finalize_NT();
# else
  rc=MPI_Finalize();
# endif
#endif
  return rc;
}/*}}}*/
int ISSMMPI_Gather(void *sendbuf, int sendcnt, ISSMMPI_Datatype sendtype, void *recvbuf, int recvcnt, ISSMMPI_Datatype recvtype, int root, ISSMMPI_Comm comm){  /*{{{*/

  int rc=0;
  assert(sendtype==recvtype && sendcnt==recvcnt); // we handle only identical representations
#ifdef _HAVE_MPI_
# ifdef _HAVE_AMPI_
  rc=AMPI_Gather(sendbuf,
		 sendcnt,
		 sendtype,
		 recvbuf,
		 recvcnt,
		 recvtype,
		 root,
		 comm);
# else
  rc=MPI_Gather(sendbuf,
		sendcnt,
		sendtype,
		recvbuf,
		recvcnt,
		recvtype,
		root,
		comm);
# endif
#else
# ifdef _HAVE_ADOLC_
  if (sendtype==ISSMMPI_DOUBLE) { 
    IssmDouble* activeSendBuf=(IssmDouble*)sendbuf;
    IssmDouble* activeRecvBuf=(IssmDouble*)recvbuf;
    for(int i=0;i<sendcnt;++i) activeRecvBuf[i]=activeSendBuf[i];
  }
  else 
# endif
    memcpy(recvbuf,sendbuf,sizeHelper(sendtype)*sendcnt);
#endif
  return rc;
}/*}}}*/
int ISSMMPI_Gatherv(void *sendbuf, int sendcnt, ISSMMPI_Datatype sendtype, void *recvbuf, int *recvcnts, int *displs, ISSMMPI_Datatype recvtype, int root, ISSMMPI_Comm comm){/*{{{*/
 
  int rc=0;
  assert(sendtype==recvtype); // we handle only identical representations
#ifdef _HAVE_MPI_
# ifdef _HAVE_AMPI_
  rc=AMPI_Gatherv(sendbuf,
		  sendcnt,
		  sendtype,
		  recvbuf,
		  recvcnts,
		  displs,
		  recvtype,
		  root,
		  comm);
# else
  rc=MPI_Gatherv(sendbuf,
		 sendcnt,
		 sendtype,
		 recvbuf,
		 recvcnts,
		 displs,
		 recvtype,
		 root,
		 comm);
# endif
#else
  assert(sendcnt==recvcnts[0]); // we handle only identical representations 
# ifdef _HAVE_ADOLC_
  if (sendtype==ISSMMPI_DOUBLE) { 
    IssmDouble* activeSendBuf=(IssmDouble*)sendbuf;
    IssmDouble* activeRecvBuf=(IssmDouble*)(recvbuf)+displs[0];
    for(int i=0;i<sendcnt;++i) activeRecvBuf[i]=activeSendBuf[i];
  }
  else 
# endif
    memcpy((char*)recvbuf+(sizeHelper(recvtype)*displs[0]),sendbuf,sizeHelper(sendtype)*sendcnt);
#endif
  return rc;
}/*}}}*/
int ISSMMPI_Init(int *argc, char ***argv){  /*{{{*/

  int rc=0;
#ifdef _HAVE_MPI_
# ifdef _HAVE_AMPI_
  rc=AMPI_Init_NT(argc,argv);
# else
  rc=MPI_Init(argc,argv);
# endif
#endif
  return rc;
}/*}}}*/
int ISSMMPI_Recv(void *buf, int count, ISSMMPI_Datatype datatype, int source, int tag, ISSMMPI_Comm comm, ISSMMPI_Status *status){ /*{{{*/

  int rc=0;
#ifdef _HAVE_MPI_
# ifdef _HAVE_AMPI_
  rc=AMPI_Recv(buf, 
	       count,
	       datatype,
	       source,
	       tag,
	       AMPI_SEND, // as long as there are no other variants
	       comm,
	       status);
# else
  rc=MPI_Recv(buf, 
	      count,
	      datatype,
	      source,
	      tag,
	      comm,
	      status);
# endif
#else 
// nothing to be done here 
// as long as nobody tries to do anything with 'status' 
// we won't do anything to it here either
#endif
  return rc;
}/*}}}*/
int ISSMMPI_Reduce(void *sendbuf, void *recvbuf, int count, ISSMMPI_Datatype datatype, ISSMMPI_Op op, int root, ISSMMPI_Comm comm){ /*{{{*/

  int rc=0;
#ifdef _HAVE_MPI_
# ifdef _HAVE_AMPI_
  rc=AMPI_Reduce(sendbuf,
		 recvbuf,
		 count,
		 datatype,
		 op,
		 root,
		 comm);
# else
  rc=MPI_Reduce(sendbuf,
		recvbuf,
		count,
		datatype,
		op,
		root,
		comm);
# endif
#else
# ifdef _HAVE_ADOLC_
  if (datatype==ISSMMPI_DOUBLE) { 
    IssmDouble* activeSendBuf=(IssmDouble*)sendbuf;
    IssmDouble* activeRecvBuf=(IssmDouble*)recvbuf;
    for(int i=0;i<count;++i) activeRecvBuf[i]=activeSendBuf[i];
  }
  else 
# endif
    memcpy(recvbuf,sendbuf,sizeHelper(datatype)*count);
#endif
  return rc;
}/*}}}*/
int ISSMMPI_Scatter(void *sendbuf, int sendcnt, ISSMMPI_Datatype sendtype, void *recvbuf, int recvcnt, ISSMMPI_Datatype recvtype, int root, ISSMMPI_Comm comm){ /*{{{*/

  int rc=0;
  assert(sendtype==recvtype && sendcnt==recvcnt); // we handle only identical representations
#ifdef _HAVE_MPI_
# ifdef _HAVE_AMPI_
  rc=AMPI_Scatter(sendbuf,
		  sendcnt,
		  sendtype,
		  recvbuf,
		  recvcnt,
		  recvtype,
		  root,
		  comm);
# else
  rc=MPI_Scatter(sendbuf,
		 sendcnt,
		 sendtype,
		 recvbuf,
		 recvcnt,
		 recvtype,
		 root,
		 comm);
# endif
#else
# ifdef _HAVE_ADOLC_
  if (sendtype==ISSMMPI_DOUBLE) { 
    IssmDouble* activeSendBuf=(IssmDouble*)sendbuf;
    IssmDouble* activeRecvBuf=(IssmDouble*)recvbuf;
    for(int i=0;i<recvcnt;++i) activeRecvBuf[i]=activeSendBuf[i];
  }
  else 
# endif
    memcpy(recvbuf,sendbuf,sizeHelper(sendtype)*recvcnt);
#endif
  return rc;
}/*}}}*/
int ISSMMPI_Scatterv(void *sendbuf, int *sendcnts, int *displs, ISSMMPI_Datatype sendtype, void *recvbuf, int recvcnt, ISSMMPI_Datatype recvtype, int root, ISSMMPI_Comm comm){ /*{{{*/

  int rc=0;
  assert(sendtype==recvtype); // we handle only identical representations
#ifdef _HAVE_MPI_
# ifdef _HAVE_AMPI_
  rc=AMPI_Scatterv(sendbuf,
		   sendcnts,
		   displs,
		   sendtype,
		   recvbuf,
		   recvcnt,
		   recvtype,
		   root,
		   comm);
# else
  rc=MPI_Scatterv(sendbuf,
		  sendcnts,
		  displs,
		  sendtype,
		  recvbuf,
		  recvcnt,
		  recvtype,
		  root,
		  comm);
# endif
#else
  assert(sendcnts[0]==recvcnt); // we handle only identical representations 
# ifdef _HAVE_ADOLC_
  if (sendtype==ISSMMPI_DOUBLE) { 
    IssmDouble* activeSendBuf=(IssmDouble*)(sendbuf)+displs[0];
    IssmDouble* activeRecvBuf=(IssmDouble*)recvbuf;
    for(int i=0;i<recvcnt;++i) activeRecvBuf[i]=activeSendBuf[i];
  }
  else 
# endif
    memcpy(recvbuf,(char*)sendbuf+(sizeHelper(sendtype)*displs[0]),sizeHelper(sendtype)*recvcnt);
#endif
  return rc;
}/*}}}*/
int ISSMMPI_Send(void *buf, int count, ISSMMPI_Datatype datatype, int dest, int tag, ISSMMPI_Comm comm){ /*{{{*/

  int rc=0;
#ifdef _HAVE_MPI_
# ifdef _HAVE_AMPI_
  rc=AMPI_Send(buf, 
	       count,
	       datatype,
	       dest,
	       tag,
	       AMPI_RECV, // as long as there are no other variants
	       comm);
# else
  rc=MPI_Send(buf, 
	      count,
	      datatype,
	      dest,
	      tag,
	      comm);
# endif
#else 
// nothing to be done here 
#endif
  return rc;
}/*}}}*/
double ISSMMPI_Wtime(void){/*{{{*/
 
  assert(0); // to be implemented
  return 0.0;
}/*}}}*/
void ISSMMPI_ContiguousInAdolc(size_t aSize) { /*{{{*/

#ifdef _HAVE_ADOLC_
  ensureContiguousLocations(aSize);
#endif
}/*}}}*/

