%MASSTRANSPORT class definition
%
%   Usage:
%      masstransport=masstransport();

classdef masstransport
	properties (SetAccess=public) 
		 spcthickness           = NaN;
		 min_thickness          = 0;
		 hydrostatic_adjustment = 0;
		 stabilization          = 0;
		 vertex_pairing         = NaN;
		 penalty_factor         = 0;
		 requested_outputs      = NaN;
	end
	methods (Static)
		function obj = loadobj(obj) % {{{
			% This function is directly called by matlab when a model object is
			% loaded. If the input is a struct it is an old version of this class and
			% old fields must be recovered (make sure they are in the deprecated
			% model properties)

			if verLessThan('matlab','7.9'),
				disp('Warning: your matlab version is old and there is a risk that load does not work correctly');
				disp('         if the model is not loaded correctly, rename temporarily loadobj so that matlab does not use it');

				% This is a Matlab bug: all the fields of md have their default value
				% Example of error message:
				% Warning: Error loading an object of class 'model':
				% Undefined function or method 'exist' for input arguments of type 'cell'
				%
				% This has been fixed in MATLAB 7.9 (R2009b) and later versions
			end

			if isstruct(obj)
				disp('Recovering masstransport from older version');
				objstruct = obj;
				obj = structtoobj(masstransport(),objstruct);
			end
		end% }}}
	end
	methods
		function obj = masstransport(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				case 1
					inputstruct=varargin{1};
					list1 = properties('masstransport');
					list2 = fieldnames(inputstruct);
					for i=1:length(list1)
						fieldname = list1{i};
						if ismember(fieldname,list2),
							obj.(fieldname) = inputstruct.(fieldname);
						end
					end
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%Type of stabilization to use 0:nothing 1:artificial_diffusivity 3:Discontinuous Galerkin
			obj.stabilization=1;

			%Factor applied to compute the penalties kappa=max(stiffness matrix)*10^penalty_factor
			obj.penalty_factor=3;

			%Minimum ice thickness that can be used
			obj.min_thickness=1;

			%Hydrostatic adjustment
			obj.hydrostatic_adjustment='Absolute';
		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{

			%Early return
			if ~ismember(MasstransportAnalysisEnum(),analyses) |  (solution==TransientSolutionEnum() & md.transient.ismasstransport==0), return; end

			md = checkfield(md,'masstransport.spcthickness','forcing',1);
			md = checkfield(md,'masstransport.hydrostatic_adjustment','values',{'Absolute' 'Incremental'});
			md = checkfield(md,'masstransport.stabilization','values',[0 1 2 3]);
			md = checkfield(md,'masstransport.min_thickness','>',0);
			if ~isempty(md.masstransport.requested_outputs),
				md = checkfield(md,'masstransport.requested_outputs','size',[NaN 1]);
			end

		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   Masstransport solution parameters:'));
			fielddisplay(obj,'spcthickness','thickness constraints (NaN means no constraint) [m]');
			fielddisplay(obj,'min_thickness','minimum ice thickness allowed [m]');
			fielddisplay(obj,'hydrostatic_adjustment','adjustment of ice shelves surface and bed elevations: ''Incremental'' or ''Absolute'' ');
			fielddisplay(obj,'stabilization','0: no, 1: artificial_diffusivity, 2: streamline upwinding, 3: discontinuous Galerkin');

			disp(sprintf('\n      %s','Penalty options:'));
			fielddisplay(obj,'penalty_factor','offset used by penalties: penalty = Kmax*10^offset');
			fielddisplay(obj,'vertex_pairing','pairs of vertices that are penalized');
			fielddisplay(obj,'requested_outputs','additional outputs requested');

		end % }}}
		function marshall(obj,md,fid) % {{{
			WriteData(fid,'object',obj,'fieldname','spcthickness','format','DoubleMat','mattype',1,'forcinglength',md.mesh.numberofvertices+1);
			WriteData(fid,'object',obj,'fieldname','min_thickness','format','Double');
			WriteData(fid,'data',StringToEnum(obj.hydrostatic_adjustment),'format','Integer','enum',MasstransportHydrostaticAdjustmentEnum());
			WriteData(fid,'object',obj,'fieldname','stabilization','format','Integer');
			WriteData(fid,'object',obj,'fieldname','vertex_pairing','format','DoubleMat','mattype',3);
			WriteData(fid,'object',obj,'fieldname','penalty_factor','format','Double');
			WriteData(fid,'object',obj,'fieldname','requested_outputs','format','DoubleMat','mattype',3);
		end % }}}
	end
end
