/*!\file IoModelToConstraintsx
 * \brief: used in ModelProcessor, to retrieve a nodal vector  and to create corresponding constraints using Constraints objects
 */

#include "./IoModelToConstraintsx.h"
#include "../../shared/shared.h"
#include "../../toolkits/toolkits.h"
#include "../ModelProcessorx/ModelProcessorx.h"

void IoModelToConstraintsx(Constraints* constraints,IoModel* iomodel,int vector_enum,int analysis_type,int finite_element,int dof){

	/*intermediary: */
	int         i,j,count;
	IssmDouble  yts;
	FILE       *fid              = NULL;
	int         code,vector_layout;
	IssmDouble *times            = NULL;
	IssmDouble *values           = NULL;
	bool        spcpresent       = false;

	/*P2 finite elements*/
	int   v1,v2;
	bool *my_edges = NULL;

	/*variables being fetched: */
	IssmDouble *spcdata = NULL;
	int         M,N;

	/*Fetch parameters: */
	iomodel->Constant(&yts,ConstantsYtsEnum);

	/*First of, find the record for the enum, and get code  of data type: */
	fid=iomodel->SetFilePointerToData(&code, &vector_layout,vector_enum);
	if(code!=7)_error_("expecting a IssmDouble vector for constraints with enum " << EnumToStringx(vector_enum));
	if(vector_layout!=1)_error_("expecting a nodal vector for constraints with enum " << EnumToStringx(vector_enum));

	/*Fetch vector:*/
	iomodel->FetchData(&spcdata,&M,&N,vector_enum);

	/*Partition edges if we are using P2 finite elements*/
	if(finite_element==P2Enum){
		EdgesPartitioning(&my_edges,iomodel);
	}

	if(M==iomodel->numberofvertices){
		/*Static constraint*/
		count=0;
		for (i=0;i<iomodel->numberofvertices;i++){
			if((iomodel->my_vertices[i])){
				if (!xIsNan<IssmDouble>(spcdata[i])){
					constraints->AddObject(new SpcStatic(iomodel->constraintcounter+count+1,iomodel->nodecounter+i+1,dof,spcdata[i],analysis_type));
					count++;
				}
			}
		}
		if(finite_element==P2Enum){
			for(i=0;i<iomodel->numberofedges;i++){
				if(my_edges[i]){
					v1 = iomodel->edges[3*i+0]-1;
					v2 = iomodel->edges[3*i+1]-1;
					if(!xIsNan<IssmDouble>(spcdata[v1]) && !xIsNan<IssmDouble>(spcdata[v2])){
						constraints->AddObject(new SpcStatic(iomodel->constraintcounter+count+1,iomodel->nodecounter+iomodel->numberofvertices+i+1,
										dof,(spcdata[v1]+spcdata[v2])/2.,analysis_type));
						count++;
					}
				}
			}
		}
	}
	else if (M==(iomodel->numberofvertices+1)){
		/*transient: create transient SpcTransient objects. Same logic, except we need to retrieve 
		 * various times and values to initialize an SpcTransient object: */
		count=0;

		/*figure out times: */
		times=xNew<IssmDouble>(N);
		for(j=0;j<N;j++) times[j]=spcdata[(M-1)*N+j]*yts;

		/*Create constraints: */
		for (i=0;i<iomodel->numberofvertices;i++){
			if((iomodel->my_vertices[i])){

				/*figure out times and values: */
				values=xNew<IssmDouble>(N);
				spcpresent=false;
				for(j=0;j<N;j++){
					values[j]=spcdata[i*N+j];
					if(!xIsNan<IssmDouble>(values[j]))spcpresent=true; //NaN means no spc by default
				}

				if(spcpresent){
					constraints->AddObject(new SpcTransient(iomodel->constraintcounter+count+1,iomodel->nodecounter+i+1,dof,N,times,values,analysis_type));
					count++;
				}
				xDelete<IssmDouble>(values);
			}
		}
		if(finite_element==P2Enum){
			for(i=0;i<iomodel->numberofedges;i++){
				if(my_edges[i]){
					v1 = iomodel->edges[3*i+0]-1;
					v2 = iomodel->edges[3*i+1]-1;
					values=xNew<IssmDouble>(N);
					spcpresent=false;
					for(j=0;j<N;j++){
						values[j]=(spcdata[v1*N+j]+spcdata[v2*N+j])/2.;
						if(!xIsNan<IssmDouble>(values[j])) spcpresent=true; //NaN means no spc by default
					}
					if(spcpresent){
						constraints->AddObject(new SpcTransient(iomodel->constraintcounter+count+1,iomodel->nodecounter+iomodel->numberofvertices+i+1,dof,
										N,times,values,analysis_type));
						count++;
					}
					xDelete<IssmDouble>(values);
				}
			}
		}
	}
	else{
		_error_("Size of field " << EnumToStringx(vector_enum) << " not supported");
	}

	/*Free ressources:*/
	xDelete<IssmDouble>(spcdata);
	xDelete<IssmDouble>(times);
	xDelete<IssmDouble>(values);
	xDelete<bool>(my_edges);
}
