%STRESSBALANC Eclass definition
%
%   Usage:
%      stressbalance=stressbalance();

classdef stressbalance
	properties (SetAccess=public) 
		spcvx                  = NaN;
		spcvy                  = NaN;
		spcvz                  = NaN;
		restol                 = 0;
		reltol                 = 0;
		abstol                 = 0;
		isnewton               = 0;
		FSreconditioning       = 0;
		viscosity_overshoot    = 0;
		maxiter                = 0;
		shelf_dampening        = 0;
		vertex_pairing         = NaN;
		penalty_factor         = NaN;
		rift_penalty_lock      = NaN;
		rift_penalty_threshold = 0;
		referential            = NaN;
		loadingforce           = NaN;
		requested_outputs      = NaN;
	end
	methods
		function obj = stressbalance(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				case 1
					inputstruct=varargin{1};
					list1 = properties('stressbalance');
					list2 = fieldnames(inputstruct);
					for i=1:length(list1)
						fieldname = list1{i};
						if ismember(fieldname,list2),
							obj.(fieldname) = inputstruct.(fieldname);
						end
					end
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			 %maximum of non-linear iterations.
			 obj.maxiter=100;

			 %Convergence criterion: absolute, relative and residual
			 obj.restol=10^-4; 
			 obj.reltol=0.01;
			 obj.abstol=10;

			 obj.FSreconditioning=10^13;
			 obj.shelf_dampening=0;

			 %Penalty factor applied kappa=max(stiffness matrix)*10^penalty_factor
			 obj.penalty_factor=3;

			 %coefficient to update the viscosity between each iteration of
			 %a stressbalance according to the following formula
			 %viscosity(n)=viscosity(n)+viscosity_overshoot(viscosity(n)-viscosity(n-1))
			 obj.viscosity_overshoot=0;

			 %Stop the iterations of rift if below a threshold
			 obj.rift_penalty_threshold=0;

			 %in some solutions, it might be needed to stop a run when only
			 %a few constraints remain unstable. For thermal computation, this
			 %parameter is often used.
			 obj.rift_penalty_lock=10;

		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{

			%Early return
			if ~ismember(DiagnosticHorizAnalysisEnum(),analyses), return; end

			md = checkfield(md,'diagnostic.spcvx','forcing',1);
			md = checkfield(md,'diagnostic.spcvy','forcing',1);
			if md.mesh.dimension==3, md = checkfield(md,'diagnostic.spcvz','forcing',1); end
			md = checkfield(md,'diagnostic.restol','size',[1 1],'>',0,'NaN',1);
			md = checkfield(md,'diagnostic.reltol','size',[1 1]);
			md = checkfield(md,'diagnostic.abstol','size',[1 1]);
			md = checkfield(md,'diagnostic.isnewton','numel',[1],'values',[0 1 2]);
			md = checkfield(md,'diagnostic.FSreconditioning','size',[1 1],'NaN',1);
			md = checkfield(md,'diagnostic.viscosity_overshoot','size',[1 1],'NaN',1);
			md = checkfield(md,'diagnostic.maxiter','size',[1 1],'>=',1);
			md = checkfield(md,'diagnostic.referential','size',[md.mesh.numberofvertices 6]);
			md = checkfield(md,'diagnostic.loadingforce','size',[md.mesh.numberofvertices 3]);
			if ~isempty(md.diagnostic.requested_outputs),
				md = checkfield(md,'diagnostic.requested_outputs','size',[NaN 1]);
			end

			%singular solution
			if ~(any(~isnan(md.diagnostic.spcvx)) & any(~isnan(md.diagnostic.spcvy))),
				md = checkmessage(md,['model is not well posed (singular). You need at least one node with fixed velocity!']);
			end
			%CHECK THAT EACH LINES CONTAINS ONLY NAN VALUES OR NO NAN VALUES
			if any(sum(isnan(md.diagnostic.referential),2)~=0 & sum(isnan(md.diagnostic.referential),2)~=6),
				md = checkmessage(md,['Each line of diagnostic.referential should contain either only NaN values or no NaN values']);
			end
			%CHECK THAT THE TWO VECTORS PROVIDED ARE ORTHOGONAL
			if any(sum(isnan(md.diagnostic.referential),2)==0),
				pos=find(sum(isnan(md.diagnostic.referential),2)==0);
				if any(abs(dot(md.diagnostic.referential(pos,1:3),md.diagnostic.referential(pos,4:6),2))>eps),
					md = checkmessage(md,['Vectors in diagnostic.referential (columns 1 to 3 and 4 to 6) must be orthogonal']);
				end
			end
			%CHECK THAT NO rotation specified for FS Grounded ice at base
			if md.mesh.dimension==3 & md.flowequation.isFS,
				pos=find(md.mask.vertexongroundedice & md.mesh.vertexonbed);
				if any(~isnan(md.diagnostic.referential(pos,:))),
					md = checkmessage(md,['no referential should be specified for basal vertices of grounded ice']);
				end
				md = checkfield(md,'diagnostic.FSreconditioning','>',0);
			end
		end % }}}
		function disp(obj) % {{{

			disp(sprintf('   StressBalance solution parameters:'));

			disp(sprintf('\n      %s','Convergence criteria:'));
			fielddisplay(obj,'restol','mechanical equilibrium residual convergence criterion');
			fielddisplay(obj,'reltol','velocity relative convergence criterion, NaN: not applied');
			fielddisplay(obj,'abstol','velocity absolute convergence criterion, NaN: not applied');
			fielddisplay(obj,'isnewton','0: Picard''s fixed point, 1: Newton''s method, 2: hybrid');
			fielddisplay(obj,'maxiter','maximum number of nonlinear iterations');
			fielddisplay(obj,'viscosity_overshoot','over-shooting constant new=new+C*(new-old)');

			disp(sprintf('\n      %s','boundary conditions:'));
			fielddisplay(obj,'spcvx','x-axis velocity constraint (NaN means no constraint) [m/yr]');
			fielddisplay(obj,'spcvy','y-axis velocity constraint (NaN means no constraint) [m/yr]');
			fielddisplay(obj,'spcvz','z-axis velocity constraint (NaN means no constraint) [m/yr]');

			disp(sprintf('\n      %s','Rift options:'));
			fielddisplay(obj,'rift_penalty_threshold','threshold for instability of mechanical constraints');
			fielddisplay(obj,'rift_penalty_lock','number of iterations before rift penalties are locked');

			disp(sprintf('\n      %s','Penalty options:'));
			fielddisplay(obj,'penalty_factor','offset used by penalties: penalty = Kmax*10^offset');
			fielddisplay(obj,'vertex_pairing','pairs of vertices that are penalized');

			disp(sprintf('\n      %s','Other:'));
			fielddisplay(obj,'shelf_dampening','use dampening for floating ice ? Only for FS model');
			fielddisplay(obj,'FSreconditioning','multiplier for incompressibility equation. Only for FS model');
			fielddisplay(obj,'referential','local referential');
			fielddisplay(obj,'loadingforce','loading force applied on each point [N/m^3]');
			fielddisplay(obj,'requested_outputs','additional outputs requested');

		end % }}}
		function marshall(obj,md,fid) % {{{

			yts=365.0*24.0*3600.0;

			WriteData(fid,'object',obj,'class','diagnostic','fieldname','spcvx','format','DoubleMat','mattype',1,'scale',1./yts,'forcinglength',md.mesh.numberofvertices+1);
			WriteData(fid,'object',obj,'class','diagnostic','fieldname','spcvy','format','DoubleMat','mattype',1,'scale',1./yts,'forcinglength',md.mesh.numberofvertices+1);
			WriteData(fid,'object',obj,'class','diagnostic','fieldname','spcvz','format','DoubleMat','mattype',1,'scale',1./yts,'forcinglength',md.mesh.numberofvertices+1);
			WriteData(fid,'object',obj,'class','diagnostic','fieldname','restol','format','Double');
			WriteData(fid,'object',obj,'class','diagnostic','fieldname','reltol','format','Double');
			WriteData(fid,'object',obj,'class','diagnostic','fieldname','abstol','format','Double');
			WriteData(fid,'object',obj,'class','diagnostic','fieldname','isnewton','format','Integer');
			WriteData(fid,'object',obj,'class','diagnostic','fieldname','FSreconditioning','format','Double');
			WriteData(fid,'object',obj,'class','diagnostic','fieldname','viscosity_overshoot','format','Double');
			WriteData(fid,'object',obj,'class','diagnostic','fieldname','maxiter','format','Integer');
			WriteData(fid,'object',obj,'class','diagnostic','fieldname','shelf_dampening','format','Integer');
			WriteData(fid,'object',obj,'class','diagnostic','fieldname','vertex_pairing','format','DoubleMat','mattype',3);
			WriteData(fid,'object',obj,'class','diagnostic','fieldname','penalty_factor','format','Double');
			WriteData(fid,'object',obj,'class','diagnostic','fieldname','rift_penalty_lock','format','Integer');
			WriteData(fid,'object',obj,'class','diagnostic','fieldname','rift_penalty_threshold','format','Integer');
			WriteData(fid,'object',obj,'class','diagnostic','fieldname','referential','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'class','diagnostic','fieldname','requested_outputs','format','DoubleMat','mattype',3);
			WriteData(fid,'data',obj.loadingforce(:,1),'format','DoubleMat','mattype',1,'enum',LoadingforceXEnum);
			WriteData(fid,'data',obj.loadingforce(:,2),'format','DoubleMat','mattype',1,'enum',LoadingforceYEnum);
			WriteData(fid,'data',obj.loadingforce(:,3),'format','DoubleMat','mattype',1,'enum',LoadingforceZEnum);
		end % }}}
	end
end
