function ismodelselfconsistent(md),
%ISMODELSELFCONSISTENT - check that model forms a closed form solvable problem.
%
%   Usage:
%      ismodelselfconsistent(md),

%initialize consistency as true
md.private.isconsistent=true;

%Get solution and associated analyses
solution=md.private.solution;
[analyses,numanalyses]=AnalysisConfiguration(solution);

%Go through a model field, check that it is a class, and call checkconsistency
fields=properties('model');
for i=1:length(fields),
	field=fields{i};

	%Some properties do not need to be checked
	if ismember(field,{'results' 'debug' 'radaroverlay'}),
		continue;
	end

	%Check that current field is an object
	if ~isobject(md.(field))
		md=checkmessage(md,['field ''' char(field) ''' is not an object']);
	end

	%Check consistency of the object
	md=checkconsistency(md.(field),md,solution,analyses);
end

%error message if mode is not consistent
if md.private.isconsistent==false,
	error('Model not consistent, see messages above');
end
end

function [analyses,numanalyses]=AnalysisConfiguration(solutiontype), % {{{
%ANALYSISCONFIGURATION - return type of analyses, number of analyses 
%
%   Usage:
%      [analyses, numanalyses]=AnalysisConfiguration(solutiontype);

switch solutiontype,

	case DiagnosticSolutionEnum(),
		numanalyses=5;
		analyses=[DiagnosticHorizAnalysisEnum();DiagnosticVertAnalysisEnum();DiagnosticHutterAnalysisEnum();SurfaceSlopeAnalysisEnum();BedSlopeAnalysisEnum()];

	case SteadystateSolutionEnum(),
		numanalyses=7; 
		analyses=[DiagnosticHorizAnalysisEnum();DiagnosticVertAnalysisEnum();DiagnosticHutterAnalysisEnum();SurfaceSlopeAnalysisEnum();BedSlopeAnalysisEnum();ThermalAnalysisEnum();MeltingAnalysisEnum()];

	case ThermalSolutionEnum(),
		numanalyses=2; 
		analyses=[ThermalAnalysisEnum();MeltingAnalysisEnum()];

	case EnthalpySolutionEnum(),
		numanalyses=1; 
		analyses=[EnthalpyAnalysisEnum()];

	case PrognosticSolutionEnum(),
		numanalyses=1; 
		analyses=[PrognosticAnalysisEnum()];

	case BalancethicknessSolutionEnum(),
		numanalyses=1; 
		analyses=[BalancethicknessAnalysisEnum()];

	case BalancethicknessSoftSolutionEnum(),
		numanalyses=1; 
		analyses=[BalancethicknessAnalysisEnum()];

	case SurfaceSlopeSolutionEnum(),
		numanalyses=1; 
		analyses=[SurfaceSlopeAnalysisEnum()];

	case BedSlopeSolutionEnum(),
		numanalyses=1; 
		analyses=[BedSlopeAnalysisEnum()];

	case GiaSolutionEnum(),
		numanalyses=1; 
		analyses=[GiaAnalysisEnum()];

	case TransientSolutionEnum(),
		numanalyses=9; 
		analyses=[DiagnosticHorizAnalysisEnum();DiagnosticVertAnalysisEnum();DiagnosticHutterAnalysisEnum();SurfaceSlopeAnalysisEnum();BedSlopeAnalysisEnum();ThermalAnalysisEnum();MeltingAnalysisEnum();EnthalpyAnalysisEnum();PrognosticAnalysisEnum()];

	case FlaimSolutionEnum(),
		numanalyses=1; 
		analyses=[FlaimAnalysisEnum()];

	case HydrologySolutionEnum(),
		numanalyses=3; 
		analyses=[BedSlopeAnalysisEnum();SurfaceSlopeAnalysisEnum();HydrologyShreveAnalysisEnum();HydrologyDCInefficientAnalysisEnum();HydrologyDCEfficientAnalysisEnum()];

	otherwise
		error('%s%s%s',' solution type: ',EnumToString(solutiontype),' not supported yet!');

	end
end % }}}
