%MESH class definition
%
%   Usage:
%      mesh=mesh();

classdef mesh
	properties (SetAccess=public) 
		x                           = NaN;
		y                           = NaN;
		z                           = NaN
		elements                    = NaN
		dimension                   = 0;
		numberoflayers              = 0;
		numberofelements            = 0;
		numberofvertices            = 0;
		numberofedges               = 0;

		lat                         = NaN
		long                        = NaN
		hemisphere                  = NaN

		elementonbed                = NaN
		elementonsurface            = NaN
		vertexonbed                 = NaN
		vertexonsurface             = NaN
		lowerelements               = NaN
		lowervertex                 = NaN
		upperelements               = NaN
		uppervertex                 = NaN
		vertexonboundary            = NaN

		edges                       = NaN
		segments                    = NaN
		segmentmarkers              = NaN
		vertexconnectivity          = NaN
		elementconnectivity         = NaN
		average_vertex_connectivity = 0;

		x2d                         = NaN
		y2d                         = NaN
		elements2d                  = NaN
		numberofvertices2d          = 0;
		numberofelements2d          = 0;

		extractedvertices           = NaN
		extractedelements           = NaN
	end
	methods (Static)
		function obj = loadobj(obj) % {{{
			% This function is directly called by matlab when a model object is
			% loaded. Update old properties here

			%2012 June 28th
			if numel(obj.edges)>1 & any(isnan(obj.edges(:)))
				disp('Update model edges from previous version');
				obj.edges(isnan(obj.edges))=-1;
			end

		end% }}}
	end
	methods
		function obj = mesh(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%the connectivity is the averaged number of nodes linked to a
			%given node through an edge. This connectivity is used to initially
			%allocate memory to the stiffness matrix. A value of 16 seems to
			%give a good memory/time ration. This value can be checked in
			%trunk/test/Miscellaneous/runme.m
			obj.average_vertex_connectivity=25;
		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{

			md = checkfield(md,'mesh.x','NaN',1,'size',[md.mesh.numberofvertices 1]);
			md = checkfield(md,'mesh.y','NaN',1,'size',[md.mesh.numberofvertices 1]);
			md = checkfield(md,'mesh.z','NaN',1,'size',[md.mesh.numberofvertices 1]);
			md = checkfield(md,'mesh.elements','NaN',1,'>',0,'values',1:md.mesh.numberofvertices);
			if(md.mesh.dimension==2),
				md = checkfield(md,'mesh.elements','size',[md.mesh.numberofelements 3]);
			else
				md = checkfield(md,'mesh.elements','size',[md.mesh.numberofelements 6]);
			end
			if any(~ismember(1:md.mesh.numberofvertices,sort(unique(md.mesh.elements(:)))));
				md = checkmessage(md,'orphan nodes have been found. Check the mesh outline');
			end
			md = checkfield(md,'mesh.dimension','values',[2 3]);
			md = checkfield(md,'mesh.numberoflayers','>=',0);
			md = checkfield(md,'mesh.numberofelements','>',0);
			md = checkfield(md,'mesh.numberofvertices','>',0);
			%no checks for numberofedges lat long and hemisphere
			md = checkfield(md,'mesh.elementonbed','size',[md.mesh.numberofelements 1],'values',[0 1]);
			md = checkfield(md,'mesh.elementonsurface','size',[md.mesh.numberofelements 1],'values',[0 1]);
			md = checkfield(md,'mesh.vertexonbed','size',[md.mesh.numberofvertices 1],'values',[0 1]);
			md = checkfield(md,'mesh.vertexonsurface','size',[md.mesh.numberofvertices 1],'values',[0 1]);
			if (md.mesh.dimension==3),
				md = checkfield(md,'mesh.z','>=',md.geometry.bed-10^-10,'message','''mesh.z'' lower than bedrock');
				md = checkfield(md,'mesh.z','<=',md.geometry.surface+10^-10,'message','''mesh.z'' higher than surface elevation');
			end
			if (md.mesh.dimension==2),
				md = checkfield(md,'mesh.average_vertex_connectivity','>=',9,'message','''mesh.average_vertex_connectivity'' should be at least 9 in 2d');
			else
				md = checkfield(md,'mesh.average_vertex_connectivity','>=',24,'message','''mesh.average_vertex_connectivity'' should be at least 24 in 3d');
			end
			md = checkfield(md,'mesh.elementconnectivity','size',[md.mesh.numberofelements 3],'NaN',1);

			%Solution specific checks
			switch(solution),
				case PrognosticSolutionEnum(),
					if md.prognostic.stabilization==3,
						md = checkfield(md,'mesh.dimension','values',2,'message','Discontinuous Galerkin only supported for 2d meshes');
						md = checkfield(md,'mesh.edges','size',[NaN 4]);
						md = checkfield(md,'mesh.edges(:,1:3)','>',0);
					end
				case BalancethicknessSolutionEnum(),
					if md.balancethickness.stabilization==3,
						md = checkfield(md,'mesh.dimension','values',2,'message','Discontinuous Galerkin only supported for 2d meshes');
						md = checkfield(md,'mesh.edges','size',[NaN 4]);
						md = checkfield(md,'mesh.edges(:,1:3)','>',0);
					end
				case TransientSolutionEnum(),
					if md.transient.isprognostic & md.prognostic.stabilization==3,
						md = checkfield(md,'mesh.dimension','values',2,'message','Discontinuous Galerkin only supported for 2d meshes');
						md = checkfield(md,'mesh.edges','size',[NaN 4]);
						md = checkfield(md,'mesh.edges(:,1:3)','>',0);
					end
				case ThermalSolutionEnum(),
					md = checkfield(md,'mesh.dimension','values',3,'message','thermal solution only supported on 3d meshes');
			end
		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   Mesh:')); 

			if obj.dimension==3,
				disp(sprintf('\n      Elements and vertices of the original 2d mesh:'));
				fielddisplay(obj,'numberofelements2d','number of elements');
				fielddisplay(obj,'numberofvertices2d','number of vertices');
				fielddisplay(obj,'elements2d','vertex indices of the mesh elements');
				fielddisplay(obj,'x2d','vertices x coordinate [m]');
				fielddisplay(obj,'y2d','vertices y coordinate [m]');

				disp(sprintf('\n      Elements and vertices of the extruded 3d mesh:'));
			else
				disp(sprintf('\n      Elements and vertices:'));
			end
			fielddisplay(obj,'numberofelements','number of elements');
			fielddisplay(obj,'numberofvertices','number of vertices');
			fielddisplay(obj,'elements','vertex indices of the mesh elements');
			fielddisplay(obj,'x','vertices x coordinate [m]');
			fielddisplay(obj,'y','vertices y coordinate [m]');
			fielddisplay(obj,'z','vertices z coordinate [m]');
			fielddisplay(obj,'edges','edges of the 2d mesh (vertex1 vertex2 element1 element2)');
			fielddisplay(obj,'numberofedges','number of edges of the 2d mesh');

			disp(sprintf('\n      Properties:'));
			fielddisplay(obj,'dimension','mesh dimension');
			fielddisplay(obj,'numberoflayers','number of extrusion layers');
			fielddisplay(obj,'vertexonbed','lower vertices flags list');
			fielddisplay(obj,'elementonbed','lower elements flags list');
			fielddisplay(obj,'vertexonsurface','upper vertices flags list');
			fielddisplay(obj,'elementonsurface','upper elements flags list');
			fielddisplay(obj,'uppervertex','upper vertex list (NaN for vertex on the upper surface)');
			fielddisplay(obj,'upperelements','upper element list (NaN for element on the upper layer)');
			fielddisplay(obj,'lowervertex','lower vertex list (NaN for vertex on the lower surface)');
			fielddisplay(obj,'lowerelements','lower element list (NaN for element on the lower layer');
			fielddisplay(obj,'vertexonboundary','vertices on the boundary of the domain flag list');

			fielddisplay(obj,'segments','edges on domain boundary (vertex1 vertex2 element)');
			fielddisplay(obj,'segmentmarkers','number associated to each segment');
			fielddisplay(obj,'vertexconnectivity','list of vertices connected to vertex_i');
			fielddisplay(obj,'elementconnectivity','list of vertices connected to element_i');
			fielddisplay(obj,'average_vertex_connectivity','average number of vertices connected to one vertex');

			disp(sprintf('\n      Extracted model:'));
			fielddisplay(obj,'extractedvertices','vertices extracted from the model');
			fielddisplay(obj,'extractedelements','elements extracted from the model');

			disp(sprintf('\n      Projection:'));
			fielddisplay(obj,'lat','vertices latitude [degrees]');
			fielddisplay(obj,'long','vertices longitude [degrees]');
			fielddisplay(obj,'hemisphere','Indicate hemisphere ''n'' or ''s'' ');
		end % }}}
		function marshall(obj,md,fid) % {{{
			WriteData(fid,'object',obj,'fieldname','x','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'fieldname','y','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'fieldname','z','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'fieldname','elements','format','DoubleMat','mattype',2);
			WriteData(fid,'object',obj,'fieldname','dimension','format','Integer');
			WriteData(fid,'object',obj,'fieldname','numberoflayers','format','Integer');
			WriteData(fid,'object',obj,'fieldname','numberofelements','format','Integer');
			WriteData(fid,'object',obj,'fieldname','numberofvertices','format','Integer');
			WriteData(fid,'object',obj,'fieldname','numberofedges','format','Integer');
			WriteData(fid,'object',obj,'fieldname','elementonbed','format','BooleanMat','mattype',2);
			WriteData(fid,'object',obj,'fieldname','elementonsurface','format','BooleanMat','mattype',2);
			WriteData(fid,'object',obj,'fieldname','vertexonbed','format','BooleanMat','mattype',1);
			WriteData(fid,'object',obj,'fieldname','vertexonsurface','format','BooleanMat','mattype',1);
			WriteData(fid,'object',obj,'fieldname','lowerelements','format','DoubleMat','mattype',2);
			WriteData(fid,'object',obj,'fieldname','upperelements','format','DoubleMat','mattype',2);
			WriteData(fid,'object',obj,'fieldname','edges','format','DoubleMat','mattype',3);
			WriteData(fid,'object',obj,'fieldname','elementconnectivity','format','DoubleMat','mattype',3);
			WriteData(fid,'object',obj,'fieldname','average_vertex_connectivity','format','Integer');
			WriteData(fid,'object',obj,'fieldname','elements2d','format','DoubleMat','mattype',3);
			WriteData(fid,'object',obj,'fieldname','numberofvertices2d','format','Integer');
			WriteData(fid,'object',obj,'fieldname','numberofelements2d','format','Integer');
		end % }}}
	end
end
