/*!\file MpiDenseMumpsSolve.cpp
 * \brief: solve dense matrix system with MUMPS
 */

/*Header files: {{{*/
#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../../shared/Numerics/types.h"
#include "../../shared/MemOps/MemOps.h"
#include "../../shared/Exceptions/exceptions.h"
#include "../../shared/io/Comm/Comm.h"
#include "../mpi/patches/mpipatches.h"

/*Mumps header files: */
#include "dmumps_c.h"

#define JOB_INIT -1 
#define JOB_SOLVE 6
#define JOB_END  -2
/*}}}*/

void MpiDenseMumpsSolve( /*output: */ IssmPDouble* uf, int uf_M, int uf_m, /*matrix input: */ IssmPDouble* Kff, int Kff_M, int Kff_N, int Kff_m, /*right hand side vector: */ IssmPDouble* pf, int pf_M, int pf_m){ /*{{{*/

	/*Variables: {{{*/

	MPI_Comm   comm;
	int        my_rank;
	int        num_procs;
	int        i;
	int        j;
	int         nnz       ,local_nnz;
	int        *irn_loc = NULL;
	int        *jcn_loc = NULL;
	IssmPDouble *a_loc   = NULL;
	int         count;
	int         lower_row;
	int         upper_row;
	IssmPDouble* rhs=NULL;
	int*        recvcounts=NULL;
	int*        displs=NULL;
	/*}}}*/
	/*Communicator info:{{{ */
	my_rank=IssmComm::GetRank();
	num_procs=IssmComm::GetSize();
	comm=IssmComm::GetComm();
	/*}}}*/
	/*First, some checks:{{{ */
	if (Kff_M!=Kff_N)_error_("stiffness matrix Kff should be square");
	if (uf_M!=Kff_M | uf_M!=pf_M)_error_("solution vector should be the same size as stiffness matrix Kff and load vector pf");
	if (uf_m!=Kff_m | uf_m!=pf_m)_error_("solution vector should be locally the same size as stiffness matrix Kff and load vector pf");
	/*}}}*/
	/*Initialize mumps: {{{*/
	DMUMPS_STRUC_C id;
	id.job          = JOB_INIT;
	id.par          = 1;  
	id.sym          = 0;
	id.comm_fortran = MPI_Comm_c2f(comm);
	dmumps_c(&id);
	/*}}}*/
	/*Control statements:{{{ */
	id.icntl[1-1] = 6; //error verbose
	id.icntl[2-1] = 1; //std verbose
	id.icntl[4-1] = 4; //verbose everything
	id.icntl[5-1] = 0;
	id.icntl[18-1] = 3;

	id.icntl[20-1] = 0;
	id.icntl[21-1] = 0;
	id.icntl[30-1] = 0;
	/*}}}*/
	/*Initialize matrix:{{{ */
	id.n=Kff_M;

	/*figure out number of non-zero entries: */
	local_nnz=0;
	for(i=0;i<Kff_m;i++){
		for(j=0;j<Kff_N;j++){
			if (Kff[i*Kff_N+j]!=0)local_nnz++;
		}
	}

	MPI_Reduce(&local_nnz,&nnz,1,MPI_INT,MPI_SUM,0,comm);
	MPI_Bcast(&nnz,1,MPI_INT,0,comm);
	id.nz=nnz;
	id.nz_loc=local_nnz;

	/*Allocate: */
	if(local_nnz){
		irn_loc=xNew<int>(local_nnz);
		jcn_loc=xNew<int>(local_nnz);
		a_loc=xNew<IssmPDouble>(local_nnz);
	}

	/*Populate the triplets: */
	GetOwnershipBoundariesFromRange(&lower_row,&upper_row,Kff_m,comm);
	count=0;
	for(i=0;i<Kff_m;i++){
		for(j=0;j<Kff_N;j++){
			if (Kff[i*Kff_N+j]!=0){
				irn_loc[count]=lower_row+i+1; //fortran indexing
				jcn_loc[count]=j+1; //fortran indexing
				a_loc[count]=Kff[i*Kff_N+j];
				count++;
			}
		}
	}
	id.irn_loc=irn_loc;
	id.jcn_loc=jcn_loc;
	id.a_loc=a_loc;

	/*Deal with right hand side. We need to MPI_Gather it onto cpu 0: */
	rhs=xNew<IssmPDouble>(pf_M);

	recvcounts=xNew<int>(num_procs);
	displs=xNew<int>(num_procs);

	/*recvcounts:*/
	MPI_Allgather(&pf_m,1,MPI_INT,recvcounts,1,MPI_INT,comm);

	/*displs: */
	MPI_Allgather(&lower_row,1,MPI_INT,displs,1,MPI_INT,comm);

	/*Gather:*/
	MPI_Gatherv(pf, pf_m, MPI_DOUBLE, rhs, recvcounts, displs, MPI_DOUBLE,0,comm);
	id.rhs=rhs;
	id.nrhs=1;
	id.lrhs=1;

	/*}}}*/
	/*Solve system: {{{*/
	id.job = JOB_SOLVE;
	dmumps_c (&id);
	/*}}}*/
	/*Now scatter from cpu 0 to all other cpus: {{{*/
	MPI_Scatterv( rhs, recvcounts, displs, MPI_DOUBLE, uf, uf_m, MPI_DOUBLE, 0, comm); 

	/*}}}*/
	/*Cleanup: {{{*/
	id.job = JOB_END; 
	dmumps_c (&id);

	xDelete<int>(irn_loc);
	xDelete<int>(jcn_loc);
	xDelete<IssmPDouble>(a_loc);
	xDelete<IssmPDouble>(rhs);
	xDelete<int>(recvcounts);
	xDelete<int>(displs);

	/*}}}*/
} /*}}}*/

#ifdef _HAVE_ADOLC_
void MpiDenseMumpsSolve( /*output: */ IssmDouble* uf, int uf_M, int uf_m, /*matrix input: */ IssmDouble* Kff, int Kff_M, int Kff_N, int Kff_m, /*right hand side vector: */ IssmDouble* pf, int pf_M, int pf_m){ /*{{{*/
	_error_("not supported yet!");
} /*}}}*/
#endif
