%SETTINGS class definition
%
%   Usage:
%      settings=settings();

classdef settings
	properties (SetAccess=public) 
		io_gather           = 0;
		lowmem              = 0;
		results_as_patches  = 0;
		output_frequency    = 0;
		waitonlock          = 0;
		upload_server       = '';
		upload_path         = '';
		upload_login        = '';
		upload_port         = 0;
		upload_filename     = '';

	end
	methods
		function obj = settings(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%are we short in memory ? (0 faster but requires more memory)
			obj.lowmem=0;

			%i/o:
			obj.io_gather=1;

			%results frequency by default every step
			obj.output_frequency=1;

			%do not use patches by default (difficult to plot)
			obj.results_as_patches=0;

			%this option can be activated to load automatically the results
			%onto the model after a parallel run by waiting for the lock file
			%N minutes that is generated once the solution has converged
			%0 to deactivate
			obj.waitonlock=Inf;
		
			%upload options: 
			upload_port         = 0;

		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{

			md = checkfield(md,'settings.io_gather','numel',[1],'values',[0 1]);
			md = checkfield(md,'settings.lowmem','numel',[1],'values',[0 1]);
			md = checkfield(md,'settings.results_as_patches','numel',[1],'values',[0 1]);
			md = checkfield(md,'settings.output_frequency','numel',[1],'>=',1);
			md = checkfield(md,'settings.waitonlock','numel',[1]);

		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   general settings parameters:'));

			fielddisplay(obj,'io_gather','I/O gathering strategy for result outputs (default 1)');
			fielddisplay(obj,'lowmem','is the memory limited ? (0 or 1)');
			fielddisplay(obj,'results_as_patches','provide results as patches for each element (0 or 1)');
			fielddisplay(obj,'output_frequency','frequency at which results are saved in all solutions with multiple time_steps');
			fielddisplay(obj,'waitonlock','maximum number of minutes to wait for batch results (NaN to deactivate)');
			fielddisplay(obj,'upload_server','server hostname where model should be uploaded');
			fielddisplay(obj,'upload_path','path on server where model should be uploaded');
			fielddisplay(obj,'upload_login','server login');
			fielddisplay(obj,'upload_port','port login (default is 0)');
			fielddisplay(obj,'upload_filename','unique id generated when uploading the file to server');

		end % }}}
		function marshall(obj,md,fid) % {{{
			WriteData(fid,'object',obj,'fieldname','io_gather','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','lowmem','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','results_as_patches','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','output_frequency','format','Integer');
			if obj.waitonlock>0,
				WriteData(fid,'enum',SettingsWaitonlockEnum(),'data',true,'format','Boolean');
			else
				WriteData(fid,'enum',SettingsWaitonlockEnum(),'data',false,'format','Boolean');
			end
		end % }}}
	end
end
