from fielddisplay import fielddisplay
from EnumDefinitions import *
from StringToEnum import StringToEnum
from checkfield import *
from WriteData import *

class prognostic(object):
	"""
	PROGNOSTIC class definition

	   Usage:
	      prognostic=prognostic();
	"""

	def __init__(self): # {{{
		self.spcthickness           = float('NaN')
		self.min_thickness          = 0
		self.hydrostatic_adjustment = 0
		self.stabilization          = 0
		self.vertex_pairing         = float('NaN')
		self.penalty_factor         = 0
		self.requested_outputs      = float('NaN')

		#set defaults
		self.setdefaultparameters()

		#}}}
	def __repr__(self): # {{{
		string='   Prognostic solution parameters:'
		string="%s\n%s"%(string,fielddisplay(self,'spcthickness','thickness constraints (NaN means no constraint) [m]'))
		string="%s\n%s"%(string,fielddisplay(self,'min_thickness','minimum ice thickness allowed [m]'))
		string="%s\n%s"%(string,fielddisplay(self,'hydrostatic_adjustment','adjustment of ice shelves surface and bed elevations: ''Incremental'' or ''Absolute'' '))
		string="%s\n%s"%(string,fielddisplay(self,'stabilization','0: no, 1: artificial_diffusivity, 2: streamline upwinding, 3: discontinuous Galerkin'))
		string="%s\n%s"%(string,fielddisplay(self,'requested_outputs','additional outputs requested'))

		return string
		#}}}
	def setdefaultparameters(self): # {{{
		
		#Type of stabilization to use 0:nothing 1:artificial_diffusivity 3:Discontinuous Galerkin
		self.stabilization=1

		#Factor applied to compute the penalties kappa=max(stiffness matrix)*10^penalty_factor
		self.penalty_factor=3

		#Minimum ice thickness that can be used
		self.min_thickness=1

		#Hydrostatic adjustment
		self.hydrostatic_adjustment='Absolute'

		return self
	#}}}
	def checkconsistency(self,md,solution,analyses):    # {{{

		#Early return
		if (PrognosticAnalysisEnum() not in analyses) or (solution==TransientSolutionEnum() and not md.transient.isprognostic):
			return md

		md = checkfield(md,'prognostic.spcthickness','forcing',1)
		md = checkfield(md,'prognostic.hydrostatic_adjustment','values',['Absolute','Incremental'])
		md = checkfield(md,'prognostic.stabilization','values',[0,1,2,3])
		md = checkfield(md,'prognostic.min_thickness','>',0)
		if not md.prognostic.requested_outputs:
			md = checkfield(md,'prognostic.requested_outputs','size',[float('NaN'),1])

		return md
	# }}}
	def marshall(self,md,fid):    # {{{
		WriteData(fid,'object',self,'fieldname','spcthickness','format','DoubleMat','mattype',1,'forcinglength',md.mesh.numberofvertices+1)
		WriteData(fid,'object',self,'fieldname','min_thickness','format','Double')
		WriteData(fid,'data',StringToEnum(self.hydrostatic_adjustment)[0],'format','Integer','enum',PrognosticHydrostaticAdjustmentEnum())
		WriteData(fid,'object',self,'fieldname','stabilization','format','Integer')
		WriteData(fid,'object',self,'fieldname','vertex_pairing','format','DoubleMat','mattype',3)
		WriteData(fid,'object',self,'fieldname','penalty_factor','format','Double')
		WriteData(fid,'object',self,'fieldname','requested_outputs','format','DoubleMat','mattype',3);
	# }}}
