/*!\file: solutionsequence_hydro_nonlinear.cpp
 * \brief: core of the hydro solution 
 */ 

#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../shared/shared.h"
#include "../modules/modules.h"

void solutionsequence_hydro_nonlinear(FemModel* femmodel){

	/*solution : */
	Vector<IssmDouble>* ug_sed=NULL;
	Vector<IssmDouble>* ug_epl=NULL; 
	Vector<IssmDouble>* uf=NULL;
	Vector<IssmDouble>* old_uf=NULL; 
	Vector<IssmDouble>* aged_ug_sed=NULL; 
	Vector<IssmDouble>* aged_ug_epl=NULL; 
	Vector<IssmDouble>* ys=NULL; 
	Vector<IssmDouble>* dug=NULL; 

	Matrix<IssmDouble>* Kff=NULL;
	Matrix<IssmDouble>* Kfs=NULL;
	Vector<IssmDouble>* pf=NULL;
	Vector<IssmDouble>* df=NULL;

	bool       sedconverged,eplconverged,hydroconverged;
	bool       isefficientlayer;
	int        constraints_converged;
	int        num_unstable_constraints;
	int        sedcount,eplcount,hydrocount;
	int        hydro_maxiter;
	IssmDouble sediment_kmax,time;
	IssmDouble eps_hyd;
	IssmDouble ndu_sed,nu_sed;
	IssmDouble ndu_epl,nu_epl;	

	/*Recover parameters: */
	femmodel->SetCurrentConfiguration(HydrologyDCInefficientAnalysisEnum);//FIXME
	femmodel->parameters->FindParam(&isefficientlayer,HydrologydcIsefficientlayerEnum);
	femmodel->parameters->FindParam(&eps_hyd,HydrologydcRelTolEnum);
	femmodel->parameters->FindParam(&time,TimeEnum);
	femmodel->BasisIntegralsx();
	hydro_maxiter=100;
	hydrocount=1;
	sedconverged=false;
	eplconverged=false;
	hydroconverged=false;

	/*Iteration on the two layers + transfer*/
	femmodel->SetCurrentConfiguration(HydrologyDCInefficientAnalysisEnum);
	GetSolutionFromInputsx(&ug_sed, femmodel->elements, femmodel->nodes, femmodel->vertices,femmodel->loads, femmodel->materials, femmodel->parameters);
	if(isefficientlayer) {
		femmodel->SetCurrentConfiguration(HydrologyDCEfficientAnalysisEnum);
		GetSolutionFromInputsx(&ug_epl, femmodel->elements, femmodel->nodes, femmodel->vertices,femmodel->loads, femmodel->materials, femmodel->parameters);
}

	for(;;){
		sedcount=1;
		eplcount=1;
		//save pointer to old velocity
		delete aged_ug_sed;
		aged_ug_sed=ug_sed;
		if(isefficientlayer){
			delete aged_ug_epl;
			aged_ug_epl=ug_epl;
		}

		femmodel->SetCurrentConfiguration(HydrologyDCInefficientAnalysisEnum);
		InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,true,ResetPenaltiesEnum);
		InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,false,ConvergedEnum);
		femmodel->UpdateConstraintsx();
		femmodel->parameters->SetParam(HydrologySedimentEnum,HydrologyLayerEnum);

		/*Iteration on the sediment layer*/
		for(;;){
			femmodel->HydrologyTransferx();
			femmodel->SystemMatricesx(&Kff, &Kfs, &pf,&df, &sediment_kmax);
			CreateNodalConstraintsx(&ys,femmodel->nodes,HydrologyDCInefficientAnalysisEnum);
			Reduceloadx(pf,Kfs,ys); delete Kfs;
			delete uf;
			Solverx(&uf, Kff, pf,old_uf, df, femmodel->parameters);
			delete old_uf; old_uf=uf->Duplicate();
			if(sedcount>1)delete ug_sed; /*Not on first time to avoid deleting aged_ug_sed*/
			delete Kff; delete pf; delete df;

			Mergesolutionfromftogx(&ug_sed,uf,ys,femmodel->nodes,femmodel->parameters); delete ys;
			InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,ug_sed);
			ConstraintsStatex(&constraints_converged,&num_unstable_constraints, femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);

			if (!sedconverged){
				if(VerboseConvergence()) _printf0_("   #unstable constraints = " << num_unstable_constraints << "\n");
				if(num_unstable_constraints==0) sedconverged = true;
				if (sedcount>=hydro_maxiter){
					_error_("   maximum number of Sediment iterations (" << hydro_maxiter << ") exceeded");
				}
			}
			sedcount++;

			if(sedconverged){
				femmodel->parameters->SetParam(sediment_kmax,HydrologySedimentKmaxEnum);
				InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,sedconverged,ConvergedEnum);
				InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,ug_sed);
				InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,sediment_kmax,HydrologySedimentKmaxEnum);
				break;
			}
		}

		/*Second layer*/
		if(!isefficientlayer) break;
		femmodel->SetCurrentConfiguration(HydrologyDCEfficientAnalysisEnum);
		InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,true,ResetPenaltiesEnum);
		InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,false,ConvergedEnum);
		femmodel->UpdateConstraintsx();
		femmodel->parameters->SetParam(HydrologyEfficientEnum,HydrologyLayerEnum);
		eplconverged = false;
		/*Iteration on the EPL layer*/
		for(;;){
			femmodel->HydrologyTransferx();
			femmodel->SystemMatricesx(&Kff, &Kfs, &pf,&df,NULL);
			CreateNodalConstraintsx(&ys,femmodel->nodes,HydrologyDCEfficientAnalysisEnum);
			Reduceloadx(pf,Kfs,ys); delete Kfs;
			delete uf;
			Solverx(&uf, Kff, pf,old_uf, df, femmodel->parameters);
			delete old_uf; old_uf=uf->Duplicate();			
			if(eplcount>1) delete ug_epl; 
			delete Kff;delete pf;
			delete df;
			Mergesolutionfromftogx(&ug_epl,uf,ys,femmodel->nodes,femmodel->parameters); delete ys;
			InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,ug_epl);
			ConstraintsStatex(&constraints_converged,&num_unstable_constraints, femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);

			if (!eplconverged){
				if(VerboseConvergence()) _printf0_("   #unstable constraints = " << num_unstable_constraints << "\n");
				if(num_unstable_constraints==0) eplconverged = true;
				if (eplcount>=hydro_maxiter){
					_error_("   maximum number of EPL iterations (" << hydro_maxiter << ") exceeded");
				}
			}
			eplcount++;

			if(eplconverged){
				InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,eplconverged,ConvergedEnum);
				InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,sediment_kmax,MeltingOffsetEnum);
				InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,ug_epl);
				break;
			}
		}

		/*System convergence check*/
		if(!hydroconverged){
			//compute norm(du)/norm(u)
			dug=ug_sed->Duplicate(); _assert_(dug);
			aged_ug_sed->Copy(dug);	
			dug->AYPX(ug_sed,-1.0);
			ndu_sed=dug->Norm(NORM_TWO); nu_sed=aged_ug_sed->Norm(NORM_TWO);
			if (xIsNan<IssmDouble>(ndu_sed) || xIsNan<IssmDouble>(nu_sed)) _error_("Sed convergence criterion is NaN!");
			if (!xIsNan<IssmDouble>(eps_hyd)){
				if (!isefficientlayer){
					if ((ndu_sed/nu_sed)<eps_hyd){
						if(VerboseConvergence()) _printf0_(setw(50) << left << "   Converged");
						hydroconverged=true;
					}
					else{ 
						if(VerboseConvergence()) _printf0_(setw(50) << left << "   Sediment Convergence criterion:" << ndu_sed/nu_sed*100 << " > " << eps_hyd*100 << " %\n");
						hydroconverged=false;
					}
				}
				else{
					dug=ug_epl->Duplicate();_assert_(dug); 
					aged_ug_epl->Copy(dug);_assert_(aged_ug_epl); 
					dug->AYPX(ug_epl,-1.0);
					ndu_epl=dug->Norm(NORM_TWO); 
					nu_epl=aged_ug_epl->Norm(NORM_TWO);

					if (xIsNan<IssmDouble>(ndu_epl) || xIsNan<IssmDouble>(nu_epl)) _error_("EPL convergence criterion is NaN!");
					if (ndu_epl==0.0 && nu_epl==0.0) nu_epl=1.0e-6; /*Hacking the case where the EPL is used but empty*/
					if ((ndu_epl/nu_epl)<eps_hyd && (ndu_sed/nu_sed)<eps_hyd){
						if (VerboseConvergence()) _printf0_(setw(50) << left << "   Converged");
						hydroconverged=true;
					}
					else{ 
						if(VerboseConvergence()) _printf0_(setw(50) << left << "   Sediment Convergence criterion:" << ndu_sed/nu_sed*100 << " aiming lower than " << eps_hyd*100 << " %\n");
						if(VerboseConvergence()) _printf0_(setw(50) << left << "   EPL Convergence criterion:" << ndu_epl/nu_epl*100 << " aiming lower than " << eps_hyd*100 << " %\n");
						hydroconverged=false;
					}
				}
			}
			else _printf0_(setw(50) << left << "   Convergence criterion:" << ndu_sed/nu_sed*100 << " %\n");
			if (hydrocount>=hydro_maxiter){
				_error_("   maximum number for hydrological global iterations (" << hydro_maxiter << ") exceeded");
			}
		}
		hydrocount++;
		if(hydroconverged)break;
	}

	InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,ug_sed);
	InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,ug_epl);
	
	/*Free ressources: */
	delete ug_epl;
	delete ug_sed;
	delete uf;
	delete old_uf;
	delete aged_ug_sed;
	delete aged_ug_epl;
	delete dug;

}
