% This file can be run to check that the current version of issm is giving 
% coherent results. This test deals with an icesheet with icefront for a 3d model. The geometry 
% is square. Just run this file in Matlab, with a properly setup ISSM code. 
% The results of this test will indicate if there is a difference between current computations 
% and archived results.

% Errors  between archived results and the current version will get flagged if they are not within 
% a certain tolerance. The current tolerance is 10^-12. If you have good reasons to believe this 
% tolerance should be lowered (for example, if you are running single precision compilers?), feel 
% free to tweak the tolerance variable.

%packages and solutions to be tested
packages={'macayeal','ice','cielo_serial','cielo_parallel'};
solutions={'diagnostic','thermal_steady','thermal_transient','prognostic','transient'};

%Initialize log message for nightly runs.
testname='IceSheeoIceFrontM3d_8';
tolerance=10^-12;

%go through the solutions requested
for i=1:length(packages),
	package=packages{i};

	for j=1:length(solutions),
		solution=solutions{j};

		%check package
		if ~(strcmpi(package,'macayeal') | strcmpi(package,'ice') | strcmpi(package,'cielo_serial') | strcmpi(package,'cielo_parallel'));
			error(['package: ' package  ' in test: ' testname  ' not supported yet']);
		%check solution
		elseif ~(strcmpi(solution,'diagnostic') | strcmpi(solution,'thermal_steady') | strcmpi(solution,'thermal_transient') |...
				strcmpi(solution,'prognostic') | strcmpi(solution,'transient'));
			error(['solution: ' solution  ' in test: ' testname  ' not supported yet']);
		end
		%check solution requested
		if (~strcmpi(package,'ice')),
			disp(sprintf(['\nsolution: ' solution  ', with package: ' package  ', in test: ' testname  ', not supported yet.\n']));
			continue
		end

		%initialize model
		md=model;
		md=mesh(md,'DomainOutline.exp',50000);
		md=geography(md,'','');
		md=parameterize(md,'Square.par');
		md=extrude(md,10,3);
		md=setelementstype(md,'macayeal','all');
		if strcmpi(package,'cielo_parallel'), md.cluster='wilkes'; end
		if md.numberofgrids==3880
			load Velocities; md.vx=1.2*vx; md.vy=1.2*vy;
		end

		%compute solution
		if strcmpi(package,'cielo_parallel') & strcmpi(solution,'diagnostic'),
			md=solve(md,'diagnostic_horiz','cielo');
		elseif strcmpi(package,'cielo_serial'),
			eval(['md=solve(md,''' solution ''',''cielo'');']);
		else
			eval(['md=solve(md,''' solution ''',''' package ''');']);
		end

		%compute field to be checked
		if strcmpi(solution,'diagnostic'),
			fields={'vy','vz'};
		elseif strcmpi(solution,'thermal_steady'),
			fields={'temperature','melting'};
		elseif strcmpi(solution,'thermal_transient'),
			fields={'thermaltransient_results(end).temperature','thermaltransient_results(end).melting'};
		elseif strcmpi(solution,'prognostic'),
			fields={'new_thickness'};
		elseif strcmpi(solution,'transient'),
			fields={'transient_results(end).vy','transient_results(end).vz','transient_results(end).temperature','transient_results(end).melting','transient_results(end).thickness'};
		end

		%load archive
		eval(['load Archive' package solution ]);

		for k=1:length(fields),
			field=fields{k};

			%compare to archive
			eval(['Archive=Archive' package solution '_field' num2str(k) ';']);
			eval(['error_diff=abs(norm((Archive(find(Archive))-md.' field  '(find(Archive)))./Archive(find(Archive)),2));']);

			%disp test result
			if (error_diff>tolerance);
				disp(sprintf(['\n\nERROR (difference = %8.3g > %g)   --->   test: ' testname ', solution: ' solution  ', package: ' package ', field tested: ' field  '.\n\n'],error_diff,tolerance));
			else
				disp(sprintf(['\n\nSUCCESS (difference = %8.3g < %g)   --->   test: ' testname ', solution: ' solution  ', package: ' package ', field tested: ' field  '.\n\n'],error_diff,tolerance));
			end

		end
	end
end
