/*\file Shp2Kml.c
 *\brief: shp to kml file conversion mex module.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "./Shp2Kml.h"

void Shp2KmlUsage(void){/*{{{*/
	_printf0_("Shp2Kml - shp to kml file conversion module:" << "\n");
	_printf0_("" << "\n");
	_printf0_("   This module converts a file from shp to kml format." << "\n");
	_printf0_("" << "\n");
	_printf0_("   Usage:" << "\n");
	_printf0_("      [ret]=Shp2Kml(filshp,filkml,sgn,'param name',param,...);" << "\n");
	_printf0_("" << "\n");
	_printf0_("      filshp      file name of shp file to be read (char, extension optional)" << "\n");
	_printf0_("      filkml      file name of kml file to be written (char)" << "\n");
	_printf0_("      sgn         sign for hemisphere (double, +1 (north) -1 (south) or 0 (no translation))" << "\n");
	_printf0_("" << "\n");
	_printf0_("      central_meridian     central meridian (double, optional, but must specify with sp)" << "\n");
	_printf0_("      standard_parallel    standard parallel (double, optional, but must specify with cm)" << "\n");
	_printf0_("" << "\n");
	_printf0_("      ret         return code (non-zero for warning)" << "\n");
	_printf0_("" << "\n");
	_printf0_("   Examples:" << "\n");
	_printf0_("      [ret]=Shp2Kml('file.shp','file.kml', 0);" << "\n");
	_printf0_("      [ret]=Shp2Kml('file.shp','file.kml', 1,'central_meridian',45,'standard_parallel',70);" << "\n");
	_printf0_("      [ret]=Shp2Kml('file.shp','file.kml',-1,'central_meridian', 0,'standard_parallel',71);" << "\n");
	_printf0_("" << "\n");
}/*}}}*/
WRAPPER(Shp2Kml){

	int i,verbose=1;

	/*input: */
	char    *filshp=NULL,*filkml=NULL;
	int     sgn;
	Options* options=NULL;
	double   cm=0.,sp=0.;

	/* output: */
	int     iret=0;

	#ifndef _HAVE_SHAPELIB_ //only works if shapelib library has been compiled in.
	_error_("Shapelib not available! Cannot carry out shp file translation!");
	#endif

	/*Boot module: */
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	if (nlhs > NLHS) {
		Shp2KmlUsage(); _error_("Shp2Kml usage error");
	}
	if (nrhs < NRHS) {
		Shp2KmlUsage(); _error_("Shp2Kml usage error");
	}

	/*Input datasets: */
	FetchData(&filshp,SHP_IN);
	FetchData(&filkml,KML_IN);
	FetchData(&sgn,SGN_IN);
	FetchData(&options,NRHS,nrhs,prhs);

	/*  defaults are in Xy2lldef, so don't duplicate them here, and only use user values if both have been specified  */
	if (options->GetOption("central_meridian") || options->GetOption("standard_parallel")) {
		options->Get(&cm,"central_meridian");
		if (verbose) _printf_("  cm=" << cm << "\n");
		options->Get(&sp,"standard_parallel");
		if (verbose) _printf_("  sp=" << sp << "\n");
	}

	/*some checks*/
	if (sgn < -1 || sgn > +1) _error_("Hemisphere sgn=" << sgn << " must be +1 (north), -1 (south), or 0 (no translation).");
	if (fabs(cm)      > 180.) _error_("Central meridian cm=" << cm << " must be between -180 (west) and +180 (east) degrees.");
	if (sp < 0. || sp >  90.) _error_("Standard parallel sp=" << sp << " must be between 0 and 90 degrees (in specified hemisphere).");

	/* Run core computations: */
	if (options->GetOption("central_meridian") && options->GetOption("standard_parallel"))
		iret=Shp2Kmlx(filshp,filkml,sgn,cm,sp);
	else
		iret=Shp2Kmlx(filshp,filkml,sgn);

	/*Write data: */
	WriteData(RET_OUT,iret);

	/*Clean-up*/
	delete options;
	xDelete<char>(filkml);
	xDelete<char>(filshp);

	/*end module: */
	MODULEEND();
}
