/*\file Exp2Kml.c
 *\brief: exp to kml file conversion mex module.
 */
#include "./Exp2Kml.h"

void Exp2KmlUsage(void){/*{{{*/
	_printf0_("Exp2Kml - exp to kml file conversion module:" << "\n");
	_printf0_("" << "\n");
	_printf0_("   This module converts a file from exp to kml format." << "\n");
	_printf0_("" << "\n");
	_printf0_("   Usage:" << "\n");
	_printf0_("      [ret]=Exp2Kml(filexp,filkml,sgn,'param name',param,...);" << "\n");
	_printf0_("" << "\n");
	_printf0_("      filexp      file name of exp file to be read (char)" << "\n");
	_printf0_("      filkml      file name of kml file to be written (char)" << "\n");
	_printf0_("      sgn         sign for hemisphere (double, +1 (north) or -1 (south))" << "\n");
	_printf0_("" << "\n");
	_printf0_("      central_meridian     central meridian (double, optional, but must specify with sp)" << "\n");
	_printf0_("      standard_parallel    standard parallel (double, optional, but must specify with cm)" << "\n");
	_printf0_("      holes       flag for treatment of multiple profiles (char, optional, 'yes' for holes))" << "\n");
	_printf0_("" << "\n");
	_printf0_("      ret         return code (non-zero for warning)" << "\n");
	_printf0_("" << "\n");
	_printf0_("   Examples:" << "\n");
	_printf0_("      [ret]=Exp2Kml('file.exp','file.kml', 1);" << "\n");
	_printf0_("      [ret]=Exp2Kml('file.exp','file.kml', 1,'central_meridian',45,'standard_parallel',70,'holes','yes');" << "\n");
	_printf0_("      [ret]=Exp2Kml('file.exp','file.kml',-1,'central_meridian', 0,'standard_parallel',71,'holes','yes');" << "\n");
	_printf0_("" << "\n");
}/*}}}*/
WRAPPER(Exp2Kml){

	int i,verbose=1;

	/*input: */
	char    *filexp=NULL,*filkml=NULL;
	int     sgn;

	Options* options=NULL;
	char     *choles=NULL;
	bool     holes=false;
	double   cm=0.,sp=0.;

	/* output: */
	int     iret=0;

	/*Boot module: */
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	if (nlhs > NLHS) {
		Exp2KmlUsage(); _error_("Exp2Kml usage error");
	}
	if (nrhs < NRHS) {
		Exp2KmlUsage(); _error_("Exp2Kml usage error");
	}

	/*Input datasets: */
	FetchData(&filexp,EXP_IN);
	FetchData(&filkml,KML_IN);
	FetchData(&sgn,SGN_IN);
	FetchData(&options,NRHS,nrhs,prhs);

	options->Get(&choles,"holes",(char*)"no");
	if (!strncmp(choles,"y",1) || !strncmp(choles,"on",2)) holes=true;

	/*  defaults are in Xy2lldef, so don't duplicate them here, and only use user values if both have been specified  */
	if (options->GetOption("central_meridian") || options->GetOption("standard_parallel")) {
		options->Get(&cm,"central_meridian");
		if (verbose) _printf_("  cm=" << cm << "\n");
		options->Get(&sp,"standard_parallel");
		if (verbose) _printf_("  sp=" << sp << "\n");
	}

	/*some checks*/
	if (sgn !=+1 && sgn !=-1) _error_("Hemisphere sgn=" << sgn << " must be +1 (north) or -1 (south).");
	if (fabs(cm)      > 180.) _error_("Central meridian cm=" << cm << " must be between -180 (west) and +180 (east) degrees.");
	if (sp < 0. || sp >  90.) _error_("Standard parallel sp=" << sp << " must be between 0 and 90 degrees (in specified hemisphere).");

	/* Run core computations: */
	if (options->GetOption("central_meridian") && options->GetOption("standard_parallel"))
		iret=Exp2Kmlx(filexp,filkml,sgn,cm,sp,holes);
	else
		iret=Exp2Kmlx(filexp,filkml,sgn,holes);

	/*Write data: */
	WriteData(RET_OUT,iret);

	/*Clean-up*/
	xDelete<char>(choles);
	xDelete<char>(filkml);
	xDelete<char>(filexp);
	delete options;

	/*end module: */
	MODULEEND();
}
