/*!\file: solutionsequence_hydro_nonlinear.cpp
 * \brief: core of the hydro solution 
 */ 

#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../shared/shared.h"
#include "../modules/modules.h"

void solutionsequence_hydro_nonlinear(FemModel* femmodel){
	
	/*solution : */
	Vector<IssmDouble>* ug=NULL; 
	Vector<IssmDouble>* uf_sed=NULL; 
	Vector<IssmDouble>* uf_epl=NULL; 
	Vector<IssmDouble>* old_uf=NULL; 
	Vector<IssmDouble>* aged_uf_sed=NULL; 
	Vector<IssmDouble>* aged_uf_epl=NULL; 
	Vector<IssmDouble>* ys=NULL; 
	Vector<IssmDouble>* duf=NULL; 
	
	Matrix<IssmDouble>* Kff=NULL;
	Matrix<IssmDouble>* Kfs=NULL;
	Vector<IssmDouble>* pf=NULL;
	Vector<IssmDouble>* df=NULL;
	
	bool       sedconverged,eplconverged,hydroconverged;
	bool       isefficientlayer;
	int        constraints_converged;
	int        num_unstable_constraints;
	int        sedcount,eplcount,hydrocount;
	int        hydro_maxiter;
	IssmDouble sediment_kmax,time;
	IssmDouble eps_hyd;
	IssmDouble ndu_sed,nu_sed;
	IssmDouble ndu_epl,nu_epl;	

	/*Recover parameters: */
	femmodel->SetCurrentConfiguration(HydrologyDCInefficientAnalysisEnum);//FIXME
	femmodel->parameters->FindParam(&isefficientlayer,HydrologydcIsefficientlayerEnum);
	femmodel->parameters->FindParam(&eps_hyd,HydrologydcRelTolEnum);
	femmodel->parameters->FindParam(&time,TimeEnum);
	femmodel->BasisIntegralsx();
	hydro_maxiter=100;
	hydrocount=1;
	sedconverged=false;
	eplconverged=false;
	hydroconverged=false;

	/*Iteration on the two layers + transfer*/
	femmodel->SetCurrentConfiguration(HydrologyDCInefficientAnalysisEnum);
	GetSolutionFromInputsx(&ug, femmodel->elements, femmodel->nodes, femmodel->vertices,femmodel->loads, femmodel->materials, femmodel->parameters);
	Reducevectorgtofx(&uf_sed, ug, femmodel->nodes,femmodel->parameters); _assert_(uf_sed);
	if(isefficientlayer) {
		femmodel->SetCurrentConfiguration(HydrologyDCEfficientAnalysisEnum);
		GetSolutionFromInputsx(&ug, femmodel->elements, femmodel->nodes, femmodel->vertices,femmodel->loads, femmodel->materials, femmodel->parameters);
		Reducevectorgtofx(&uf_epl, ug, femmodel->nodes,femmodel->parameters);
	}


	for(;;){
		sedcount=1;
		eplcount=1;
		//save pointer to old velocity
		delete aged_uf_sed;aged_uf_sed=uf_sed;
		if(isefficientlayer){
			delete aged_uf_epl;
			aged_uf_epl=uf_epl;
		}

		femmodel->SetCurrentConfiguration(HydrologyDCInefficientAnalysisEnum);
		InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,true,ResetPenaltiesEnum);
		InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,false,ConvergedEnum);
		femmodel->UpdateConstraintsx();
		femmodel->parameters->SetParam(HydrologySedimentEnum,HydrologyLayerEnum);
		femmodel->HydrologyTransferx();
		
		/*Iteration on the sediment layer*/
		for(;;){
			femmodel->SystemMatricesx(&Kff, &Kfs, &pf,&df, &sediment_kmax);
			CreateNodalConstraintsx(&ys,femmodel->nodes,HydrologyDCInefficientAnalysisEnum);
			Reduceloadx(pf,Kfs,ys); delete Kfs;
			if(sedcount>1)delete uf_sed;
			Solverx(&uf_sed, Kff, pf,old_uf, df, femmodel->parameters);
			delete old_uf; old_uf=uf_sed->Duplicate();
			delete Kff; delete pf; delete ug; delete df;
			
			Mergesolutionfromftogx(&ug,uf_sed,ys,femmodel->nodes,femmodel->parameters); delete ys;
			InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,ug);
			ConstraintsStatex(&constraints_converged,&num_unstable_constraints, femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);

			if (!sedconverged){
				if(VerboseConvergence()) _pprintLine_("   #unstable constraints = " << num_unstable_constraints);
				if(num_unstable_constraints==0) sedconverged = true;
				if (sedcount>=hydro_maxiter){
					_error_("   maximum number of Sediment iterations (" << hydro_maxiter << ") exceeded");
				}
			}
			sedcount++;

			if(sedconverged){
				femmodel->parameters->SetParam(sediment_kmax,HydrologySedimentKmaxEnum);
				InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,sedconverged,ConvergedEnum);
				InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,ug);
				InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,sediment_kmax,HydrologySedimentKmaxEnum);
				break;
			}
		}

		/*Second layer*/
		if(!isefficientlayer) break;
		femmodel->SetCurrentConfiguration(HydrologyDCEfficientAnalysisEnum);
		InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,true,ResetPenaltiesEnum);
		InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,false,ConvergedEnum);
		femmodel->UpdateConstraintsx();
		femmodel->parameters->SetParam(HydrologyEfficientEnum,HydrologyLayerEnum);
		eplconverged = false;
		/*Iteration on the EPL layer*/
		for(;;){
			femmodel->SystemMatricesx(&Kff, &Kfs, &pf,&df,NULL);
			CreateNodalConstraintsx(&ys,femmodel->nodes,HydrologyDCEfficientAnalysisEnum);
			Reduceloadx(pf,Kfs,ys); delete Kfs;
			if(eplcount>1) delete uf_epl;
			Solverx(&uf_epl, Kff, pf,old_uf, df, femmodel->parameters);
			delete old_uf; old_uf=uf_epl->Duplicate();
			delete Kff;delete pf; delete ug; delete df;
			Mergesolutionfromftogx(&ug,uf_epl,ys,femmodel->nodes,femmodel->parameters); delete ys;
			InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,ug);

			ConstraintsStatex(&constraints_converged,&num_unstable_constraints, femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);

			if (!eplconverged){
				if(VerboseConvergence()) _pprintLine_("   #unstable constraints = " << num_unstable_constraints);
				if(num_unstable_constraints==0) eplconverged = true;
				if (eplcount>=hydro_maxiter){
					_error_("   maximum number of EPL iterations (" << hydro_maxiter << ") exceeded");
				}
			}
			eplcount++;

			if(eplconverged){
				InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,eplconverged,ConvergedEnum);
				InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,sediment_kmax,MeltingOffsetEnum);
				InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,ug);
				break;
			}
		}

		/*System convergence check*/
		if(!hydroconverged){
			//compute norm(du)/norm(u)
			_assert_(aged_uf_sed); _assert_(uf_sed);
			duf=uf_sed->Duplicate(); _assert_(duf);
			aged_uf_sed->Copy(duf);	
			duf->AYPX(uf_sed,-1.0);
			ndu_sed=duf->Norm(NORM_TWO); nu_sed=aged_uf_sed->Norm(NORM_TWO);
			if (xIsNan<IssmDouble>(ndu_sed) || xIsNan<IssmDouble>(nu_sed)) _error_("convergence criterion is NaN!");
			if(isefficientlayer){
				duf=aged_uf_epl->Duplicate(); aged_uf_epl->Copy(duf); duf->AYPX(uf_epl,-1.0);
				ndu_epl=duf->Norm(NORM_TWO); nu_epl=aged_uf_epl->Norm(NORM_TWO);
				if (xIsNan<IssmDouble>(ndu_epl) || xIsNan<IssmDouble>(nu_epl)) _error_("convergence criterion is NaN!");
			}
			//print
			if (!xIsNan<IssmDouble>(eps_hyd)){
				if((ndu_sed/nu_sed)<eps_hyd){
					if(VerboseConvergence()) _pprintLine_(setw(50) << left << "   Sediment Convergence criterion:" << ndu_sed/nu_sed*100 << " < " << eps_hyd*100 << " %");
					hydroconverged=true;
				}
				else{ 
					if(VerboseConvergence()) _pprintLine_(setw(50) << left << "   Sediment Convergence criterion:" << ndu_sed/nu_sed*100 << " > " << eps_hyd*100 << " %");
					hydroconverged=false;
				}
				if(isefficientlayer){
					if((ndu_epl/nu_epl)<eps_hyd){
						if(VerboseConvergence()) _pprintLine_(setw(50) << left << "   EPL Convergence criterion:" << ndu_epl/nu_epl*100 << " < " << eps_hyd*100 << " %");
					}
					else{ 
						if(VerboseConvergence()) _pprintLine_(setw(50) << left << "   EPL Convergence criterion:" << ndu_epl/nu_epl*100 << " > " << eps_hyd*100 << " %");
						hydroconverged=false;
					}
				}
			}
			else _pprintLine_(setw(50) << left << "   Convergence criterion:" << ndu_sed/nu_sed*100 << " %");
			if (hydrocount>=hydro_maxiter){
				_error_("   maximum number for hydrological global iterations (" << hydro_maxiter << ") exceeded");
			}
		}
		hydrocount++;
			
		if(hydroconverged)break;
	}

	InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,ug);
	
	/*Free ressources: */
	delete ug;
	delete uf_sed;
	delete uf_epl;
	delete old_uf;
	delete aged_uf_sed;
	delete aged_uf_epl;
	delete duf;
	
}
