/*  Gauss point structures and prototypes  */
#include <math.h>
#include <float.h>

#include "./GaussPoints.h"
#include "../MemOps/MemOps.h"
#include "../Exceptions/exceptions.h"
#include "../Numerics/constants.h"

/*General Gauss points*/
/*FUNCTION GaussLegendreLinear {{{*/
void GaussLegendreLinear( IssmPDouble** pxgaus, IssmPDouble** pxwgt, int ngaus){
	/* Gauss-Legendre quadrature points.

		The recurrence coefficients for Legendre polynomials on (-1,1)
		are defined (from the ORTHPOL subroutine RECUR with ipoly=1) as:

		alpha(i)=0.
		beta (i)=1./(4.-1./(i-1)^2))

		For degree p, the required number of Gauss-Legendre points is
		n>=(p+1)/2.*/

	/*Intermediaries*/
	int i;
	IssmPDouble *alpha,*beta;

	/*p= 1, npoint= 1*/
	static IssmPDouble wgt1[]={2.000000000000000};
	static IssmPDouble xi1[]={0.000000000000000};

	/*p= 3, npoint= 2*/
	static IssmPDouble wgt2[]={1.000000000000000, 1.000000000000000};
	static IssmPDouble xi2[]={-0.577350269189626, 0.577350269189626};

	/*p= 5, npoint= 3*/
	static IssmPDouble wgt3[]={0.555555555555556, 0.888888888888889, 0.555555555555556};
	static IssmPDouble xi3[]={-0.774596669241483, 0.000000000000000, 0.774596669241483};

	/*p= 7, npoint= 4*/
	static IssmPDouble wgt4[]={0.347854845137454, 0.652145154862546, 0.652145154862546, 0.347854845137454};
	static IssmPDouble xi4[]={-0.861136311594053,-0.339981043584856, 0.339981043584856, 0.861136311594053};

	static IssmPDouble* wgtp[MAX_LINE_GAUS_PTS]={wgt1 ,wgt2 ,wgt3 ,wgt4 };
	static IssmPDouble* xip [MAX_LINE_GAUS_PTS]={xi1  ,xi2  ,xi3  ,xi4  };

	static int np[MAX_LINE_GAUS_PTS]={sizeof(wgt1 )/sizeof(IssmPDouble),
		sizeof(wgt2 )/sizeof(IssmPDouble),
		sizeof(wgt3 )/sizeof(IssmPDouble),
		sizeof(wgt4 )/sizeof(IssmPDouble)};

	//	_pprintLine_("Gauss-Legendre recurrence coefficients ngaus=" << ngaus);
	*pxgaus =xNew<IssmPDouble>(ngaus);
	*pxwgt  =xNew<IssmPDouble>(ngaus);

	/*  check to see if Gauss points need to be calculated  */
	if (ngaus <= MAX_LINE_GAUS_PTS) {

		/*  copy the points from the static arrays (noting that the pointers
			 could be returned directly, but then the calling function would
			 have to know to not free them)  */

		for (i=0; i<ngaus; i++) {
			(*pxgaus)[i]=xip [ngaus-1][i];
			(*pxwgt )[i]=wgtp[ngaus-1][i];
		}
	}
	else {

		/*  calculate the Gauss points using recurrence relations  */
		alpha=xNew<IssmPDouble>(ngaus);
		beta =xNew<IssmPDouble>(ngaus);

		/*  calculate the Legendre recurrence coefficients  */
		alpha[0]=0.;
		beta [0]=2.;

		for (i=1; i<ngaus; i++) {
			alpha[i]=0.;
			beta [i]=1./(4.-1./(i*i));
		}

		/*  calculate the Gauss points  */
		GaussRecur(*pxgaus, *pxwgt, ngaus, alpha, beta );
		xDelete<IssmPDouble>(beta);
		xDelete<IssmPDouble>(alpha);
	}
}/*}}}*/
/*FUNCTION GaussLegendreTria{{{*/
void GaussLegendreTria( int* pngaus, IssmDouble** pl1, IssmDouble** pl2, IssmDouble** pl3, IssmDouble** pwgt, int iord ) {
	/*Gauss quadrature points for the triangle.

	  Higher-order points from D.A. Dunavant, "High Degree Efficient
	  Symmetrical Gaussian Quadrature Rules for the Triangle", IJNME,
	  Vol. 21, pp. 1129-1148 (1985), as transcribed for Probe rules3.*/

	/*Intermediaries*/
	int i,j,ipt,nigaus;
	IssmPDouble xi,eta;
	IssmPDouble *xgaus=NULL,*xwgt=NULL,*egaus,*ewgt;

	/*Hardcoded Gauss points declaration*/
	/*p= 1, npoint= 1{{{*/
	static IssmPDouble wgt1[]={
		1.732050807568877};
	static IssmPDouble l11[]={
		0.333333333333333};
	static IssmPDouble l21[]={
		0.333333333333333};
	static IssmPDouble l31[]={
		0.333333333333333};
	/*}}}*/
	/*p= 2, npoint= 3  {{{*/
	static IssmPDouble wgt2[]={
		0.577350269189625, 0.577350269189625, 0.577350269189625};
	static IssmPDouble l12[]={
		0.666666666666667, 0.166666666666667, 0.166666666666667};
	static IssmPDouble l22[]={
		0.166666666666667, 0.666666666666667, 0.166666666666667};
	static IssmPDouble l32[]={
		0.166666666666667, 0.166666666666667, 0.666666666666667};
	/*}}}*/
	/*p= 3, npoint= 4  {{{*/
	static IssmPDouble wgt3[]={
		-0.974278579257493, 0.902109795608790, 0.902109795608790, 
		0.902109795608790};
	static IssmPDouble l13[]={
		0.333333333333333, 0.600000000000000, 0.200000000000000, 
		0.200000000000000};
	static IssmPDouble l23[]={
		0.333333333333333, 0.200000000000000, 0.600000000000000, 
		0.200000000000000};
	static IssmPDouble l33[]={
		0.333333333333333, 0.200000000000000, 0.200000000000000, 
		0.600000000000000};
	/*}}}*/
	/*p= 4, npoint= 6  {{{*/
	static IssmPDouble wgt4[]={
		0.386908262797819, 0.386908262797819, 0.386908262797819, 
		0.190442006391807, 0.190442006391807, 0.190442006391807};
	static IssmPDouble l14[]={
		0.108103018168070, 0.445948490915965, 0.445948490915965, 
		0.816847572980459, 0.091576213509771, 0.091576213509771};
	static IssmPDouble l24[]={
		0.445948490915965, 0.108103018168070, 0.445948490915965, 
		0.091576213509771, 0.816847572980459, 0.091576213509771};
	static IssmPDouble l34[]={
		0.445948490915965, 0.445948490915965, 0.108103018168070, 
		0.091576213509771, 0.091576213509771, 0.816847572980459};
	/*}}}*/
	/*p= 5, npoint= 7  {{{*/
	static IssmPDouble wgt5[]={
		0.389711431702997, 0.229313399254729, 0.229313399254729, 
		0.229313399254729, 0.218133059367230, 0.218133059367230, 
		0.218133059367230};
	static IssmPDouble l15[]={
		0.333333333333333, 0.059715871789770, 0.470142064105115, 
		0.470142064105115, 0.797426985353087, 0.101286507323456, 
		0.101286507323456};
	static IssmPDouble l25[]={
		0.333333333333333, 0.470142064105115, 0.059715871789770, 
		0.470142064105115, 0.101286507323456, 0.797426985353087, 
		0.101286507323456};
	static IssmPDouble l35[]={
		0.333333333333333, 0.470142064105115, 0.470142064105115, 
		0.059715871789770, 0.101286507323456, 0.101286507323456, 
		0.797426985353087};
	/*}}}*/
	/*p= 6, npoint=12  {{{*/
	static IssmPDouble wgt6[]={
		0.202279763184836, 0.202279763184836, 0.202279763184836, 
		0.088065961139281, 0.088065961139281, 0.088065961139281, 
		0.143502272432755, 0.143502272432755, 0.143502272432755, 
		0.143502272432755, 0.143502272432755, 0.143502272432755};
	static IssmPDouble l16[]={
		0.501426509658179, 0.249286745170910, 0.249286745170910, 
		0.873821971016996, 0.063089014491502, 0.063089014491502, 
		0.053145049844817, 0.053145049844817, 0.310352451033784, 
		0.636502499121399, 0.310352451033784, 0.636502499121399};
	static IssmPDouble l26[]={
		0.249286745170910, 0.501426509658179, 0.249286745170910, 
		0.063089014491502, 0.873821971016996, 0.063089014491502, 
		0.310352451033784, 0.636502499121399, 0.053145049844817, 
		0.053145049844817, 0.636502499121399, 0.310352451033784};
	static IssmPDouble l36[]={
		0.249286745170910, 0.249286745170910, 0.501426509658179, 
		0.063089014491502, 0.063089014491502, 0.873821971016996, 
		0.636502499121399, 0.310352451033784, 0.636502499121399, 
		0.310352451033784, 0.053145049844817, 0.053145049844817};
	/*}}}*/
	/*p= 7, npoint=13  {{{*/
	static IssmPDouble wgt7[]={
		-0.259062916308362, 0.304174548458604, 0.304174548458604, 
		0.304174548458604, 0.092400122517855, 0.092400122517855, 
		0.092400122517855, 0.133564951824643, 0.133564951824643, 
		0.133564951824643, 0.133564951824643, 0.133564951824643, 
		0.133564951824643};
	static IssmPDouble l17[]={
		0.333333333333333, 0.479308067841920, 0.260345966079040, 
		0.260345966079040, 0.869739794195568, 0.065130102902216, 
		0.065130102902216, 0.048690315425316, 0.048690315425316, 
		0.312865496004874, 0.638444188569810, 0.312865496004874, 
		0.638444188569810};
	static IssmPDouble l27[]={
		0.333333333333333, 0.260345966079040, 0.479308067841920, 
		0.260345966079040, 0.065130102902216, 0.869739794195568, 
		0.065130102902216, 0.312865496004874, 0.638444188569810, 
		0.048690315425316, 0.048690315425316, 0.638444188569810, 
		0.312865496004874};
	static IssmPDouble l37[]={
		0.333333333333333, 0.260345966079040, 0.260345966079040, 
		0.479308067841920, 0.065130102902216, 0.065130102902216, 
		0.869739794195568, 0.638444188569810, 0.312865496004874, 
		0.638444188569810, 0.312865496004874, 0.048690315425316, 
		0.048690315425316};
	/*}}}*/
	/*p= 8, npoint=16  {{{*/
	static IssmPDouble wgt8[]={
		0.249961964823104, 0.164703541925695, 0.164703541925695, 
		0.164703541925695, 0.178777729989794, 0.178777729989794, 
		0.178777729989794, 0.056219767020733, 0.056219767020733, 
		0.056219767020733, 0.047164287656184, 0.047164287656184, 
		0.047164287656184, 0.047164287656184, 0.047164287656184, 
		0.047164287656184};
	static IssmPDouble l18[]={
		0.333333333333333, 0.081414823414554, 0.459292588292723, 
		0.459292588292723, 0.658861384496480, 0.170569307751760, 
		0.170569307751760, 0.898905543365938, 0.050547228317031, 
		0.050547228317031, 0.008394777409958, 0.008394777409958, 
		0.263112829634638, 0.728492392955404, 0.263112829634638, 
		0.728492392955404};
	static IssmPDouble l28[]={
		0.333333333333333, 0.459292588292723, 0.081414823414554, 
		0.459292588292723, 0.170569307751760, 0.658861384496480, 
		0.170569307751760, 0.050547228317031, 0.898905543365938, 
		0.050547228317031, 0.263112829634638, 0.728492392955404, 
		0.008394777409958, 0.008394777409958, 0.728492392955404, 
		0.263112829634638};
	static IssmPDouble l38[]={
		0.333333333333333, 0.459292588292723, 0.459292588292723, 
		0.081414823414554, 0.170569307751760, 0.170569307751760, 
		0.658861384496480, 0.050547228317031, 0.050547228317031, 
		0.898905543365938, 0.728492392955404, 0.263112829634638, 
		0.728492392955404, 0.263112829634638, 0.008394777409958, 
		0.008394777409958};
	/*}}}*/
	/*p= 9, npoint=19  {{{*/
	static IssmPDouble wgt9[]={
		0.168244134395468, 0.054273292833345, 0.054273292833345, 
		0.054273292833345, 0.134801255248419, 0.134801255248419, 
		0.134801255248419, 0.137953930529909, 0.137953930529909, 
		0.137953930529909, 0.044301833780383, 0.044301833780383, 
		0.044301833780383, 0.074969289332873, 0.074969289332873, 
		0.074969289332873, 0.074969289332873, 0.074969289332873, 
		0.074969289332873};
	static IssmPDouble l19[]={
		0.333333333333333, 0.020634961602525, 0.489682519198738, 
		0.489682519198738, 0.125820817014127, 0.437089591492937, 
		0.437089591492937, 0.623592928761935, 0.188203535619033, 
		0.188203535619033, 0.910540973211095, 0.044729513394453, 
		0.044729513394453, 0.036838412054736, 0.036838412054736, 
		0.221962989160766, 0.741198598784498, 0.221962989160766, 
		0.741198598784498};
	static IssmPDouble l29[]={
		0.333333333333333, 0.489682519198738, 0.020634961602525, 
		0.489682519198738, 0.437089591492937, 0.125820817014127, 
		0.437089591492937, 0.188203535619033, 0.623592928761935, 
		0.188203535619033, 0.044729513394453, 0.910540973211095, 
		0.044729513394453, 0.221962989160766, 0.741198598784498, 
		0.036838412054736, 0.036838412054736, 0.741198598784498, 
		0.221962989160766};
	static IssmPDouble l39[]={
		0.333333333333333, 0.489682519198738, 0.489682519198738, 
		0.020634961602525, 0.437089591492937, 0.437089591492937, 
		0.125820817014127, 0.188203535619033, 0.188203535619033, 
		0.623592928761935, 0.044729513394453, 0.044729513394453, 
		0.910540973211095, 0.741198598784498, 0.221962989160766, 
		0.741198598784498, 0.221962989160766, 0.036838412054736, 
		0.036838412054736};
	/*}}}*/
	/*p=10, npoint=25  {{{*/
	static IssmPDouble wgt10[]={
		0.157301373584232, 0.063611224790829, 0.063611224790829, 
		0.063611224790829, 0.078498377595183, 0.078498377595183, 
		0.078498377595183, 0.126020408629139, 0.126020408629139, 
		0.126020408629139, 0.126020408629139, 0.126020408629139, 
		0.126020408629139, 0.049064223302117, 0.049064223302117, 
		0.049064223302117, 0.049064223302117, 0.049064223302117, 
		0.049064223302117, 0.016318805873179, 0.016318805873179, 
		0.016318805873179, 0.016318805873179, 0.016318805873179, 
		0.016318805873179};
	static IssmPDouble l110[]={
		0.333333333333333, 0.028844733232685, 0.485577633383657, 
		0.485577633383657, 0.781036849029926, 0.109481575485037, 
		0.109481575485037, 0.141707219414880, 0.141707219414880, 
		0.307939838764121, 0.550352941820999, 0.307939838764121, 
		0.550352941820999, 0.025003534762686, 0.025003534762686, 
		0.246672560639903, 0.728323904597411, 0.246672560639903, 
		0.728323904597411, 0.009540815400299, 0.009540815400299, 
		0.066803251012200, 0.923655933587500, 0.066803251012200, 
		0.923655933587500};
	static IssmPDouble l210[]={
		0.333333333333333, 0.485577633383657, 0.028844733232685, 
		0.485577633383657, 0.109481575485037, 0.781036849029926, 
		0.109481575485037, 0.307939838764121, 0.550352941820999, 
		0.141707219414880, 0.141707219414880, 0.550352941820999, 
		0.307939838764121, 0.246672560639903, 0.728323904597411, 
		0.025003534762686, 0.025003534762686, 0.728323904597411, 
		0.246672560639903, 0.066803251012200, 0.923655933587500, 
		0.009540815400299, 0.009540815400299, 0.923655933587500, 
		0.066803251012200};
	static IssmPDouble l310[]={
		0.333333333333333, 0.485577633383657, 0.485577633383657, 
		0.028844733232685, 0.109481575485037, 0.109481575485037, 
		0.781036849029926, 0.550352941820999, 0.307939838764121, 
		0.550352941820999, 0.307939838764121, 0.141707219414880, 
		0.141707219414880, 0.728323904597411, 0.246672560639903, 
		0.728323904597411, 0.246672560639903, 0.025003534762686, 
		0.025003534762686, 0.923655933587500, 0.066803251012200, 
		0.923655933587500, 0.066803251012200, 0.009540815400299, 
		0.009540815400299};
	/*}}}*/
	/*p=11, npoint=27  {{{*/
	static IssmPDouble wgt11[]={
		0.001605622060698, 0.001605622060698, 0.001605622060698, 
		0.133626914252765, 0.133626914252765, 0.133626914252765, 
		0.102750410879760, 0.102750410879760, 0.102750410879760, 
		0.062673462600454, 0.062673462600454, 0.062673462600454, 
		0.023659348114362, 0.023659348114362, 0.023659348114362, 
		0.090650537039958, 0.090650537039958, 0.090650537039958, 
		0.090650537039958, 0.090650537039958, 0.090650537039958, 
		0.035866718600836, 0.035866718600836, 0.035866718600836, 
		0.035866718600836, 0.035866718600836, 0.035866718600836};
	static IssmPDouble l111[]={
		-0.069222096541517, 0.534611048270758, 0.534611048270758, 
		0.202061394068290, 0.398969302965855, 0.398969302965855, 
		0.593380199137435, 0.203309900431282, 0.203309900431282, 
		0.761298175434837, 0.119350912282581, 0.119350912282581, 
		0.935270103777448, 0.032364948111276, 0.032364948111276, 
		0.050178138310495, 0.050178138310495, 0.356620648261293, 
		0.593201213428213, 0.356620648261293, 0.593201213428213, 
		0.021022016536166, 0.021022016536166, 0.171488980304042, 
		0.807489003159792, 0.171488980304042, 0.807489003159792};
	static IssmPDouble l211[]={
		0.534611048270758,-0.069222096541517, 0.534611048270758, 
		0.398969302965855, 0.202061394068290, 0.398969302965855, 
		0.203309900431282, 0.593380199137435, 0.203309900431282, 
		0.119350912282581, 0.761298175434837, 0.119350912282581, 
		0.032364948111276, 0.935270103777448, 0.032364948111276, 
		0.356620648261293, 0.593201213428213, 0.050178138310495, 
		0.050178138310495, 0.593201213428213, 0.356620648261293, 
		0.171488980304042, 0.807489003159792, 0.021022016536166, 
		0.021022016536166, 0.807489003159792, 0.171488980304042};
	static IssmPDouble l311[]={
		0.534611048270758, 0.534611048270758,-0.069222096541517, 
		0.398969302965855, 0.398969302965855, 0.202061394068290, 
		0.203309900431282, 0.203309900431282, 0.593380199137435, 
		0.119350912282581, 0.119350912282581, 0.761298175434837, 
		0.032364948111276, 0.032364948111276, 0.935270103777448, 
		0.593201213428213, 0.356620648261293, 0.593201213428213, 
		0.356620648261293, 0.050178138310495, 0.050178138310495, 
		0.807489003159792, 0.171488980304042, 0.807489003159792, 
		0.171488980304042, 0.021022016536166, 0.021022016536166};
	/*}}}*/
	/*p=12, npoint=33  {{{*/
	static IssmPDouble wgt12[]={
		0.044567514407799, 0.044567514407799, 0.044567514407799, 
		0.075677707051848, 0.075677707051848, 0.075677707051848, 
		0.108873638018933, 0.108873638018933, 0.108873638018933, 
		0.060268635501892, 0.060268635501892, 0.060268635501892, 
		0.010680277434033, 0.010680277434033, 0.010680277434033, 
		0.069925589232074, 0.069925589232074, 0.069925589232074, 
		0.069925589232074, 0.069925589232074, 0.069925589232074, 
		0.038723067079683, 0.038723067079683, 0.038723067079683, 
		0.038723067079683, 0.038723067079683, 0.038723067079683, 
		0.029992592075802, 0.029992592075802, 0.029992592075802, 
		0.029992592075802, 0.029992592075802, 0.029992592075802};
	static IssmPDouble l112[]={
		0.023565220452390, 0.488217389773805, 0.488217389773805, 
		0.120551215411079, 0.439724392294460, 0.439724392294460, 
		0.457579229975768, 0.271210385012116, 0.271210385012116, 
		0.744847708916828, 0.127576145541586, 0.127576145541586, 
		0.957365299093579, 0.021317350453210, 0.021317350453210, 
		0.115343494534698, 0.115343494534698, 0.275713269685514, 
		0.608943235779788, 0.275713269685514, 0.608943235779788, 
		0.022838332222257, 0.022838332222257, 0.281325580989940, 
		0.695836086787803, 0.281325580989940, 0.695836086787803, 
		0.025734050548330, 0.025734050548330, 0.116251915907597, 
		0.858014033544073, 0.116251915907597, 0.858014033544073};
	static IssmPDouble l212[]={
		0.488217389773805, 0.023565220452390, 0.488217389773805, 
		0.439724392294460, 0.120551215411079, 0.439724392294460, 
		0.271210385012116, 0.457579229975768, 0.271210385012116, 
		0.127576145541586, 0.744847708916828, 0.127576145541586, 
		0.021317350453210, 0.957365299093579, 0.021317350453210, 
		0.275713269685514, 0.608943235779788, 0.115343494534698, 
		0.115343494534698, 0.608943235779788, 0.275713269685514, 
		0.281325580989940, 0.695836086787803, 0.022838332222257, 
		0.022838332222257, 0.695836086787803, 0.281325580989940, 
		0.116251915907597, 0.858014033544073, 0.025734050548330, 
		0.025734050548330, 0.858014033544073, 0.116251915907597};
	static IssmPDouble l312[]={
		0.488217389773805, 0.488217389773805, 0.023565220452390, 
		0.439724392294460, 0.439724392294460, 0.120551215411079, 
		0.271210385012116, 0.271210385012116, 0.457579229975768, 
		0.127576145541586, 0.127576145541586, 0.744847708916828, 
		0.021317350453210, 0.021317350453210, 0.957365299093579, 
		0.608943235779788, 0.275713269685514, 0.608943235779788, 
		0.275713269685514, 0.115343494534698, 0.115343494534698, 
		0.695836086787803, 0.281325580989940, 0.695836086787803, 
		0.281325580989940, 0.022838332222257, 0.022838332222257, 
		0.858014033544073, 0.116251915907597, 0.858014033544073, 
		0.116251915907597, 0.025734050548330, 0.025734050548330};
	/*}}}*/
	/*p=13, npoint=37  {{{*/
	static IssmPDouble wgt13[]={
		0.090968907790622, 0.019537784619314, 0.019537784619314, 
		0.019537784619314, 0.054427130356344, 0.054427130356344, 
		0.054427130356344, 0.081531965976677, 0.081531965976677, 
		0.081531965976677, 0.082036138309652, 0.082036138309652, 
		0.082036138309652, 0.053983743853694, 0.053983743853694, 
		0.053983743853694, 0.013814441407066, 0.013814441407066, 
		0.013814441407066, 0.063823305703923, 0.063823305703923, 
		0.063823305703923, 0.063823305703923, 0.063823305703923, 
		0.063823305703923, 0.030140218568265, 0.030140218568265, 
		0.030140218568265, 0.030140218568265, 0.030140218568265, 
		0.030140218568265, 0.026884523429480, 0.026884523429480, 
		0.026884523429480, 0.026884523429480, 0.026884523429480, 
		0.026884523429480};
	static IssmPDouble l113[]={
		0.333333333333333, 0.009903630120591, 0.495048184939705, 
		0.495048184939705, 0.062566729780852, 0.468716635109574, 
		0.468716635109574, 0.170957326397447, 0.414521336801277, 
		0.414521336801277, 0.541200855914337, 0.229399572042831, 
		0.229399572042831, 0.771151009607340, 0.114424495196330, 
		0.114424495196330, 0.950377217273082, 0.024811391363459, 
		0.024811391363459, 0.094853828379579, 0.094853828379579, 
		0.268794997058761, 0.636351174561660, 0.268794997058761, 
		0.636351174561660, 0.018100773278807, 0.018100773278807, 
		0.291730066734288, 0.690169159986905, 0.291730066734288, 
		0.690169159986905, 0.022233076674090, 0.022233076674090, 
		0.126357385491669, 0.851409537834241, 0.126357385491669, 
		0.851409537834241};
	static IssmPDouble l213[]={
		0.333333333333333, 0.495048184939705, 0.009903630120591, 
		0.495048184939705, 0.468716635109574, 0.062566729780852, 
		0.468716635109574, 0.414521336801277, 0.170957326397447, 
		0.414521336801277, 0.229399572042831, 0.541200855914337, 
		0.229399572042831, 0.114424495196330, 0.771151009607340, 
		0.114424495196330, 0.024811391363459, 0.950377217273082, 
		0.024811391363459, 0.268794997058761, 0.636351174561660, 
		0.094853828379579, 0.094853828379579, 0.636351174561660, 
		0.268794997058761, 0.291730066734288, 0.690169159986905, 
		0.018100773278807, 0.018100773278807, 0.690169159986905, 
		0.291730066734288, 0.126357385491669, 0.851409537834241, 
		0.022233076674090, 0.022233076674090, 0.851409537834241, 
		0.126357385491669};
	static IssmPDouble l313[]={
		0.333333333333333, 0.495048184939705, 0.495048184939705, 
		0.009903630120591, 0.468716635109574, 0.468716635109574, 
		0.062566729780852, 0.414521336801277, 0.414521336801277, 
		0.170957326397447, 0.229399572042831, 0.229399572042831, 
		0.541200855914337, 0.114424495196330, 0.114424495196330, 
		0.771151009607340, 0.024811391363459, 0.024811391363459, 
		0.950377217273082, 0.636351174561660, 0.268794997058761, 
		0.636351174561660, 0.268794997058761, 0.094853828379579, 
		0.094853828379579, 0.690169159986905, 0.291730066734288, 
		0.690169159986905, 0.291730066734288, 0.018100773278807, 
		0.018100773278807, 0.851409537834241, 0.126357385491669, 
		0.851409537834241, 0.126357385491669, 0.022233076674090, 
		0.022233076674090};
	/*}}}*/
	/*p=14, npoint=42{{{*/
	static IssmPDouble wgt14[]={
		0.037903474783419, 0.037903474783419, 0.037903474783419, 
		0.056791094234956, 0.056791094234956, 0.056791094234956, 
		0.089675379523011, 0.089675379523011, 0.089675379523011, 
		0.073027745871103, 0.073027745871103, 0.073027745871103, 
		0.024999901169244, 0.024999901169244, 0.024999901169244, 
		0.008527585185524, 0.008527585185524, 0.008527585185524, 
		0.042722337771116, 0.042722337771116, 0.042722337771116, 
		0.042722337771116, 0.042722337771116, 0.042722337771116, 
		0.066807816407881, 0.066807816407881, 0.066807816407881, 
		0.066807816407881, 0.066807816407881, 0.066807816407881, 
		0.025004419126360, 0.025004419126360, 0.025004419126360, 
		0.025004419126360, 0.025004419126360, 0.025004419126360, 
		0.008677970905831, 0.008677970905831, 0.008677970905831, 
		0.008677970905831, 0.008677970905831, 0.008677970905831};
	static IssmPDouble l114[]={
		0.022072179275643, 0.488963910362179, 0.488963910362179, 
		0.164710561319092, 0.417644719340454, 0.417644719340454, 
		0.453044943382323, 0.273477528308839, 0.273477528308839, 
		0.645588935174913, 0.177205532412543, 0.177205532412543, 
		0.876400233818255, 0.061799883090873, 0.061799883090873, 
		0.961218077502598, 0.019390961248701, 0.019390961248701, 
		0.057124757403648, 0.057124757403648, 0.172266687821356, 
		0.770608554774996, 0.172266687821356, 0.770608554774996, 
		0.092916249356972, 0.092916249356972, 0.336861459796345, 
		0.570222290846683, 0.336861459796345, 0.570222290846683, 
		0.014646950055654, 0.014646950055654, 0.298372882136258, 
		0.686980167808088, 0.298372882136258, 0.686980167808088, 
		0.001268330932872, 0.001268330932872, 0.118974497696957, 
		0.879757171370171, 0.118974497696957, 0.879757171370171};
	static IssmPDouble l214[]={
		0.488963910362179, 0.022072179275643, 0.488963910362179, 
		0.417644719340454, 0.164710561319092, 0.417644719340454, 
		0.273477528308839, 0.453044943382323, 0.273477528308839, 
		0.177205532412543, 0.645588935174913, 0.177205532412543, 
		0.061799883090873, 0.876400233818255, 0.061799883090873, 
		0.019390961248701, 0.961218077502598, 0.019390961248701, 
		0.172266687821356, 0.770608554774996, 0.057124757403648, 
		0.057124757403648, 0.770608554774996, 0.172266687821356, 
		0.336861459796345, 0.570222290846683, 0.092916249356972, 
		0.092916249356972, 0.570222290846683, 0.336861459796345, 
		0.298372882136258, 0.686980167808088, 0.014646950055654, 
		0.014646950055654, 0.686980167808088, 0.298372882136258, 
		0.118974497696957, 0.879757171370171, 0.001268330932872, 
		0.001268330932872, 0.879757171370171, 0.118974497696957};
	static IssmPDouble l314[]={
		0.488963910362179, 0.488963910362179, 0.022072179275643, 
		0.417644719340454, 0.417644719340454, 0.164710561319092, 
		0.273477528308839, 0.273477528308839, 0.453044943382323, 
		0.177205532412543, 0.177205532412543, 0.645588935174913, 
		0.061799883090873, 0.061799883090873, 0.876400233818255, 
		0.019390961248701, 0.019390961248701, 0.961218077502598, 
		0.770608554774996, 0.172266687821356, 0.770608554774996, 
		0.172266687821356, 0.057124757403648, 0.057124757403648, 
		0.570222290846683, 0.336861459796345, 0.570222290846683, 
		0.336861459796345, 0.092916249356972, 0.092916249356972, 
		0.686980167808088, 0.298372882136258, 0.686980167808088, 
		0.298372882136258, 0.014646950055654, 0.014646950055654, 
		0.879757171370171, 0.118974497696957, 0.879757171370171, 
		0.118974497696957, 0.001268330932872, 0.001268330932872};
	/*}}}*/
	/*p=15, npoint=48{{{*/
	static IssmPDouble wgt15[]={
		0.003320126005206, 0.003320126005206, 0.003320126005206, 
		0.076641563419124, 0.076641563419124, 0.076641563419124, 
		0.088657703045151, 0.088657703045151, 0.088657703045151, 
		0.041028362044303, 0.041028362044303, 0.041028362044303, 
		0.023018566716310, 0.023018566716310, 0.023018566716310, 
		0.008225364846296, 0.008225364846296, 0.008225364846296, 
		0.066770684377964, 0.066770684377964, 0.066770684377964, 
		0.066770684377964, 0.066770684377964, 0.066770684377964, 
		0.047139173172681, 0.047139173172681, 0.047139173172681, 
		0.047139173172681, 0.047139173172681, 0.047139173172681, 
		0.003779468865339, 0.003779468865339, 0.003779468865339, 
		0.003779468865339, 0.003779468865339, 0.003779468865339, 
		0.037248306609289, 0.037248306609289, 0.037248306609289, 
		0.037248306609289, 0.037248306609289, 0.037248306609289, 
		0.013291658531346, 0.013291658531346, 0.013291658531346, 
		0.013291658531346, 0.013291658531346, 0.013291658531346};
	static IssmPDouble l115[]={
		-0.013945833716486, 0.506972916858243, 0.506972916858243, 
		0.137187291433955, 0.431406354283023, 0.431406354283023, 
		0.444612710305711, 0.277693644847144, 0.277693644847144, 
		0.747070217917492, 0.126464891041254, 0.126464891041254, 
		0.858383228050628, 0.070808385974686, 0.070808385974686, 
		0.962069659517853, 0.018965170241073, 0.018965170241073, 
		0.133734161966621, 0.133734161966621, 0.261311371140087, 
		0.604954466893291, 0.261311371140087, 0.604954466893291, 
		0.036366677396917, 0.036366677396917, 0.388046767090269, 
		0.575586555512814, 0.388046767090269, 0.575586555512814, 
		-0.010174883126571,-0.010174883126571, 0.285712220049916, 
		0.724462663076655, 0.285712220049916, 0.724462663076655, 
		0.036843869875878, 0.036843869875878, 0.215599664072284, 
		0.747556466051838, 0.215599664072284, 0.747556466051838, 
		0.012459809331199, 0.012459809331199, 0.103575616576386, 
		0.883964574092416, 0.103575616576386, 0.883964574092416};
	static IssmPDouble l215[]={
		0.506972916858243,-0.013945833716486, 0.506972916858243, 
		0.431406354283023, 0.137187291433955, 0.431406354283023, 
		0.277693644847144, 0.444612710305711, 0.277693644847144, 
		0.126464891041254, 0.747070217917492, 0.126464891041254, 
		0.070808385974686, 0.858383228050628, 0.070808385974686, 
		0.018965170241073, 0.962069659517853, 0.018965170241073, 
		0.261311371140087, 0.604954466893291, 0.133734161966621, 
		0.133734161966621, 0.604954466893291, 0.261311371140087, 
		0.388046767090269, 0.575586555512814, 0.036366677396917, 
		0.036366677396917, 0.575586555512814, 0.388046767090269, 
		0.285712220049916, 0.724462663076655,-0.010174883126571, 
		-0.010174883126571, 0.724462663076655, 0.285712220049916, 
		0.215599664072284, 0.747556466051838, 0.036843869875878, 
		0.036843869875878, 0.747556466051838, 0.215599664072284, 
		0.103575616576386, 0.883964574092416, 0.012459809331199, 
		0.012459809331199, 0.883964574092416, 0.103575616576386};
	static IssmPDouble l315[]={
		0.506972916858243, 0.506972916858243,-0.013945833716486, 
		0.431406354283023, 0.431406354283023, 0.137187291433955, 
		0.277693644847144, 0.277693644847144, 0.444612710305711, 
		0.126464891041254, 0.126464891041254, 0.747070217917492, 
		0.070808385974686, 0.070808385974686, 0.858383228050628, 
		0.018965170241073, 0.018965170241073, 0.962069659517853, 
		0.604954466893291, 0.261311371140087, 0.604954466893291, 
		0.261311371140087, 0.133734161966621, 0.133734161966621, 
		0.575586555512814, 0.388046767090269, 0.575586555512814, 
		0.388046767090269, 0.036366677396917, 0.036366677396917, 
		0.724462663076655, 0.285712220049916, 0.724462663076655, 
		0.285712220049916,-0.010174883126571,-0.010174883126571, 
		0.747556466051838, 0.215599664072284, 0.747556466051838, 
		0.215599664072284, 0.036843869875878, 0.036843869875878, 
		0.883964574092416, 0.103575616576386, 0.883964574092416, 
		0.103575616576386, 0.012459809331199, 0.012459809331199};
	/*}}}*/
	/*p=16, npoint=52  {{{*/
	static IssmPDouble wgt16[]={
		0.081191089584902, 0.011095307165226, 0.011095307165226, 
		0.011095307165226, 0.072244353151393, 0.072244353151393, 
		0.072244353151393, 0.046577417012049, 0.046577417012049, 
		0.046577417012049, 0.072975670074230, 0.072975670074230, 
		0.072975670074230, 0.051961986412307, 0.051961986412307, 
		0.051961986412307, 0.024595292810646, 0.024595292810646, 
		0.024595292810646, 0.006205006808607, 0.006205006808607, 
		0.006205006808607, 0.056764756525753, 0.056764756525753, 
		0.056764756525753, 0.056764756525753, 0.056764756525753, 
		0.056764756525753, 0.026497443692048, 0.026497443692048, 
		0.026497443692048, 0.026497443692048, 0.026497443692048, 
		0.026497443692048, 0.004133096181263, 0.004133096181263, 
		0.004133096181263, 0.004133096181263, 0.004133096181263, 
		0.004133096181263, 0.033055830705140, 0.033055830705140, 
		0.033055830705140, 0.033055830705140, 0.033055830705140, 
		0.033055830705140, 0.011864642509229, 0.011864642509229, 
		0.011864642509229, 0.011864642509229, 0.011864642509229, 
		0.011864642509229};
	static IssmPDouble l116[]={
		0.333333333333333, 0.005238916103123, 0.497380541948438, 
		0.497380541948438, 0.173061122901295, 0.413469438549352, 
		0.413469438549352, 0.059082801866017, 0.470458599066991, 
		0.470458599066991, 0.518892500060958, 0.240553749969521, 
		0.240553749969521, 0.704068411554854, 0.147965794222573, 
		0.147965794222573, 0.849069624685052, 0.075465187657474, 
		0.075465187657474, 0.966807194753950, 0.016596402623025, 
		0.016596402623025, 0.103575692245252, 0.103575692245252, 
		0.296555596579887, 0.599868711174861, 0.296555596579887, 
		0.599868711174861, 0.020083411655416, 0.020083411655416, 
		0.337723063403079, 0.642193524941505, 0.337723063403079, 
		0.642193524941505,-0.004341002614139,-0.004341002614139, 
		0.204748281642812, 0.799592720971327, 0.204748281642812, 
		0.799592720971327, 0.041941786468010, 0.041941786468010, 
		0.189358492130623, 0.768699721401368, 0.189358492130623, 
		0.768699721401368, 0.014317320230681, 0.014317320230681, 
		0.085283615682657, 0.900399064086661, 0.085283615682657, 
		0.900399064086661};
	static IssmPDouble l216[]={
		0.333333333333333, 0.497380541948438, 0.005238916103123, 
		0.497380541948438, 0.413469438549352, 0.173061122901295, 
		0.413469438549352, 0.470458599066991, 0.059082801866017, 
		0.470458599066991, 0.240553749969521, 0.518892500060958, 
		0.240553749969521, 0.147965794222573, 0.704068411554854, 
		0.147965794222573, 0.075465187657474, 0.849069624685052, 
		0.075465187657474, 0.016596402623025, 0.966807194753950, 
		0.016596402623025, 0.296555596579887, 0.599868711174861, 
		0.103575692245252, 0.103575692245252, 0.599868711174861, 
		0.296555596579887, 0.337723063403079, 0.642193524941505, 
		0.020083411655416, 0.020083411655416, 0.642193524941505, 
		0.337723063403079, 0.204748281642812, 0.799592720971327, 
		-0.004341002614139,-0.004341002614139, 0.799592720971327, 
		0.204748281642812, 0.189358492130623, 0.768699721401368, 
		0.041941786468010, 0.041941786468010, 0.768699721401368, 
		0.189358492130623, 0.085283615682657, 0.900399064086661, 
		0.014317320230681, 0.014317320230681, 0.900399064086661, 
		0.085283615682657};
	static IssmPDouble l316[]={
		0.333333333333333, 0.497380541948438, 0.497380541948438, 
		0.005238916103123, 0.413469438549352, 0.413469438549352, 
		0.173061122901295, 0.470458599066991, 0.470458599066991, 
		0.059082801866017, 0.240553749969521, 0.240553749969521, 
		0.518892500060958, 0.147965794222573, 0.147965794222573, 
		0.704068411554854, 0.075465187657474, 0.075465187657474, 
		0.849069624685052, 0.016596402623025, 0.016596402623025, 
		0.966807194753950, 0.599868711174861, 0.296555596579887, 
		0.599868711174861, 0.296555596579887, 0.103575692245252, 
		0.103575692245252, 0.642193524941505, 0.337723063403079, 
		0.642193524941505, 0.337723063403079, 0.020083411655416, 
		0.020083411655416, 0.799592720971327, 0.204748281642812, 
		0.799592720971327, 0.204748281642812,-0.004341002614139, 
		-0.004341002614139, 0.768699721401368, 0.189358492130623, 
		0.768699721401368, 0.189358492130623, 0.041941786468010, 
		0.041941786468010, 0.900399064086661, 0.085283615682657, 
		0.900399064086661, 0.085283615682657, 0.014317320230681, 
		0.014317320230681};
	/*}}}*/
	/*p=17, npoint=61{{{*/
	static IssmPDouble wgt17[]={
		0.057914928034477, 0.008822054327014, 0.008822054327014, 
		0.008822054327014, 0.025410682752829, 0.025410682752829, 
		0.025410682752829, 0.042176958517489, 0.042176958517489, 
		0.042176958517489, 0.053879858604088, 0.053879858604088, 
		0.053879858604088, 0.054138904728481, 0.054138904728481, 
		0.054138904728481, 0.042981974139367, 0.042981974139367, 
		0.042981974139367, 0.024345832713105, 0.024345832713105, 
		0.024345832713105, 0.005533341446715, 0.005533341446715, 
		0.005533341446715, 0.014063655552443, 0.014063655552443, 
		0.014063655552443, 0.014063655552443, 0.014063655552443, 
		0.014063655552443, 0.046428907569036, 0.046428907569036, 
		0.046428907569036, 0.046428907569036, 0.046428907569036, 
		0.046428907569036, 0.031973646148520, 0.031973646148520, 
		0.031973646148520, 0.031973646148520, 0.031973646148520, 
		0.031973646148520, 0.014682366990538, 0.014682366990538, 
		0.014682366990538, 0.014682366990538, 0.014682366990538, 
		0.014682366990538, 0.031684053418215, 0.031684053418215, 
		0.031684053418215, 0.031684053418215, 0.031684053418215, 
		0.031684053418215, 0.011545213295771, 0.011545213295771, 
		0.011545213295771, 0.011545213295771, 0.011545213295771, 
		0.011545213295771};
	static IssmPDouble l117[]={
		0.333333333333333, 0.005658918886452, 0.497170540556774, 
		0.497170540556774, 0.035647354750751, 0.482176322624625, 
		0.482176322624625, 0.099520061958437, 0.450239969020782, 
		0.450239969020782, 0.199467521245206, 0.400266239377397, 
		0.400266239377397, 0.495717464058095, 0.252141267970953, 
		0.252141267970953, 0.675905990683077, 0.162047004658461, 
		0.162047004658461, 0.848248235478508, 0.075875882260746, 
		0.075875882260746, 0.968690546064356, 0.015654726967822, 
		0.015654726967822, 0.010186928826919, 0.010186928826919, 
		0.334319867363658, 0.655493203809423, 0.334319867363658, 
		0.655493203809423, 0.135440871671036, 0.135440871671036, 
		0.292221537796944, 0.572337590532020, 0.292221537796944, 
		0.572337590532020, 0.054423924290583, 0.054423924290583, 
		0.319574885423190, 0.626001190286228, 0.319574885423190, 
		0.626001190286228, 0.012868560833637, 0.012868560833637, 
		0.190704224192292, 0.796427214974071, 0.190704224192292, 
		0.796427214974071, 0.067165782413524, 0.067165782413524, 
		0.180483211648746, 0.752351005937729, 0.180483211648746, 
		0.752351005937729, 0.014663182224828, 0.014663182224828, 
		0.080711313679564, 0.904625504095608, 0.080711313679564, 
		0.904625504095608};
	static IssmPDouble l217[]={
		0.333333333333333, 0.497170540556774, 0.005658918886452, 
		0.497170540556774, 0.482176322624625, 0.035647354750751, 
		0.482176322624625, 0.450239969020782, 0.099520061958437, 
		0.450239969020782, 0.400266239377397, 0.199467521245206, 
		0.400266239377397, 0.252141267970953, 0.495717464058095, 
		0.252141267970953, 0.162047004658461, 0.675905990683077, 
		0.162047004658461, 0.075875882260746, 0.848248235478508, 
		0.075875882260746, 0.015654726967822, 0.968690546064356, 
		0.015654726967822, 0.334319867363658, 0.655493203809423, 
		0.010186928826919, 0.010186928826919, 0.655493203809423, 
		0.334319867363658, 0.292221537796944, 0.572337590532020, 
		0.135440871671036, 0.135440871671036, 0.572337590532020, 
		0.292221537796944, 0.319574885423190, 0.626001190286228, 
		0.054423924290583, 0.054423924290583, 0.626001190286228, 
		0.319574885423190, 0.190704224192292, 0.796427214974071, 
		0.012868560833637, 0.012868560833637, 0.796427214974071, 
		0.190704224192292, 0.180483211648746, 0.752351005937729, 
		0.067165782413524, 0.067165782413524, 0.752351005937729, 
		0.180483211648746, 0.080711313679564, 0.904625504095608, 
		0.014663182224828, 0.014663182224828, 0.904625504095608, 
		0.080711313679564};
	static IssmPDouble l317[]={
		0.333333333333333, 0.497170540556774, 0.497170540556774, 
		0.005658918886452, 0.482176322624625, 0.482176322624625, 
		0.035647354750751, 0.450239969020782, 0.450239969020782, 
		0.099520061958437, 0.400266239377397, 0.400266239377397, 
		0.199467521245206, 0.252141267970953, 0.252141267970953, 
		0.495717464058095, 0.162047004658461, 0.162047004658461, 
		0.675905990683077, 0.075875882260746, 0.075875882260746, 
		0.848248235478508, 0.015654726967822, 0.015654726967822, 
		0.968690546064356, 0.655493203809423, 0.334319867363658, 
		0.655493203809423, 0.334319867363658, 0.010186928826919, 
		0.010186928826919, 0.572337590532020, 0.292221537796944, 
		0.572337590532020, 0.292221537796944, 0.135440871671036, 
		0.135440871671036, 0.626001190286228, 0.319574885423190, 
		0.626001190286228, 0.319574885423190, 0.054423924290583, 
		0.054423924290583, 0.796427214974071, 0.190704224192292, 
		0.796427214974071, 0.190704224192292, 0.012868560833637, 
		0.012868560833637, 0.752351005937729, 0.180483211648746, 
		0.752351005937729, 0.180483211648746, 0.067165782413524, 
		0.067165782413524, 0.904625504095608, 0.080711313679564, 
		0.904625504095608, 0.080711313679564, 0.014663182224828, 
		0.014663182224828};
	/*}}}*/
	/*p=18, npoint=70  {{{*/

	static IssmPDouble wgt18[]={
		0.053364381350150, 0.015713921277179, 0.015713921277179, 
		0.015713921277179, 0.032495554156279, 0.032495554156279, 
		0.032495554156279, 0.033672969465771, 0.033672969465771, 
		0.033672969465771, 0.048071249104579, 0.048071249104579, 
		0.048071249104579, 0.055869421169115, 0.055869421169115, 
		0.055869421169115, 0.043429498443148, 0.043429498443148, 
		0.043429498443148, 0.026451755176745, 0.026451755176745, 
		0.026451755176745, 0.011767418126433, 0.011767418126433, 
		0.011767418126433,-0.003850519950463,-0.003850519950463, 
		-0.003850519950463, 0.010967196889496, 0.010967196889496, 
		0.010967196889496, 0.010967196889496, 0.010967196889496, 
		0.010967196889496, 0.047211440790349, 0.047211440790349, 
		0.047211440790349, 0.047211440790349, 0.047211440790349, 
		0.047211440790349, 0.030617090859378, 0.030617090859378, 
		0.030617090859378, 0.030617090859378, 0.030617090859378, 
		0.030617090859378, 0.031834201210069, 0.031834201210069, 
		0.031834201210069, 0.031834201210069, 0.031834201210069, 
		0.031834201210069, 0.014037809005559, 0.014037809005559, 
		0.014037809005559, 0.014037809005559, 0.014037809005559, 
		0.014037809005559, 0.013222699422034, 0.013222699422034, 
		0.013222699422034, 0.013222699422034, 0.013222699422034, 
		0.013222699422034, 0.000079999375178, 0.000079999375178, 
		0.000079999375178, 0.000079999375178, 0.000079999375178, 
		0.000079999375178};
	static IssmPDouble l118[]={
		0.333333333333333, 0.013310382738157, 0.493344808630921, 
		0.493344808630921, 0.061578811516086, 0.469210594241957, 
		0.469210594241957, 0.127437208225989, 0.436281395887006, 
		0.436281395887006, 0.210307658653168, 0.394846170673416, 
		0.394846170673416, 0.500410862393686, 0.249794568803157, 
		0.249794568803157, 0.677135612512315, 0.161432193743843, 
		0.161432193743843, 0.846803545029257, 0.076598227485371, 
		0.076598227485371, 0.951495121293100, 0.024252439353450, 
		0.024252439353450, 0.913707265566071, 0.043146367216965, 
		0.043146367216965, 0.008430536202420, 0.008430536202420, 
		0.358911494940944, 0.632657968856636, 0.358911494940944, 
		0.632657968856636, 0.131186551737188, 0.131186551737188, 
		0.294402476751957, 0.574410971510855, 0.294402476751957, 
		0.574410971510855, 0.050203151565675, 0.050203151565675, 
		0.325017801641814, 0.624779046792512, 0.325017801641814, 
		0.624779046792512, 0.066329263810916, 0.066329263810916, 
		0.184737559666046, 0.748933176523037, 0.184737559666046, 
		0.748933176523037, 0.011996194566236, 0.011996194566236, 
		0.218796800013321, 0.769207005420443, 0.218796800013321, 
		0.769207005420443, 0.014858100590125, 0.014858100590125, 
		0.101179597136408, 0.883962302273467, 0.101179597136408, 
		0.883962302273467,-0.035222015287949,-0.035222015287949, 
		0.020874755282586, 1.014347260005363, 0.020874755282586, 
		1.014347260005363};
	static IssmPDouble l218[]={
		0.333333333333333, 0.493344808630921, 0.013310382738157, 
		0.493344808630921, 0.469210594241957, 0.061578811516086, 
		0.469210594241957, 0.436281395887006, 0.127437208225989, 
		0.436281395887006, 0.394846170673416, 0.210307658653168, 
		0.394846170673416, 0.249794568803157, 0.500410862393686, 
		0.249794568803157, 0.161432193743843, 0.677135612512315, 
		0.161432193743843, 0.076598227485371, 0.846803545029257, 
		0.076598227485371, 0.024252439353450, 0.951495121293100, 
		0.024252439353450, 0.043146367216965, 0.913707265566071, 
		0.043146367216965, 0.358911494940944, 0.632657968856636, 
		0.008430536202420, 0.008430536202420, 0.632657968856636, 
		0.358911494940944, 0.294402476751957, 0.574410971510855, 
		0.131186551737188, 0.131186551737188, 0.574410971510855, 
		0.294402476751957, 0.325017801641814, 0.624779046792512, 
		0.050203151565675, 0.050203151565675, 0.624779046792512, 
		0.325017801641814, 0.184737559666046, 0.748933176523037, 
		0.066329263810916, 0.066329263810916, 0.748933176523037, 
		0.184737559666046, 0.218796800013321, 0.769207005420443, 
		0.011996194566236, 0.011996194566236, 0.769207005420443, 
		0.218796800013321, 0.101179597136408, 0.883962302273467, 
		0.014858100590125, 0.014858100590125, 0.883962302273467, 
		0.101179597136408, 0.020874755282586, 1.014347260005363, 
		-0.035222015287949,-0.035222015287949, 1.014347260005363, 
		0.020874755282586};
	static IssmPDouble l318[]={
		0.333333333333333, 0.493344808630921, 0.493344808630921, 
		0.013310382738157, 0.469210594241957, 0.469210594241957, 
		0.061578811516086, 0.436281395887006, 0.436281395887006, 
		0.127437208225989, 0.394846170673416, 0.394846170673416, 
		0.210307658653168, 0.249794568803157, 0.249794568803157, 
		0.500410862393686, 0.161432193743843, 0.161432193743843, 
		0.677135612512315, 0.076598227485371, 0.076598227485371, 
		0.846803545029257, 0.024252439353450, 0.024252439353450, 
		0.951495121293100, 0.043146367216965, 0.043146367216965, 
		0.913707265566071, 0.632657968856636, 0.358911494940944, 
		0.632657968856636, 0.358911494940944, 0.008430536202420, 
		0.008430536202420, 0.574410971510855, 0.294402476751957, 
		0.574410971510855, 0.294402476751957, 0.131186551737188, 
		0.131186551737188, 0.624779046792512, 0.325017801641814, 
		0.624779046792512, 0.325017801641814, 0.050203151565675, 
		0.050203151565675, 0.748933176523037, 0.184737559666046, 
		0.748933176523037, 0.184737559666046, 0.066329263810916, 
		0.066329263810916, 0.769207005420443, 0.218796800013321, 
		0.769207005420443, 0.218796800013321, 0.011996194566236, 
		0.011996194566236, 0.883962302273467, 0.101179597136408, 
		0.883962302273467, 0.101179597136408, 0.014858100590125, 
		0.014858100590125, 1.014347260005363, 0.020874755282586, 
		1.014347260005363, 0.020874755282586,-0.035222015287949, 
		-0.035222015287949};
	/*}}}*/
	/*p=19, npoint=73  {{{*/

	static IssmPDouble wgt19[]={
		0.056995437856306, 0.017893352515055, 0.017893352515055, 
		0.017893352515055, 0.038775849701151, 0.038775849701151, 
		0.038775849701151, 0.052422467754193, 0.052422467754193, 
		0.052422467754193, 0.052811905405354, 0.052811905405354, 
		0.052811905405354, 0.041844983939388, 0.041844983939388, 
		0.041844983939388, 0.027800807314648, 0.027800807314648, 
		0.027800807314648, 0.014002903771278, 0.014002903771278, 
		0.014002903771278, 0.003601560678933, 0.003601560678933, 
		0.003601560678933, 0.006728804180578, 0.006728804180578, 
		0.006728804180578, 0.006728804180578, 0.006728804180578, 
		0.006728804180578, 0.044295745540949, 0.044295745540949, 
		0.044295745540949, 0.044295745540949, 0.044295745540949, 
		0.044295745540949, 0.015382176206141, 0.015382176206141, 
		0.015382176206141, 0.015382176206141, 0.015382176206141, 
		0.015382176206141, 0.027928534240338, 0.027928534240338, 
		0.027928534240338, 0.027928534240338, 0.027928534240338, 
		0.027928534240338, 0.004316169837400, 0.004316169837400, 
		0.004316169837400, 0.004316169837400, 0.004316169837400, 
		0.004316169837400, 0.031597525960379, 0.031597525960379, 
		0.031597525960379, 0.031597525960379, 0.031597525960379, 
		0.031597525960379, 0.017768353603780, 0.017768353603780, 
		0.017768353603780, 0.017768353603780, 0.017768353603780, 
		0.017768353603780, 0.006581669842530, 0.006581669842530, 
		0.006581669842530, 0.006581669842530, 0.006581669842530, 
		0.006581669842530};
	static IssmPDouble l119[]={
		0.333333333333333, 0.020780025853987, 0.489609987073006, 
		0.489609987073006, 0.090926214604215, 0.454536892697893, 
		0.454536892697893, 0.197166638701138, 0.401416680649431, 
		0.401416680649431, 0.488896691193805, 0.255551654403098, 
		0.255551654403098, 0.645844115695741, 0.177077942152130, 
		0.177077942152130, 0.779877893544096, 0.110061053227952, 
		0.110061053227952, 0.888942751496321, 0.055528624251840, 
		0.055528624251840, 0.974756272445543, 0.012621863777229, 
		0.012621863777229, 0.003611417848412, 0.003611417848412, 
		0.395754787356943, 0.600633794794645, 0.395754787356943, 
		0.600633794794645, 0.134466754530780, 0.134466754530780, 
		0.307929983880436, 0.557603261588784, 0.307929983880436, 
		0.557603261588784, 0.014446025776115, 0.014446025776115, 
		0.264566948406520, 0.720987025817365, 0.264566948406520, 
		0.720987025817365, 0.046933578838178, 0.046933578838178, 
		0.358539352205951, 0.594527068955871, 0.358539352205951, 
		0.594527068955871, 0.002861120350567, 0.002861120350567, 
		0.157807405968595, 0.839331473680839, 0.157807405968595, 
		0.839331473680839, 0.223861424097916, 0.223861424097916, 
		0.075050596975911, 0.701087978926173, 0.075050596975911, 
		0.701087978926173, 0.034647074816760, 0.034647074816760, 
		0.142421601113383, 0.822931324069857, 0.142421601113383, 
		0.822931324069857, 0.010161119296278, 0.010161119296278, 
		0.065494628082938, 0.924344252620784, 0.065494628082938, 
		0.924344252620784};
	static IssmPDouble l219[]={
		0.333333333333333, 0.489609987073006, 0.020780025853987, 
		0.489609987073006, 0.454536892697893, 0.090926214604215, 
		0.454536892697893, 0.401416680649431, 0.197166638701138, 
		0.401416680649431, 0.255551654403098, 0.488896691193805, 
		0.255551654403098, 0.177077942152130, 0.645844115695741, 
		0.177077942152130, 0.110061053227952, 0.779877893544096, 
		0.110061053227952, 0.055528624251840, 0.888942751496321, 
		0.055528624251840, 0.012621863777229, 0.974756272445543, 
		0.012621863777229, 0.395754787356943, 0.600633794794645, 
		0.003611417848412, 0.003611417848412, 0.600633794794645, 
		0.395754787356943, 0.307929983880436, 0.557603261588784, 
		0.134466754530780, 0.134466754530780, 0.557603261588784, 
		0.307929983880436, 0.264566948406520, 0.720987025817365, 
		0.014446025776115, 0.014446025776115, 0.720987025817365, 
		0.264566948406520, 0.358539352205951, 0.594527068955871, 
		0.046933578838178, 0.046933578838178, 0.594527068955871, 
		0.358539352205951, 0.157807405968595, 0.839331473680839, 
		0.002861120350567, 0.002861120350567, 0.839331473680839, 
		0.157807405968595, 0.075050596975911, 0.701087978926173, 
		0.223861424097916, 0.223861424097916, 0.701087978926173, 
		0.075050596975911, 0.142421601113383, 0.822931324069857, 
		0.034647074816760, 0.034647074816760, 0.822931324069857, 
		0.142421601113383, 0.065494628082938, 0.924344252620784, 
		0.010161119296278, 0.010161119296278, 0.924344252620784, 
		0.065494628082938};
	static IssmPDouble l319[]={
		0.333333333333333, 0.489609987073006, 0.489609987073006, 
		0.020780025853987, 0.454536892697893, 0.454536892697893, 
		0.090926214604215, 0.401416680649431, 0.401416680649431, 
		0.197166638701138, 0.255551654403098, 0.255551654403098, 
		0.488896691193805, 0.177077942152130, 0.177077942152130, 
		0.645844115695741, 0.110061053227952, 0.110061053227952, 
		0.779877893544096, 0.055528624251840, 0.055528624251840, 
		0.888942751496321, 0.012621863777229, 0.012621863777229, 
		0.974756272445543, 0.600633794794645, 0.395754787356943, 
		0.600633794794645, 0.395754787356943, 0.003611417848412, 
		0.003611417848412, 0.557603261588784, 0.307929983880436, 
		0.557603261588784, 0.307929983880436, 0.134466754530780, 
		0.134466754530780, 0.720987025817365, 0.264566948406520, 
		0.720987025817365, 0.264566948406520, 0.014446025776115, 
		0.014446025776115, 0.594527068955871, 0.358539352205951, 
		0.594527068955871, 0.358539352205951, 0.046933578838178, 
		0.046933578838178, 0.839331473680839, 0.157807405968595, 
		0.839331473680839, 0.157807405968595, 0.002861120350567, 
		0.002861120350567, 0.701087978926173, 0.075050596975911, 
		0.701087978926173, 0.075050596975911, 0.223861424097916, 
		0.223861424097916, 0.822931324069857, 0.142421601113383, 
		0.822931324069857, 0.142421601113383, 0.034647074816760, 
		0.034647074816760, 0.924344252620784, 0.065494628082938, 
		0.924344252620784, 0.065494628082938, 0.010161119296278, 
		0.010161119296278};
	/*}}}*/
	/*p=20, npoint=79 {{{*/
	static IssmPDouble wgt20[]={
		0.057256499746719, 0.001501721280705, 0.001501721280705, 
		0.001501721280705, 0.020195803723819, 0.020195803723819, 
		0.020195803723819, 0.039624016090841, 0.039624016090841, 
		0.039624016090841, 0.052739185030045, 0.052739185030045, 
		0.052739185030045, 0.053043868444611, 0.053043868444611, 
		0.053043868444611, 0.042206713977986, 0.042206713977986, 
		0.042206713977986, 0.027708365070095, 0.027708365070095, 
		0.027708365070095, 0.013333849876622, 0.013333849876622, 
		0.013333849876622,-0.001094760895106,-0.001094760895106, 
		-0.001094760895106, 0.003033053580543, 0.003033053580543, 
		0.003033053580543, 0.028519670065604, 0.028519670065604, 
		0.028519670065604, 0.028519670065604, 0.028519670065604, 
		0.028519670065604, 0.008381451951650, 0.008381451951650, 
		0.008381451951650, 0.008381451951650, 0.008381451951650, 
		0.008381451951650, 0.044695409202580, 0.044695409202580, 
		0.044695409202580, 0.044695409202580, 0.044695409202580, 
		0.044695409202580, 0.014672360101834, 0.014672360101834, 
		0.014672360101834, 0.014672360101834, 0.014672360101834, 
		0.014672360101834, 0.031791643800640, 0.031791643800640, 
		0.031791643800640, 0.031791643800640, 0.031791643800640, 
		0.031791643800640, 0.001220064691226, 0.001220064691226, 
		0.001220064691226, 0.001220064691226, 0.001220064691226, 
		0.001220064691226, 0.017515684095300, 0.017515684095300, 
		0.017515684095300, 0.017515684095300, 0.017515684095300, 
		0.017515684095300, 0.006190192638113, 0.006190192638113, 
		0.006190192638113, 0.006190192638113, 0.006190192638113, 
		0.006190192638113};
	static IssmPDouble l120[]={
		0.333333333333333,-0.001900928704400, 0.500950464352200, 
		0.500950464352200, 0.023574084130543, 0.488212957934729, 
		0.488212957934729, 0.089726636099435, 0.455136681950283, 
		0.455136681950283, 0.196007481363421, 0.401996259318289, 
		0.401996259318289, 0.488214180481157, 0.255892909759421, 
		0.255892909759421, 0.647023488009788, 0.176488255995106, 
		0.176488255995106, 0.791658289326483, 0.104170855336758, 
		0.104170855336758, 0.893862072318140, 0.053068963840930, 
		0.053068963840930, 0.916762569607942, 0.041618715196029, 
		0.041618715196029, 0.976836157186356, 0.011581921406822, 
		0.011581921406822, 0.048741583664839, 0.048741583664839, 
		0.344855770229001, 0.606402646106160, 0.344855770229001, 
		0.606402646106160, 0.006314115948605, 0.006314115948605, 
		0.377843269594854, 0.615842614456541, 0.377843269594854, 
		0.615842614456541, 0.134316520547348, 0.134316520547348, 
		0.306635479062357, 0.559048000390295, 0.306635479062357, 
		0.559048000390295, 0.013973893962392, 0.013973893962392, 
		0.249419362774742, 0.736606743262866, 0.249419362774742, 
		0.736606743262866, 0.075549132909764, 0.075549132909764, 
		0.212775724802802, 0.711675142287434, 0.212775724802802, 
		0.711675142287434,-0.008368153208227,-0.008368153208227, 
		0.146965436053239, 0.861402717154987, 0.146965436053239, 
		0.861402717154987, 0.026686063258714, 0.026686063258714, 
		0.137726978828923, 0.835586957912363, 0.137726978828923, 
		0.835586957912363, 0.010547719294141, 0.010547719294141, 
		0.059696109149007, 0.929756171556853, 0.059696109149007, 
		0.929756171556853};
	static IssmPDouble l220[]={
		0.333333333333333, 0.500950464352200,-0.001900928704400, 
		0.500950464352200, 0.488212957934729, 0.023574084130543, 
		0.488212957934729, 0.455136681950283, 0.089726636099435, 
		0.455136681950283, 0.401996259318289, 0.196007481363421, 
		0.401996259318289, 0.255892909759421, 0.488214180481157, 
		0.255892909759421, 0.176488255995106, 0.647023488009788, 
		0.176488255995106, 0.104170855336758, 0.791658289326483, 
		0.104170855336758, 0.053068963840930, 0.893862072318140, 
		0.053068963840930, 0.041618715196029, 0.916762569607942, 
		0.041618715196029, 0.011581921406822, 0.976836157186356, 
		0.011581921406822, 0.344855770229001, 0.606402646106160, 
		0.048741583664839, 0.048741583664839, 0.606402646106160, 
		0.344855770229001, 0.377843269594854, 0.615842614456541, 
		0.006314115948605, 0.006314115948605, 0.615842614456541, 
		0.377843269594854, 0.306635479062357, 0.559048000390295, 
		0.134316520547348, 0.134316520547348, 0.559048000390295, 
		0.306635479062357, 0.249419362774742, 0.736606743262866, 
		0.013973893962392, 0.013973893962392, 0.736606743262866, 
		0.249419362774742, 0.212775724802802, 0.711675142287434, 
		0.075549132909764, 0.075549132909764, 0.711675142287434, 
		0.212775724802802, 0.146965436053239, 0.861402717154987, 
		-0.008368153208227,-0.008368153208227, 0.861402717154987, 
		0.146965436053239, 0.137726978828923, 0.835586957912363, 
		0.026686063258714, 0.026686063258714, 0.835586957912363, 
		0.137726978828923, 0.059696109149007, 0.929756171556853, 
		0.010547719294141, 0.010547719294141, 0.929756171556853, 
		0.059696109149007};
	static IssmPDouble l320[]={
		0.333333333333333, 0.500950464352200, 0.500950464352200, 
		-0.001900928704400, 0.488212957934729, 0.488212957934729, 
		0.023574084130543, 0.455136681950283, 0.455136681950283, 
		0.089726636099435, 0.401996259318289, 0.401996259318289, 
		0.196007481363421, 0.255892909759421, 0.255892909759421, 
		0.488214180481157, 0.176488255995106, 0.176488255995106, 
		0.647023488009788, 0.104170855336758, 0.104170855336758, 
		0.791658289326483, 0.053068963840930, 0.053068963840930, 
		0.893862072318140, 0.041618715196029, 0.041618715196029, 
		0.916762569607942, 0.011581921406822, 0.011581921406822, 
		0.976836157186356, 0.606402646106160, 0.344855770229001, 
		0.606402646106160, 0.344855770229001, 0.048741583664839, 
		0.048741583664839, 0.615842614456541, 0.377843269594854, 
		0.615842614456541, 0.377843269594854, 0.006314115948605, 
		0.006314115948605, 0.559048000390295, 0.306635479062357, 
		0.559048000390295, 0.306635479062357, 0.134316520547348, 
		0.134316520547348, 0.736606743262866, 0.249419362774742, 
		0.736606743262866, 0.249419362774742, 0.013973893962392, 
		0.013973893962392, 0.711675142287434, 0.212775724802802, 
		0.711675142287434, 0.212775724802802, 0.075549132909764, 
		0.075549132909764, 0.861402717154987, 0.146965436053239, 
		0.861402717154987, 0.146965436053239,-0.008368153208227, 
		-0.008368153208227, 0.835586957912363, 0.137726978828923, 
		0.835586957912363, 0.137726978828923, 0.026686063258714, 
		0.026686063258714, 0.929756171556853, 0.059696109149007, 
		0.929756171556853, 0.059696109149007, 0.010547719294141, 
		0.010547719294141};
	/*}}}*/

	static IssmPDouble* wgtp[MAX_TRIA_SYM_ORD]={
		wgt1 ,wgt2 ,wgt3 ,wgt4 ,wgt5 ,
		wgt6 ,wgt7 ,wgt8 ,wgt9 ,wgt10,
		wgt11,wgt12,wgt13,wgt14,wgt15,
		wgt16,wgt17,wgt18,wgt19,wgt20};
	static IssmPDouble* l1p [MAX_TRIA_SYM_ORD]={
		l11  ,l12  ,l13  ,l14  ,l15  ,
		l16  ,l17  ,l18  ,l19  ,l110 ,
		l111 ,l112 ,l113 ,l114 ,l115 ,
		l116 ,l117 ,l118 ,l119 ,l120 };
	static IssmPDouble* l2p [MAX_TRIA_SYM_ORD]={
		l21  ,l22  ,l23  ,l24  ,l25  ,
		l26  ,l27  ,l28  ,l29  ,l210 ,
		l211 ,l212 ,l213 ,l214 ,l215 ,
		l216 ,l217 ,l218 ,l219 ,l220 };
	static IssmPDouble* l3p [MAX_TRIA_SYM_ORD]={
		l31  ,l32  ,l33  ,l34  ,l35  ,
		l36  ,l37  ,l38  ,l39  ,l310 ,
		l311 ,l312 ,l313 ,l314 ,l315 ,
		l316 ,l317 ,l318 ,l319 ,l320 };

	static int np[MAX_TRIA_SYM_ORD]={sizeof(wgt1 )/sizeof(IssmPDouble),
		sizeof(wgt2 )/sizeof(IssmPDouble),
		sizeof(wgt3 )/sizeof(IssmPDouble),
		sizeof(wgt4 )/sizeof(IssmPDouble),
		sizeof(wgt5 )/sizeof(IssmPDouble),
		sizeof(wgt6 )/sizeof(IssmPDouble),
		sizeof(wgt7 )/sizeof(IssmPDouble),
		sizeof(wgt8 )/sizeof(IssmPDouble),
		sizeof(wgt9 )/sizeof(IssmPDouble),
		sizeof(wgt10)/sizeof(IssmPDouble),
		sizeof(wgt11)/sizeof(IssmPDouble),
		sizeof(wgt12)/sizeof(IssmPDouble),
		sizeof(wgt13)/sizeof(IssmPDouble),
		sizeof(wgt14)/sizeof(IssmPDouble),
		sizeof(wgt15)/sizeof(IssmPDouble),
		sizeof(wgt16)/sizeof(IssmPDouble),
		sizeof(wgt17)/sizeof(IssmPDouble),
		sizeof(wgt18)/sizeof(IssmPDouble),
		sizeof(wgt19)/sizeof(IssmPDouble),
		sizeof(wgt20)/sizeof(IssmPDouble)};

	//	_pprintLine_("GaussLegendreTria: iord=" << iord);

	/*  check to see if Gauss points need to be calculated  */
	if (iord <= MAX_TRIA_SYM_ORD) {

		/*  copy the points from the static arrays (noting that the pointers
			 could be returned directly, but then the calling function would
			 have to know to not free them)  */

		*pngaus=np[iord-1];

		*pl1  =xNew<IssmDouble>(*pngaus);
		*pl2  =xNew<IssmDouble>(*pngaus);
		*pl3  =xNew<IssmDouble>(*pngaus);
		*pwgt =xNew<IssmDouble>(*pngaus);

		for (i=0; i<*pngaus; i++) {
			(*pl1 )[i]=l1p [iord-1][i];
			(*pl2 )[i]=l2p [iord-1][i];
			(*pl3 )[i]=l3p [iord-1][i];
			(*pwgt)[i]=wgtp[iord-1][i];
		}
	}
	else {

		/*  calculate the Gauss points from the collapsed quadrilateral  */
		nigaus =iord/2+1;
		*pngaus=nigaus*nigaus;

		*pl1  =xNew<IssmDouble>(*pngaus);
		*pl2  =xNew<IssmDouble>(*pngaus);
		*pl3  =xNew<IssmDouble>(*pngaus);
		*pwgt =xNew<IssmDouble>(*pngaus);

		/*  get the gauss points in each direction  */
		GaussLegendreLinear(&xgaus, &xwgt, nigaus);

		egaus=xgaus;
		ewgt =xwgt;

		/*  collapse the gauss points into the triangle and transform into
			 area coordinates  */
		ipt=0;
		for (j=0; j<nigaus; j++) {
			for (i=0; i<nigaus; i++) {
				xi        =      1./2.*(1.-egaus[j])*xgaus[i];
				eta       =SQRT3/2.*(1.+egaus[j]);
				(*pwgt)[ipt]=xwgt[i]*ewgt[j]*(SQRT3/4.*(1.-egaus[j]));

				(*pl1 )[ipt]=(1.-xi-eta/SQRT3)/2.;
				(*pl2 )[ipt]=(1.+xi-eta/SQRT3)/2.;
				(*pl3 )[ipt]=       eta/SQRT3;

				ipt++;
			}
		}
		xDelete<IssmPDouble>(xwgt );
		xDelete<IssmPDouble>(xgaus);
	}

	//	_pprintLine_("GaussLegendreTria - ngaus=" << *pngaus);
	//	for (i=0; i<*pngaus; i++)
	//		_printf_(true,"i=%d: l1gaus=%f,l2gaus=%f,l3gaus=%f,wgt=%f\n",
	//				 i,(*pl1 )[i],(*pl2 )[i],(*pl3 )[i],(*pwgt)[i]);

	return;
}/*}}}*/
/*FUNCTION GaussLegendreTetra{{{*/
void GaussLegendreTetra( int* pngaus, IssmPDouble** pl1, IssmPDouble** pl2, IssmPDouble** pl3, IssmPDouble** pl4, IssmPDouble** pwgt, int iord ) {
	/* Gauss quadrature points for the tetrahedron.

		p=2-3 points from Y. Jinyun, "Symmetric Gaussian Quadrature
		Formulae for Tetrahedronal Regions", Computer Methods in Applied
		Mechanics and Engineering, Vol. 43, pp. 349-353 (1984).

		p=4-6 points from P. Keast, "Moderate-Degree Tetrahedral
		Quadrature Formulas", Computer Methods in Applied Mechanics and
		Engineering, Vol. 55, pp. 339-348 (1986).*/

	/*Intermediaries*/
	int i,j,k,ipt,nigaus;
	IssmPDouble xi,eta,zeta;
	IssmPDouble *xgaus=NULL,*xwgt=NULL,*egaus,*ewgt,*zgaus,*zwgt;

	/*Hardcoded Gauss points definition*/
	/*p= 1, npoint= 1  {{{*/
	static IssmPDouble wgt1[]={
		1.000000000000000};
	static IssmPDouble l11[]={
		0.250000000000000};
	static IssmPDouble l21[]={
		0.250000000000000};
	static IssmPDouble l31[]={
		0.250000000000000};
	static IssmPDouble l41[]={
		0.250000000000000};
	/*}}}*/
	/*p= 2, npoint= 4  {{{*/

	static IssmPDouble wgt2[]={
		0.250000000000000, 0.250000000000000, 0.250000000000000,
		0.250000000000000};
	static IssmPDouble l12[]={
		0.585410196624969, 0.138196601125011, 0.138196601125011,
		0.138196601125011};
	static IssmPDouble l22[]={
		0.138196601125011, 0.585410196624969, 0.138196601125011,
		0.138196601125011};
	static IssmPDouble l32[]={
		0.138196601125011, 0.138196601125011, 0.585410196624969,
		0.138196601125011};
	static IssmPDouble l42[]={
		0.138196601125011, 0.138196601125011, 0.138196601125011,
		0.585410196624969};
	/*}}}*/
	/*p= 3, npoint= 5  {{{*/
	static IssmPDouble wgt3[]={
		-0.800000000000000, 0.450000000000000, 0.450000000000000, 
		0.450000000000000, 0.450000000000000};
	static IssmPDouble l13[]={
		0.250000000000000, 0.500000000000000, 0.166666666666667, 
		0.166666666666667, 0.166666666666667};
	static IssmPDouble l23[]={
		0.250000000000000, 0.166666666666667, 0.500000000000000, 
		0.166666666666667, 0.166666666666667};
	static IssmPDouble l33[]={
		0.250000000000000, 0.166666666666667, 0.166666666666667, 
		0.500000000000000, 0.166666666666667};
	static IssmPDouble l43[]={
		0.250000000000000, 0.166666666666667, 0.166666666666667, 
		0.166666666666667, 0.500000000000000};
	/*}}}*/
	/*p= 4, npoint=11  {{{*/

	static IssmPDouble wgt4[]={
		-0.013155555555556, 0.007622222222222, 0.007622222222222, 
		0.007622222222222, 0.007622222222222, 0.024888888888889,
		0.024888888888889, 0.024888888888889, 0.024888888888889, 
		0.024888888888889, 0.024888888888889};
	static IssmPDouble l14[]={
		0.250000000000000, 0.785714285714286, 0.071428571428571, 
		0.071428571428571, 0.071428571428571, 0.399403576166799,
		0.399403576166799, 0.399403576166799, 0.100596423833201, 
		0.100596423833201, 0.100596423833201};
	static IssmPDouble l24[]={
		0.250000000000000, 0.071428571428571, 0.785714285714286, 
		0.071428571428571, 0.071428571428571, 0.399403576166799,
		0.100596423833201, 0.100596423833201, 0.399403576166799, 
		0.399403576166799, 0.100596423833201};
	static IssmPDouble l34[]={
		0.250000000000000, 0.071428571428571, 0.071428571428571, 
		0.785714285714286, 0.071428571428571, 0.100596423833201, 
		0.399403576166799, 0.100596423833201, 0.399403576166799,
		0.100596423833201, 0.399403576166799};
	static IssmPDouble l44[]={
		0.250000000000000, 0.071428571428571, 0.071428571428571, 
		0.071428571428571, 0.785714285714286, 0.100596423833201, 
		0.100596423833201, 0.399403576166799, 0.100596423833201,
		0.399403576166799, 0.399403576166799};
	/*}}}*/
	/*p= 5, npoint=15  {{{*/

	static IssmPDouble wgt5[]={
		0.030283678097089, 0.006026785714286, 0.006026785714286, 
		0.006026785714286, 0.006026785714286, 0.011645249086029, 
		0.011645249086029, 0.011645249086029, 0.011645249086029,
		0.010949141561386, 0.010949141561386, 0.010949141561386,
		0.010949141561386, 0.010949141561386, 0.010949141561386};
	static IssmPDouble l15[]={
		0.250000000000000, 0.000000000000000, 0.333333333333333, 
		0.333333333333333, 0.333333333333333, 0.727272727272727, 
		0.090909090909091, 0.090909090909091, 0.090909090909091, 
		0.066550153573664, 0.066550153573664, 0.066550153573664, 
		0.433449846426336, 0.433449846426336, 0.433449846426336};
	static IssmPDouble l25[]={
		0.250000000000000, 0.333333333333333, 0.000000000000000, 
		0.333333333333333, 0.333333333333333, 0.090909090909091, 
		0.727272727272727, 0.090909090909091, 0.090909090909091, 
		0.066550153573664, 0.433449846426336, 0.433449846426336, 
		0.066550153573664, 0.066550153573664, 0.433449846426336};
	static IssmPDouble l35[]={
		0.250000000000000, 0.333333333333333, 0.333333333333333, 
		0.000000000000000, 0.333333333333333, 0.090909090909091, 
		0.090909090909091, 0.727272727272727, 0.090909090909091, 
		0.433449846426336, 0.066550153573664, 0.433449846426336, 
		0.066550153573664, 0.433449846426336, 0.066550153573664};
	static IssmPDouble l45[]={
		0.250000000000000, 0.333333333333333, 0.333333333333333, 
		0.333333333333333, 0.000000000000000, 0.090909090909091, 
		0.090909090909091, 0.090909090909091, 0.727272727272727, 
		0.433449846426336, 0.433449846426336, 0.066550153573664, 
		0.433449846426336, 0.066550153573664, 0.066550153573664};
	/*}}}*/
	/*p= 6, npoint=24  {{{*/

	static IssmPDouble wgt6[]={
		0.006653791709695, 0.006653791709695, 0.006653791709695, 
		0.006653791709695, 0.001679535175887, 0.001679535175887, 
		0.001679535175887, 0.001679535175887, 0.009226196923942, 
		0.009226196923942, 0.009226196923942, 0.009226196923942,
		0.008035714285714, 0.008035714285714, 0.008035714285714, 
		0.008035714285714, 0.008035714285714, 0.008035714285714, 
		0.008035714285714, 0.008035714285714, 0.008035714285714, 
		0.008035714285714, 0.008035714285714, 0.008035714285714};
	static IssmPDouble l16[]={
		0.356191386222545, 0.214602871259152, 0.214602871259152, 
		0.214602871259152, 0.877978124396166, 0.040673958534611, 
		0.040673958534611, 0.040673958534611, 0.032986329573173, 
		0.322337890142276, 0.322337890142276, 0.322337890142276,

		0.063661001875018, 0.063661001875018, 0.063661001875018, 
		0.063661001875018, 0.063661001875018, 0.063661001875018, 
		0.269672331458316, 0.603005664791649, 0.269672331458316, 
		0.603005664791649, 0.269672331458316, 0.603005664791649};
	static IssmPDouble l26[]={
		0.214602871259152, 0.356191386222545, 0.214602871259152, 
		0.214602871259152, 0.040673958534611, 0.877978124396166, 
		0.040673958534611, 0.040673958534611, 0.322337890142276, 
		0.032986329573173, 0.322337890142276, 0.322337890142276,

		0.063661001875018, 0.063661001875018, 0.269672331458316, 
		0.603005664791649, 0.269672331458316, 0.603005664791649, 
		0.063661001875018, 0.063661001875018, 0.063661001875018, 
		0.063661001875018, 0.603005664791649, 0.269672331458316};
	static IssmPDouble l36[]={
		0.214602871259152, 0.214602871259152, 0.356191386222545, 
		0.214602871259152, 0.040673958534611, 0.040673958534611, 
		0.877978124396166, 0.040673958534611, 0.322337890142276, 
		0.322337890142276, 0.032986329573173, 0.322337890142276,

		0.269672331458316, 0.603005664791649, 0.063661001875018, 
		0.063661001875018, 0.603005664791649, 0.269672331458316, 
		0.063661001875018, 0.063661001875018, 0.603005664791649, 
		0.269672331458316, 0.063661001875018, 0.063661001875018};
	static IssmPDouble l46[]={
		0.214602871259152, 0.214602871259152, 0.214602871259152, 
		0.356191386222545, 0.040673958534611, 0.040673958534611, 
		0.040673958534611, 0.877978124396166, 0.322337890142276, 
		0.322337890142276, 0.322337890142276, 0.032986329573173,

		0.603005664791649, 0.269672331458316, 0.603005664791649, 
		0.269672331458316, 0.063661001875018, 0.063661001875018, 
		0.603005664791649, 0.269672331458316, 0.063661001875018, 
		0.063661001875018, 0.063661001875018, 0.063661001875018};
	/*}}}*/

	static IssmPDouble* wgtp[MAX_TETRA_SYM_ORD]={wgt1,wgt2,wgt3,wgt4,wgt5,wgt6};
	static IssmPDouble* l1p [MAX_TETRA_SYM_ORD]={l11 ,l12 ,l13 ,l14 ,l15 ,l16 };
	static IssmPDouble* l2p [MAX_TETRA_SYM_ORD]={l21 ,l22 ,l32 ,l24 ,l25 ,l26 };
	static IssmPDouble* l3p [MAX_TETRA_SYM_ORD]={l31 ,l32 ,l33 ,l34 ,l35 ,l36 };
	static IssmPDouble* l4p [MAX_TETRA_SYM_ORD]={l41 ,l42 ,l43 ,l44 ,l45 ,l46 };

	static int np[MAX_TETRA_SYM_ORD]={sizeof(wgt1 )/sizeof(IssmPDouble),
		sizeof(wgt2 )/sizeof(IssmPDouble),
		sizeof(wgt3 )/sizeof(IssmPDouble),
		sizeof(wgt4 )/sizeof(IssmPDouble),
		sizeof(wgt5 )/sizeof(IssmPDouble),
		sizeof(wgt6 )/sizeof(IssmPDouble)};

	//	_pprintLine_("GaussLegendreTetra: iord=" << iord);

	/*  check to see if Gauss points need to be calculated  */
	if (iord <= MAX_TETRA_SYM_ORD) {

		/*  copy the points from the static arrays (noting that the pointers
			 could be returned directly, but then the calling function would
			 have to know to not free them), and multiply the weights by the
			 volume of the parametric tetrahedron  */

		*pngaus=np[iord-1];

		*pl1  =xNew<IssmPDouble>(*pngaus);
		*pl2  =xNew<IssmPDouble>(*pngaus);
		*pl3  =xNew<IssmPDouble>(*pngaus);
		*pl4  =xNew<IssmPDouble>(*pngaus);
		*pwgt =xNew<IssmPDouble>(*pngaus);

		for (i=0; i<*pngaus; i++) {
			(*pl1 )[i]=l1p [iord-1][i];
			(*pl2 )[i]=l2p [iord-1][i];
			(*pl3 )[i]=l3p [iord-1][i];
			(*pl4 )[i]=l4p [iord-1][i];
			(*pwgt)[i]=wgtp[iord-1][i]*2./3.*SQRT2;
		}
	}
	else {

		/*  calculate the Gauss points from the collapsed hexahedron  */
		nigaus =iord/2+1;
		*pngaus=nigaus*nigaus*nigaus;

		*pl1  =xNew<IssmPDouble>(*pngaus);
		*pl2  =xNew<IssmPDouble>(*pngaus);
		*pl3  =xNew<IssmPDouble>(*pngaus);
		*pl4  =xNew<IssmPDouble>(*pngaus);
		*pwgt =xNew<IssmPDouble>(*pngaus);

		/*  get the gauss points in each direction  */
		GaussLegendreLinear(&xgaus, &xwgt, nigaus);

		egaus=xgaus;
		ewgt =xwgt;
		zgaus=xgaus;
		zwgt =xwgt;

		/*  collapse the gauss points into the tetrahedron and transform into
			 volume coordinates  */
		ipt=0;
		for (k=0; k<nigaus; k++) {
			for (j=0; j<nigaus; j++) {
				for (i=0; i<nigaus; i++) {
					xi        =1./4.*(1.-egaus[j])*(1.-zgaus[k])*xgaus[i];
					eta       =1./4./SQRT3
					  *(5.+3.*egaus[j]-zgaus[k]-3.*egaus[j]*zgaus[k]);
					zeta      =sqrt(2./3.)*(1.+zgaus[k]);
					(*pwgt)[ipt]=xwgt[i]*ewgt[j]*zwgt[k]
					  *(SQRT2/16.
								  *(1.-egaus[j])*pow(1.-zgaus[k],2.));

					(*pl1 )[ipt]=(1.-xi-eta/SQRT3-zeta/sqrt(6.))/2.;
					(*pl2 )[ipt]=(1.+xi-eta/SQRT3-zeta/sqrt(6.))/2.;
					(*pl3 )[ipt]=(      eta         -zeta/sqrt(8.))/SQRT3;
					(*pl4 )[ipt]=(                   zeta/sqrt(8.))*SQRT3;

					ipt++;
				}
			}
		}
		xDelete<IssmPDouble>(xwgt );
		xDelete<IssmPDouble>(xgaus);
	}
}/*}}}*/
/*FUNCTION GaussLobatto{{{*/
void GaussLobatto( IssmPDouble** pxgaus, IssmPDouble** pxwgt, int ngaus ) {
	/*Gauss-Lobatto quadrature points.

	  The recurrence coefficients for Legendre polynomials on (-1,1)
	  are defined (from the ORTHPOL subroutine RECUR with ipoly=1) as:

	  alpha(i)=0.
	  beta (i)=1./(4.-1./(i-1)^2))

	  and then modified for the Gauss-Lobatto quadrature rule on (-1,1)
	  (from the ORTHPOL subroutine LOB).

	  For degree p, the required number of Gauss-Lobatto points is
	  n>=(p+1)/2+1 (one more than Gauss-Legendre).*/

	int i;
	IssmPDouble *alpha,*beta;
	IssmPDouble left=-1.,right= 1.;
	IssmPDouble p0l=0.,p0r=0.,p1l=1.,p1r=1.,pm1l,pm1r,det;

	/*p= 1, npoint= 1 (Gauss-Legendre)*/
	static IssmPDouble wgt1[]={2.000000000000000};
	static IssmPDouble xi1[]={0.000000000000000};

	/*p= 1, npoint= 2*/
	static IssmPDouble wgt2[]={1.000000000000000, 1.000000000000000};
	static IssmPDouble xi2[]={-1.000000000000000, 1.000000000000000};

	/*p= 3, npoint= 3*/
	static IssmPDouble wgt3[]={0.333333333333333, 1.333333333333333, 0.333333333333333};
	static IssmPDouble xi3[]={-1.000000000000000, 0.000000000000000, 1.000000000000000};

	/*p= 5, npoint= 4*/
	static IssmPDouble wgt4[]={0.166666666666667, 0.833333333333333, 0.833333333333333, 0.166666666666667};
	static IssmPDouble xi4[]={-1.000000000000000,-0.447213595499958, 0.447213595499958, 1.000000000000000};

	/*p= 7, npoint= 5*/
	static IssmPDouble wgt5[]={0.100000000000000, 0.544444444444444, 0.711111111111111, 0.544444444444444, 0.100000000000000};
	static IssmPDouble xi5[]={-1.000000000000000,-0.654653670707977, 0.000000000000000, 0.654653670707977, 1.000000000000000};

	static IssmPDouble* wgtp[MAX_LINE_GLOB_PTS]={wgt1 ,wgt2 ,wgt3 ,wgt4 ,wgt5 };
	static IssmPDouble* xip [MAX_LINE_GLOB_PTS]={xi1  ,xi2  ,xi3  ,xi4  ,xi5  };

	static int np[MAX_LINE_GLOB_PTS]={sizeof(wgt1 )/sizeof(IssmPDouble),
		sizeof(wgt2 )/sizeof(IssmPDouble),
		sizeof(wgt3 )/sizeof(IssmPDouble),
		sizeof(wgt4 )/sizeof(IssmPDouble),
		sizeof(wgt5 )/sizeof(IssmPDouble)};

	//	_pprintLine_("Gauss-Lobatto recurrence coefficients ngaus=" << ngaus);
	*pxgaus =xNew<IssmPDouble>(ngaus);
	*pxwgt  =xNew<IssmPDouble>(ngaus);

	/*  check to see if Gauss points need to be calculated  */
	if (ngaus <= MAX_LINE_GLOB_PTS) {

		/*  copy the points from the static arrays (noting that the pointers
			 could be returned directly, but then the calling function would
			 have to know to not free them)  */
		for (i=0; i<ngaus; i++) {
			(*pxgaus)[i]=xip [ngaus-1][i];
			(*pxwgt )[i]=wgtp[ngaus-1][i];
		}
	}
	else {

		/*  calculate the Gauss points using recurrence relations  */
		alpha=xNew<IssmPDouble>(ngaus);
		beta =xNew<IssmPDouble>(ngaus);

		/*  calculate the Legendre recurrence coefficients  */
		alpha[0]=0.;
		beta [0]=2.;

		for (i=1; i<ngaus; i++) {
			alpha[i]=0.;
			beta [i]=1./(4.-1./(i*i));
		}

		/*  calculate the Gauss-Lobatto quadrature formula  */
		for (i=0; i<ngaus-1; i++) {
			pm1l=p0l;
			p0l=p1l;
			pm1r=p0r;
			p0r=p1r;
			p1l=(left -alpha[i])*p0l-beta[i]*pm1l;
			p1r=(right-alpha[i])*p0r-beta[i]*pm1r;
		}

		/*  Normalize system to prevent underflow:
			 [ p1l p0l ]{ a } = {left *p1l}
			 [ p1r p0r ]{ b }   {right*p1r}
			 dividing by p1l in the first equation and p1r in the second.  */

		//		det=p1l*p0r-p1r*p0l;
		det=p0r/p1r-p0l/p1l;
		//		alpha[ngaus-1]=(left*p1l*p0r-right*p1r*p0l)/det;
		//		beta [ngaus-1]=(right-left)*p1l*p1r/det;
		alpha[ngaus-1]=(left *(p0r/p1r)-right*(p0l/p1l))/det;
		beta [ngaus-1]=(right          -left           )/det;

		/*  calculate the Gauss points  */
		GaussRecur(*pxgaus, *pxwgt, ngaus, alpha, beta );
		xDelete<IssmPDouble>(beta );
		xDelete<IssmPDouble>(alpha);
	}

}/*}}}*/
/*FUNCTION GaussRecur{{{*/
void GaussRecur( IssmPDouble* zero, IssmPDouble* weight, int n, IssmPDouble* alpha, IssmPDouble* beta ) {
	/*Gauss quadrature points from recursion coefficients.
	 *
	 *The routine uses the algorithm from the ORTHPOL routine GAUSS, which
	 *finds the eigenvalues of a tridiagonal matrix.*/

	/*Intermediaries*/
	int i,j,k,l,m,ii,mml,iter;
	IssmPDouble p,g,r,s,c,f,b;
	IssmPDouble* work;

	if (n==1){
		zero[0]  =alpha[0];
		weight[0]=beta[0];
		return;
	}

	work=xNew<IssmPDouble>(n);

	zero[0]  =alpha[0];
	weight[0]=1.;
	work[n-1]=0.;
	for (i=1; i<=n-1; i++){
		zero[i]=alpha[i];
		work[i-1]=sqrt(beta[i]);
		weight[i]=0;
	}

	for (l=0; l<=n-1; l++){
		iter=0;
		do {

			/*  Look for a small subdiagonal element.  */
			for (m=l; m<=n-1; m++) {
				if (m == n-1) break;
				if (fabs(work[m])
							<= DBL_EPSILON*(fabs(zero[m])+fabs(zero[m+1])))
				 break;
			}
			p=zero[l];
			if (m==l) break;
			++iter;

			/*  Form shift.  */
			g=(zero[l+1]-p)/(2.*work[l]);
			r=sqrt(g*g+1.);
			//			g=zero[m]-p+work[l]/(g+FortranSign(r,g));
			g=zero[m]-p+work[l]/(g+(g>=0 ? fabs(r) : -fabs(r)));
			s=1.;
			c=1.;
			p=0.;
			mml=m-l;

			/*  For i=m-1 step -1 until l do ...  */
			for (ii=1; ii<=mml; ii++) {
				i=m-ii;
				f=s*work[i];
				b=c*work[i];
				if (fabs(f) >= fabs(g)) {
					c=g/f;
					r=sqrt(c*c+1.);
					work[i+1]=f*r;
					s=1./r;
					c*=s;
				}
				else {
					s=f/g;
					r=sqrt(s*s+1.);
					work[i+1]=g*r;
					c=1./r;
					s*=c;
				}
				g=zero[i+1]-p;
				r=(zero[i]-g)*s+2.*c*b;
				p=s*r;
				zero[i+1]=g+p;
				g=c*r-b;

				/*  Form first component of vector.  */
				f=weight[i+1];
				weight[i+1]=s*weight[i]+c*f;
				weight[i  ]=c*weight[i]-s*f;
			}
			zero[l]-=p;
			work[l]=g;
			work[m]=0.;
		} while (iter < MAX_GAUS_ITER);
		if (iter >= MAX_GAUS_ITER) {
			xDelete<IssmPDouble>(work);
			_error_("Max iterations exceeded for l=" << MAX_GAUS_ITER);
		}
	}

	/*  Order eigenvalues and eigenvectors.  */
	for (i=0;i<n-1;i++) {
		k=i;
		p=zero[i];
		for (j=i+1;j<n;j++){
			if (zero[j] < p){
				k=j;
				p=zero[j];
			}
		}
		if (k > i){
			p=zero[i];
			zero[i]=zero[k];
			zero[k]=p;
			p=weight[i];
			weight[i]=weight[k];
			weight[k]=p;
		}
	}
	for (i=0;i<n;i++){
		weight[i]=beta[0]*weight[i]*weight[i];
	}

	/*Cleanup*/
	xDelete<IssmPDouble>(work);

}/*}}}*/

/*Element Gauss points TO BE REMOVED*/
/*FUNCTION gaussQuad{{{*/
void gaussQuad( IssmPDouble** pxgaus, IssmPDouble** pxwgt, IssmPDouble** pegaus, IssmPDouble** pewgt, int nigaus, int njgaus ) {
	/*Gauss quadrature points for the quadrilaterial.*/

	/*get the gauss points using the product of two line rules  */
	GaussLegendreLinear(pxgaus, pxwgt, nigaus);
	GaussLegendreLinear(pegaus, pewgt, njgaus);
}/*}}}*/
/*FUNCTION gaussHexa{{{*/
void gaussHexa( IssmPDouble** pxgaus, IssmPDouble** pxwgt, IssmPDouble** pegaus, IssmPDouble** pewgt, IssmPDouble** pzgaus, IssmPDouble ** pzwgt, int nigaus, int njgaus, int nkgaus ) {
	/*Gauss quadrature points for the hexahedron.*/

	/*  get the gauss points using the product of three line rules  */
	GaussLegendreLinear(pxgaus, pxwgt, nigaus);
	GaussLegendreLinear(pegaus, pewgt, njgaus);
	GaussLegendreLinear(pzgaus, pzwgt, nkgaus);
}/*}}}*/
