/*!\file: transient_3d_core.cpp
 * \brief: core of the transient_3d solution 
 */ 

#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <float.h>
#include "./analyses.h"
#include "../toolkits/toolkits.h"
#include "../classes/objects/objects.h"
#include "../shared/io/io.h"
#include "../shared/Enum/Enum.h"
#include "../modules/modules.h"
#include "../solutionsequences/solutionsequences.h"

void transient_core(FemModel* femmodel){

	/*parameters: */
	IssmDouble starttime,finaltime,dt,yts;
	bool   isdiagnostic,isprognostic,isthermal,isgroundingline,isenthalpy,isdelta18o,isgia;
	bool   save_results,dakota_analysis;
	bool   time_adapt=false;
	int    output_frequency;
	int    dim,groundingline_migration;
	int    numoutputs         = 0;
	int    *requested_outputs = NULL;

	/*intermediary: */
	int    step;
	IssmDouble time;

	//first recover parameters common to all solutions
	femmodel->parameters->FindParam(&dim,MeshDimensionEnum);
	femmodel->parameters->FindParam(&starttime,TimesteppingStartTimeEnum);
	femmodel->parameters->FindParam(&finaltime,TimesteppingFinalTimeEnum);
	femmodel->parameters->FindParam(&dt,TimesteppingTimeStepEnum);
	femmodel->parameters->FindParam(&yts,ConstantsYtsEnum);
	femmodel->parameters->FindParam(&dakota_analysis,QmuIsdakotaEnum);
	femmodel->parameters->FindParam(&output_frequency,SettingsOutputFrequencyEnum);
	femmodel->parameters->FindParam(&time_adapt,TimesteppingTimeAdaptEnum);
	femmodel->parameters->FindParam(&isdiagnostic,TransientIsdiagnosticEnum);
	femmodel->parameters->FindParam(&isprognostic,TransientIsprognosticEnum);
	femmodel->parameters->FindParam(&isthermal,TransientIsthermalEnum);
	femmodel->parameters->FindParam(&isgia,TransientIsgiaEnum);
	femmodel->parameters->FindParam(&isgroundingline,TransientIsgroundinglineEnum);
	femmodel->parameters->FindParam(&isenthalpy,ThermalIsenthalpyEnum);
	if(isgroundingline) femmodel->parameters->FindParam(&groundingline_migration,GroundinglineMigrationEnum);
	femmodel->parameters->FindParam(&numoutputs,TransientNumRequestedOutputsEnum);
	if(numoutputs) femmodel->parameters->FindParam(&requested_outputs,&numoutputs,TransientRequestedOutputsEnum);
	femmodel->parameters->FindParam(&isdelta18o,SurfaceforcingsIsdelta18oEnum);

	/*initialize: */
	step=0;
	time=starttime;

	/*for qmu analysis, reinitialize velocity so that fake sensitivities do not show up as a result of a different restart of the convergence at each trial.*/
	if(dakota_analysis){
		InputDuplicatex(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,QmuVxEnum,VxEnum);
		InputDuplicatex(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,QmuVyEnum,VyEnum);
		InputDuplicatex(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,QmuVzEnum,VzEnum);
		InputDuplicatex(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,QmuPressureEnum,PressureEnum);
		InputDuplicatex(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,QmuThicknessEnum,ThicknessEnum);
		InputDuplicatex(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,QmuSurfaceEnum,SurfaceEnum);
		InputDuplicatex(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,QmuBedEnum,BedEnum);
		if(isthermal && dim==3){
			InputDuplicatex(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,QmuTemperatureEnum,TemperatureEnum);
			InputDuplicatex(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,QmuMeltingEnum,BasalforcingsMeltingRateEnum);
		}
	}

	while(time < finaltime - (yts*DBL_EPSILON)){ //make sure we run up to finaltime.

		/*Increment*/
		if(time_adapt){
			femmodel->TimeAdaptx(&dt);
			if(time+dt>finaltime) dt=finaltime-time;
			femmodel->parameters->SetParam(dt,TimesteppingTimeStepEnum);
		}
		step+=1;
		time+=dt;
		femmodel->parameters->SetParam(time,TimeEnum);
		femmodel->parameters->SetParam(step,StepEnum);

		if(VerboseSolution()) _pprintLine_("iteration " << step << "/" << floor((finaltime-time)/dt)+step << "  time [yr]: " << time/yts << " (time step: " << dt/yts << ")");
		if(step%output_frequency==0 || time==finaltime)
		 save_results=true;
		else
		 save_results=false;
		femmodel->parameters->SetParam(save_results,SaveResultsEnum);

		if(isthermal && dim==3){
			if(VerboseSolution()) _pprintLine_("   computing temperatures");
			#ifdef _HAVE_THERMAL_
			if(isenthalpy==0){
				thermal_core(femmodel);
			}
			else{
				enthalpy_core(femmodel);
			}
			#else
			_error_("ISSM was not compiled with thermal capabilities. Exiting");
			#endif
		}

		if(isdiagnostic){
			if(VerboseSolution()) _pprintLine_("   computing new velocity");
			#ifdef _HAVE_DIAGNOSTIC_
			diagnostic_core(femmodel);
			#else
			_error_("ISSM was not compiled with diagnostic capabilities. Exiting");
			#endif
		}

		if(isprognostic){
			if(VerboseSolution()) _pprintLine_("   computing new thickness");
			prognostic_core(femmodel);
			if(VerboseSolution()) _pprintLine_("   updating vertices positions");
			femmodel->UpdateVertexPositionsx();
		}

		if(isgroundingline){
			if(VerboseSolution()) _pprintLine_("   computing new grounding line position");
			#ifdef _HAVE_GROUNDINGLINE_
			GroundinglineMigrationx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);
			#else
			_error_("ISSM was not compiled with grounding line migration capabilities. Exiting");
			#endif
		}
		if(isgia){
			if(VerboseSolution()) _pprintLine_("   computing glacial isostatic adjustment");
			#ifdef _HAVE_GIA_
			gia_core(femmodel);
			#else
			_error_("ISSM was not compiled with gia capabilities. Exiting");
			#endif

		}

		/*unload results*/
		if(save_results){
			if(VerboseSolution()) _pprintLine_("   saving transient results");
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,SurfaceEnum);
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,BedEnum);
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,SurfaceforcingsMassBalanceEnum);
			InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,MaskElementonfloatingiceEnum);
			femmodel->RequestedOutputsx(requested_outputs,numoutputs);

			if(isdelta18o){
				InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,SurfaceforcingsMonthlytemperaturesEnum);
				InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,SurfaceforcingsPrecipitationEnum);
			        InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,BasalFrictionEnum);
			}
			if(isgroundingline && (groundingline_migration==SubelementMigrationEnum || groundingline_migration==SubelementMigration2Enum)){
				InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,GLlevelsetEnum);
			}

			if(VerboseSolution()) _pprintLine_("   saving temporary results");
			OutputResultsx(femmodel->elements, femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,femmodel->results);
		}
	}

	femmodel->RequestedDependentsx();

	/*Free ressources:*/
	xDelete<int>(requested_outputs);
}
