/*!\file: solver_stokescoupling_nonlinear.cpp
 * \brief: core of the coupling between stokes and macayealpattyn
 */ 

#include "./solvers.h"
#include "../toolkits/toolkits.h"
#include "../classes/objects/objects.h"
#include "../shared/Enum/Enum.h"
#include "../shared/io/io.h"
#include "../modules/modules.h"
#include "../analyses/analyses.h"

void solver_stokescoupling_nonlinear(FemModel* femmodel,bool conserve_loads){

	/*intermediary: */
	Matrix<IssmDouble> *Kff_horiz    = NULL;
	Matrix<IssmDouble> *Kfs_horiz    = NULL;
	Vector<IssmDouble> *ug_horiz     = NULL;
	Vector<IssmDouble> *uf_horiz     = NULL;
	Vector<IssmDouble> *old_uf_horiz = NULL;
	Vector<IssmDouble> *pf_horiz     = NULL;
	Vector<IssmDouble> *df_horiz     = NULL;
	Matrix<IssmDouble> *Kff_vert     = NULL;
	Matrix<IssmDouble> *Kfs_vert     = NULL;
	Vector<IssmDouble> *ug_vert      = NULL;
	Vector<IssmDouble> *uf_vert      = NULL;
	Vector<IssmDouble> *pf_vert      = NULL;
	Vector<IssmDouble> *df_vert      = NULL;
	Vector<IssmDouble> *ys           = NULL;
	bool converged;
	int  count;

	/*parameters:*/
	int  min_mechanical_constraints;
	int  max_nonlinear_iterations;
	int  configuration_type;

	/*Recover parameters: */
	femmodel->parameters->FindParam(&min_mechanical_constraints,DiagnosticRiftPenaltyThresholdEnum);
	femmodel->parameters->FindParam(&max_nonlinear_iterations,DiagnosticMaxiterEnum);
	femmodel->UpdateConstraintsx();

	count=1;
	converged=false;

	/*First get ug_horiz:*/
	femmodel->SetCurrentConfiguration(DiagnosticHorizAnalysisEnum);
	GetSolutionFromInputsx(&ug_horiz, femmodel->elements, femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters);
	Reducevectorgtofx(&uf_horiz, ug_horiz, femmodel->nodes,femmodel->parameters);

	for(;;){

		/*First diagnostic horiz:*/
		femmodel->SetCurrentConfiguration(DiagnosticHorizAnalysisEnum);
		femmodel->parameters->FindParam(&configuration_type,ConfigurationTypeEnum);

		//Update once again the solution to make sure that vx and vxold are similar (for next step in transient or steadystate)
		InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,ug_horiz);
		delete ug_horiz;

		//save pointer to old velocity
		delete old_uf_horiz; old_uf_horiz=uf_horiz;

		/*solve: */
		femmodel->SystemMatricesx(&Kff_horiz, &Kfs_horiz, &pf_horiz, &df_horiz, NULL);
		CreateNodalConstraintsx(&ys,femmodel->nodes,configuration_type);
		Reduceloadx(pf_horiz, Kfs_horiz, ys); delete Kfs_horiz;
		Solverx(&uf_horiz, Kff_horiz, pf_horiz, old_uf_horiz, df_horiz,femmodel->parameters);
		Mergesolutionfromftogx(&ug_horiz, uf_horiz,ys,femmodel->nodes,femmodel->parameters); delete ys;
		InputUpdateFromSolutionx( femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,ug_horiz);

		convergence(&converged,Kff_horiz,pf_horiz,uf_horiz,old_uf_horiz,femmodel->parameters); delete Kff_horiz; delete pf_horiz; delete df_horiz;

		/*Second compute vertical velocity: */
		femmodel->SetCurrentConfiguration(DiagnosticVertAnalysisEnum);
		femmodel->parameters->FindParam(&configuration_type,ConfigurationTypeEnum);

		/*solve: */
		femmodel->SystemMatricesx(&Kff_vert, &Kfs_vert, &pf_vert,  &df_vert,NULL);
		CreateNodalConstraintsx(&ys,femmodel->nodes,configuration_type);
		Reduceloadx(pf_vert, Kfs_vert, ys); delete Kfs_vert;
		Solverx(&uf_vert, Kff_vert, pf_vert, NULL, df_vert,femmodel->parameters); delete Kff_vert; delete pf_vert; delete df_vert;
		Mergesolutionfromftogx(&ug_vert, uf_vert,ys,femmodel->nodes,femmodel->parameters);
		delete uf_vert; 
		delete ys; 
		InputUpdateFromSolutionx( femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,ug_vert);
		delete ug_vert;

		/*Increase count: */
		count++;
		if(converged==true)break;
		if(count>=max_nonlinear_iterations){
			_pprintLine_("   maximum number of iterations (" << max_nonlinear_iterations << ") exceeded"); 
			break;
		}
	}

	/*clean-up*/
	delete old_uf_horiz;
	delete uf_horiz;
	delete ug_horiz;
}
