/*!\file GiaDeflectionCorex
 * \brief: GIA solution from Erik Ivins. 
 * Compute deflection wi from a single disk of radius re, load history hes for 
 * numtimes time steps. 
 */

#include "./GiaDeflectionCorex.h"

#include "../../classes/classes.h"
#include "../../shared/shared.h"
#include "../../toolkits/toolkits.h"
#include "../../shared/Enum/Enum.h"
#include "../InputUpdateFromConstantx/InputUpdateFromConstantx.h"

#define Ntime 5 // number of times with load history 
#define Ntimm 4 // Ntime-1 : for slope/y-cept of load segments 
#define Ntimp 6 // Ntime+1 : for evaluation time  

/*External blocks: {{{*/
struct blockp{
	double pset[7];
};

struct blockt{
	double time[Ntimp];
	double bi[Ntimm];
	double dmi[Ntimm];
};

struct blocko{
	double rhoi;
	double hload[Ntime];
};

struct blocky{
	double zhload[Ntime];
};

struct blockn{
	int irate; 
};

struct blockrad{
	double distrad; 
};

struct blocks{
	double aswokm; 
	double asrpos; 
	double swok; 
};

extern "C" { 
	int distme_( int* pidisk,int* piedge);
	int what0_( int* pidisk,int* piedge);
	extern struct blockp blockp_;
	extern struct blockt blockt_;
	extern struct blocko blocko_;
	extern struct blocky blocky_;
	extern struct blockn blockn_;
	extern struct blockrad blockrad_;
	extern struct blocks blocks_;
}

/*}}}*/

void GiaDeflectionCorex( IssmDouble* pwi, IssmDouble* pdwidt, GiaDeflectionCoreArgs* arguments){

	/*output: */
	IssmDouble wi=0;
	IssmDouble dwidt=0;

	/*inputs: {{{*/
	/*constant: */
	int idisk=1; // disk #
	IssmDouble yts;

	/*coming from the model structure, runtime configurable:*/
	/*gia solution parameters: */
	int iedge=0; 
	int irate=0;

	/*gia inputs: */
	IssmDouble ri; //radial distance from center of disk to vertex  i
	IssmDouble re; //radius of disk
	IssmDouble* hes; //loading history (in ice thickness)
	IssmDouble* times; //loading history times
	int numtimes; //loading history length
	IssmDouble currenttime;

	/*gia material parameters: */
	IssmDouble lithosphere_shear_modulus;
	IssmDouble lithosphere_density;
	IssmDouble mantle_shear_modulus;
	IssmDouble mantle_viscosity;
	IssmDouble mantle_density;
	IssmDouble lithosphere_thickness;

	/*ice properties: */
	IssmDouble rho_ice;

	/*some debug info: */
	int disk_id;

/*}}}*/

	/*Recover material parameters and loading history: see GiaDeflectionCoreArgs for more details {{{*/
	ri=arguments->ri;
	re=arguments->re;
	hes=arguments->hes;
	times=arguments->times;
	numtimes=arguments->numtimes;
	currenttime=arguments->currenttime;
	lithosphere_shear_modulus=arguments->lithosphere_shear_modulus;
	lithosphere_density=arguments->lithosphere_density;
	mantle_shear_modulus=arguments->mantle_shear_modulus;
	mantle_viscosity=arguments->mantle_viscosity;
	mantle_density=arguments->mantle_density;
	lithosphere_thickness=arguments->lithosphere_thickness;
	rho_ice=arguments->rho_ice; 
	disk_id=arguments->idisk;
	irate=arguments->irate;
	iedge=arguments->iedge;
	yts=arguments->yts;

	/*}}}*/

	/*Modify inputs to match naruse code: */
	//from our model, irate comes in with values in [1,2], which maps into [0,1] in naruse: 
	irate=irate-1;

	/*Prepare block inputs for fortran distme and what0 routines of the naruse code: {{{*/
	/*Now, let's set pset from the data that we got in input to GiaDeflectionCorex: */
	blockp_.pset[0]=lithosphere_thickness;
	blockp_.pset[1]=mantle_viscosity;
	blockp_.pset[2]=lithosphere_shear_modulus;
	blockp_.pset[3]=mantle_shear_modulus;
	blockp_.pset[4]=lithosphere_density;
	blockp_.pset[5]=mantle_density;
	blockp_.pset[6]=re;
	blocko_.rhoi=rho_ice; 

	/*loading history: */
	blocky_.zhload[0]=hes[0];
	blocky_.zhload[1]=hes[1];
	blocky_.zhload[2]=hes[2];
	blocky_.zhload[3]=hes[3];
	blocky_.zhload[4]=hes[numtimes-1];

	/*times: */
	blockt_.time[0]=times[0]/1000.0/yts;    // in kyr
	blockt_.time[1]=times[1]/1000.0/yts;    // in kyr
	blockt_.time[2]=times[2]/1000.0/yts;    // in kyr
	blockt_.time[3]=times[3]/1000.0/yts;    // in kyr
	blockt_.time[4]=2500.0;                 // in kyr  
	blockt_.time[5]=times[numtimes-1]/1000.0/yts;  // evaluation time, in kyr 

	/*irate: */
	blockn_.irate=irate; 

	/*radial distance of i-th element: */
	blockrad_.distrad=ri/1000.0; // in km
	/*}}}*/

	/*Call distme driver: */
	distme_(&idisk,&iedge); 

	/*Call what0 driver: */
	what0_(&idisk,&iedge); 

	/*output solution: */
	if(irate==0){
		wi = blocks_.aswokm;
		*pwi=wi;
		*pdwidt=0;
	}
	else if (irate==1){
		*pwi=0;
		dwidt = blocks_.aswokm;
		*pdwidt=dwidt;
	}

}
