/*\file OptionParse.c
 *\brief: functions to parse the mex options.
 */
#ifdef HAVE_CONFIG_H
    #include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <cstring> 
#include "./matlabio.h"
#include "../../c/shared/shared.h"
#include "../../c/shared/io/io.h"

GenericOption<double>* OptionDoubleParse( char* name, const mxArray* prhs[]){ /*{{{*/

	GenericOption<double> *odouble = NULL;

	/*check and parse the name  */
	odouble=new GenericOption<double>();
	odouble->name =xNew<char>(strlen(name)+1);
	memcpy(odouble->name,name,(strlen(name)+1)*sizeof(char));
	FetchData(&odouble->value,prhs[0]);
	odouble->numel=1;
	odouble->ndims=1;
	odouble->size=NULL;

	return(odouble);
}/*}}}*/
GenericOption<double*>* OptionDoubleArrayParse( char* name, const mxArray* prhs[]){ /*{{{*/

	GenericOption<double*> *odouble = NULL;

	/*check and parse the name  */
	odouble=new GenericOption<double*>();
	odouble->name =xNew<char>(strlen(name)+1);
	memcpy(odouble->name,name,(strlen(name)+1)*sizeof(char));

	/*check and parse the value  */
	if (!mxIsClass(prhs[0],"double")){
		_error_("Value of option \"" << odouble->name  << "\" must be class \"double\", not class \"" << mxGetClassName(prhs[0]) <<"\".");
	}
	FetchData(&odouble->value,&odouble->numel,&odouble->ndims,&odouble->size,prhs[0]);

	return(odouble);
}/*}}}*/
GenericOption<bool*>* OptionLogicalParse( char* name, const mxArray* prhs[]){ /*{{{*/

	GenericOption<bool*> *ological = NULL;

	/*check and parse the name  */
	ological=new GenericOption<bool*>();
	ological->name =xNew<char>(strlen(name)+1);
	memcpy(ological->name,name,(strlen(name)+1)*sizeof(char));

	/*check and parse the value  */
	if (!mxIsClass(prhs[0],"logical")){
		_error_("Value of option \"" << ological->name  << "\" must be class \"logical\", not class \"" << mxGetClassName(prhs[0]) <<"\".");
	}
	FetchData(&ological->value,&ological->numel,&ological->ndims,&ological->size,prhs[0]);

	return(ological);
}/*}}}*/
GenericOption<char*>* OptionCharParse( char* name, const mxArray* prhs[]){ /*{{{*/

	GenericOption<char*>  *ochar = NULL;

	/*check and parse the name  */
	ochar=new GenericOption<char*>();
	ochar->name =xNew<char>(strlen(name)+1);
	memcpy(ochar->name,name,(strlen(name)+1)*sizeof(char));

	/*check and parse the value  */
	if (!mxIsClass(prhs[0],"char")){
		_error_("Value of option \"" << ochar->name  << "\" must be class \"char\", not class \"" << mxGetClassName(prhs[0]) <<"\".");
	}
	FetchData(&ochar->value,&ochar->numel,&ochar->ndims,&ochar->size,prhs[0]);

	return(ochar);
}/*}}}*/
GenericOption<Options**>* OptionStructParse( char* name, const mxArray* prhs[]){ /*{{{*/

	int            i;
	char           namei[161];
	Option*                   option      = NULL;
	GenericOption<Options**>  *ostruct    = NULL;
	const mwSize  *ipt        = NULL;
	const mxArray *structi;
	mwIndex        sindex;

	/*check and parse the name  */
	ostruct=new GenericOption<Options**>();
	ostruct->name =xNew<char>(strlen(name)+1);
	memcpy(ostruct->name,name,(strlen(name)+1)*sizeof(char));

	/*check and parse the value  */
	if (!mxIsClass(prhs[0],"struct")){
		_error_("Value of option \"" << ostruct->name  << "\" must be class \"struct\", not class \"" << mxGetClassName(prhs[0]) <<"\".");
	}
	ostruct->numel=mxGetNumberOfElements(prhs[0]);
	ostruct->ndims=mxGetNumberOfDimensions(prhs[0]);
	ipt           =mxGetDimensions(prhs[0]);
	ostruct->size =xNew<int>(ostruct->ndims);
	for (i=0; i<ostruct->ndims; i++) ostruct->size[i]=(int)ipt[i];
	if (ostruct->numel) ostruct->value=xNew<Options*>(ostruct->numel);

	/*loop through and process each element of the struct array  */
	for (sindex=0; sindex<ostruct->numel; sindex++) {
		ostruct->value[sindex]=new Options;

		/*loop through and process each field for the element  */
		for (i=0; i<mxGetNumberOfFields(prhs[0]); i++) {
			sprintf(namei,"%s.%s",name,mxGetFieldNameByNumber(prhs[0],i));
			structi=mxGetFieldByNumber(prhs[0],sindex,i);

			option=(Option*)OptionParse(namei,&structi);
			ostruct->value[sindex]->AddObject((Object*)option);
			option=NULL;
		}
	}

	return(ostruct);
}/*}}}*/
GenericOption<Options*>* OptionCellParse( char* name, const mxArray* prhs[]){ /*{{{*/

	int            i;
	int           *dims;
	char           namei[161];
	char           cstr[81];
	GenericOption<Options*> *ocell      = NULL;
	Option        *option     = NULL;
	const mwSize  *ipt        = NULL;
	const mxArray *celli;
	mwIndex        cindex;

	/*check and parse the name  */
	ocell=new GenericOption<Options*>();
	ocell->name =xNew<char>(strlen(name)+1);
	memcpy(ocell->name,name,(strlen(name)+1)*sizeof(char));

	/*check and parse the value  */
	if (!mxIsClass(prhs[0],"cell")){
		_error_("Value of option \"" << ocell->name  << "\" must be class \"cell\", not class \"" << mxGetClassName(prhs[0]) <<"\".");
	}

	ocell->numel=mxGetNumberOfElements(prhs[0]);
	ocell->ndims=mxGetNumberOfDimensions(prhs[0]);
	ipt         =mxGetDimensions(prhs[0]);
	ocell->size =xNew<int>(ocell->ndims);
	for (i=0; i<ocell->ndims; i++) ocell->size[i]=(int)ipt[i];
	ocell->value=new Options;

	/*loop through and process each element of the cell array  */
	dims=xNew<int>(ocell->ndims);
	for (cindex=0; cindex<ocell->numel; cindex++) {
		ColumnWiseDimsFromIndex(dims,(int)cindex,ocell->size,ocell->ndims);
		StringFromDims(cstr,dims,ocell->ndims);
		#ifdef _INTEL_WIN_
			_snprintf(namei,161,"%s%s",name,cstr);
		#else
			snprintf(namei,161,"%s%s",name,cstr);
		#endif
		celli=mxGetCell(prhs[0],cindex);

		option=(Option*)OptionParse(namei,&celli);
		ocell->value->AddObject((Object*)option);
		option=NULL;
	}
	xDelete<int>(dims);

	return(ocell);
}/*}}}*/
Option* OptionParse(char* name, const mxArray* prhs[]){ /*{{{*/

	Option  *option = NULL;
	mxArray *lhs[1];

	/*parse the value according to the matlab data type  */
	if     (mxIsClass(prhs[0],"double")  && (mxGetNumberOfElements(prhs[0])==1))
	 option=(Option*)OptionDoubleParse(name,prhs);
	else if(mxIsClass(prhs[0],"double")  && (mxGetNumberOfElements(prhs[0])!=1))
	 option=(Option*)OptionDoubleArrayParse(name,prhs);
	else if(mxIsClass(prhs[0],"logical"))
	 option=(Option*)OptionLogicalParse(name,prhs);
	else if(mxIsClass(prhs[0],"char"))
	 option=(Option*)OptionCharParse(name,prhs);
	else if(mxIsClass(prhs[0],"struct"))
	 option=(Option*)OptionStructParse(name,prhs);
	else if(mxIsClass(prhs[0],"cell"))
	 option=(Option*)OptionCellParse(name,prhs);
	else {
		_pprintLine_("  Converting value of option \"" << name << "\" from unrecognized class \"" << mxGetClassName(prhs[0]) << "\" to class \"" << "struct" << "\".");
		if (!mexCallMATLAB(1,lhs,1,(mxArray**)prhs,"struct")) {
			option=(Option*)OptionStructParse(name,(const mxArray**)lhs);
			mxDestroyArray(lhs[0]);
		}
		else _error_("Second argument value of option \""<< name <<"\" is of unrecognized class \""<< mxGetClassName(prhs[0]) <<"\".");
	}

	return(option);
}/*}}}*/
