/*!\file: solver_hydro_nonlinear.cpp
 * \brief: core of the hydro solution 
 */ 

#include "./solvers.h"
#include "../toolkits/toolkits.h"
#include "../classes/objects/objects.h"
#include "../io/io.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../modules/modules.h"

void solver_hydro_nonlinear(FemModel* femmodel){

	/*solution : */
	Vector<IssmDouble>* ug=NULL; 
	Vector<IssmDouble>* uf=NULL; 
	Vector<IssmDouble>* uf_old=NULL; 
	Vector<IssmDouble>* ys=NULL; 
	IssmDouble sediment_kmax;

	/*intermediary: */
	Matrix<IssmDouble>* Kff=NULL;
	Matrix<IssmDouble>* Kfs=NULL;
	Vector<IssmDouble>* pf=NULL;
	Vector<IssmDouble>* df=NULL;

	bool converged,isefficientlayer;
	int  constraints_converged;
	int  num_unstable_constraints;
	int  count;
	int  hydro_maxiter;

	/*Recover parameters: */
	femmodel->parameters->FindParam(&isefficientlayer,HydrologydcIsefficientlayerEnum);
	hydro_maxiter=100;
	count=1;
	converged=false;

	for(;;){

		/*First layer*/
		femmodel->SetCurrentConfiguration(HydrologyDCInefficientAnalysisEnum);
		InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,true,ResetPenaltiesEnum);
		InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,false,ConvergedEnum);
		femmodel->UpdateConstraintsx();
		femmodel->parameters->SetParam(HydrologySedimentEnum,HydrologyLayerEnum);
		for(;;){
			femmodel->SystemMatricesx(&Kff, &Kfs, &pf,&df, &sediment_kmax);
			CreateNodalConstraintsx(&ys,femmodel->nodes,HydrologyDCInefficientAnalysisEnum);
			Reduceloadx(pf,Kfs,ys); delete Kfs; delete uf;
			Solverx(&uf, Kff, pf,uf_old, df, femmodel->parameters);
			delete uf_old; uf_old=uf->Duplicate();
			delete Kff; delete pf; delete ug; delete df;
			Mergesolutionfromftogx(&ug,uf,ys,femmodel->nodes,femmodel->parameters); delete ys;
			InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,ug);

			ConstraintsStatex(&constraints_converged,&num_unstable_constraints, femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);

			if (!converged){
				if(VerboseConvergence()) _pprintLine_("   #unstable constraints = " << num_unstable_constraints);
				if(num_unstable_constraints==0) converged = true;
				if (count>=hydro_maxiter){
					_error_("   maximum number of iterations (" << hydro_maxiter << ") exceeded");
				}
			}
			count++;

			if(converged){
				InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,converged,ConvergedEnum);
				InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,sediment_kmax,HydrologySedimentKmaxEnum);
				InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,ug);
				break;
			}
		}

		/*Second layer*/
		if(!isefficientlayer) break;
		femmodel->SetCurrentConfiguration(HydrologyDCEfficientAnalysisEnum);
		InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,true,ResetPenaltiesEnum);
		InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,false,ConvergedEnum);
		femmodel->UpdateConstraintsx();
		femmodel->parameters->SetParam(HydrologyEfficientEnum,HydrologyLayerEnum);
		converged = false;
		for(;;){
			femmodel->SystemMatricesx(&Kff, &Kfs, &pf,&df,NULL);
			CreateNodalConstraintsx(&ys,femmodel->nodes,HydrologyDCEfficientAnalysisEnum);
			Reduceloadx(pf,Kfs,ys); delete Kfs; delete uf;
			Solverx(&uf, Kff, pf,uf_old, df, femmodel->parameters);
			delete uf_old; uf_old=uf->Duplicate();
			delete Kff;delete pf; delete ug; delete df;
			Mergesolutionfromftogx(&ug,uf,ys,femmodel->nodes,femmodel->parameters); delete ys;
			InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,ug);

			ConstraintsStatex(&constraints_converged,&num_unstable_constraints, femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);

			if (!converged){
				if(VerboseConvergence()) _pprintLine_("   #unstable constraints = " << num_unstable_constraints);
				if(num_unstable_constraints==0) converged = true;
				if (count>=hydro_maxiter){
					_error_("   maximum number of iterations (" << hydro_maxiter << ") exceeded");
				}
			}
			count++;

			if(converged){
				InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,converged,ConvergedEnum);
				InputUpdateFromConstantx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,sediment_kmax,MeltingOffsetEnum);
				InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,ug);
				break;
			}
		}

		/*System convergence check*/
		break;
	}

	InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,ug);

	/*Free ressources: */
	delete ug;
	delete uf;
	delete uf_old;
}
