/*!\file Bucket.h
 * \brief: header file for Bucket object
 */

#ifndef _BUCKET_H
#define _BUCKET_H

/*Headers:*/
/*{{{*/
#include "./Object.h"
#include "../../shared/Alloc/alloc.h"
#include "../../include/macros.h"
#include "../../Container/DataSet.h"
#include "../../toolkits/toolkitsenums.h"
/*}}}*/

/*how many MPI_Isend requests does it take to transfer the contents of a bucket to another cpu?*/
#define MATRIXBUCKETSIZEOFREQUESTS 7 
#define VECTORBUCKETSIZEOFREQUESTS 5 
typedef enum {VECTOR_BUCKET, MATRIX_BUCKET} BucketType;
template <class doubletype> class Bucket: public Object{

	private: 
		int type; //either a VECTOR_BUCKET or MATRIX_BUCKET
		int m,n; /*size of local matrix we are storing*/
		/*row and column indices of the matrix we are storing*/
		int* idxm;
		int* idxn; 
		doubletype* values; /*local matrix*/
		InsMode mode; /*mode of insertion for this bucket*/

	public: 
	
		/*constructors, destructors: */
		Bucket(){ /*{{{*/
			this->Initialize();
		} /*}}}*/
		Bucket(int min,int* idxmin,int nin,int* idxnin,doubletype* valuesin,InsMode modein){ /*{{{*/
			
			this->Initialize();

			this->type=MATRIX_BUCKET;
			this->m=min;
			this->n=nin;
			this->mode=modein;
			if(this->m){
				this->idxm=xNew<int>(this->m); 
				xMemCpy(this->idxm,idxmin,this->m);
			}
			if(this->n){
				this->idxn=xNew<int>(this->n); 
				xMemCpy(this->idxn,idxnin,this->n);
			}
			if(this->m*this->n){
				this->values=xNew<doubletype>(this->n*this->m);
				xMemCpy(this->values,valuesin,this->n*this->m);
			}
		} /*}}}*/
		Bucket(int min,int* idxmin,doubletype* valuesin,InsMode modein){ /*{{{*/ 
			this->Initialize();
			
			this->type=VECTOR_BUCKET; 
			this->m=min;
			this->mode=modein;
			if(this->m){
				this->idxm=xNew<int>(this->m); 
				xMemCpy(this->idxm,idxmin,this->m);
				
				this->values=xNew<doubletype>(this->m);
				xMemCpy(this->values,valuesin,this->m);
			}
		} /*}}}*/
		~Bucket(){ /*{{{*/
			xDelete<int>(idxm);
			xDelete<int>(idxn);
			xDelete<doubletype>(values);
		} /*}}}*/
		void Initialize(void){ /*{{{*/
			
			this->type=0;
			this->m=0;
			this->n=0;
			this->idxm=NULL;
			this->idxn=NULL;
			this->values=NULL;
			mode=INS_VAL;

		} /*}}}*/
		

		/*object virtual functions definitions:*/
		void    Echo(){ /*{{{*/
			_printLine_("Bucket echo (cpu #: "<<IssmComm::GetRank()<<")");
			_printLine_("bucket type: " << type);
			_printLine_("num rows: "<<this->m<<" num cols: "<<this->n);
		} /*}}}*/
		void    DeepEcho(){ /*{{{*/
			int i,j;

			_printLine_("Bucket echo (cpu #: "<<IssmComm::GetRank()<<")");
			_printLine_("bucket type: " << type);
			_printLine_("num rows: "<<this->m<<" num cols: "<<this->n);
			if(type=MATRIX_BUCKET){
				for (i=0;i<this->m;i++){
					_printLine_("row "<<this->idxm[i]<<", column indices: ");
					for (j=0;j<this->n;j++){
						_printLine_(" "<<this->idxn[j]);
					}
					_printLine_("values: ");
					for (j=0;j<this->n;j++){
						_printLine_(" "<<this->values[m*i+j]);
					}
				}
			}
			else if(type=VECTOR_BUCKET){
				for (i=0;i<this->m;i++){
					_printLine_("row "<<this->idxm[i]<<", value " << this->values[i]);
				}
			}
			else _error_("unknown type of bucket!");
		}
		/*}}}*/
		int     Id(){ /*{{{*/
			return -1;
		} /*}}}*/
		int     ObjectEnum(){ /*{{{*/
			return -1;
		} /*}}}*/
		Object *copy()        {/*{{{*/
			_error_("Not implemented yet (similar to Elements)"); };
		/*}}}*/
		
		/*specific routines of Bucket: */
		void SpawnBucketsPerCpu(DataSet* bucketsofcpu_i,int rank_i,int* rowranks){ /*{{{*/

			int i,j;

			/*go through our idxm index of rows this bucket owns, and spawn buckets  
			 *if these rows belong to cpu rank_i. Use rowranks to determine this.*/
			for(i=0;i<m;i++){
				if (rowranks[idxm[i]]==rank_i){
					/*This row belongs to cpu rank_i, so spawn a bucket with this row, and add it to the bucketsofcpu_i dataset: */
					if(type==MATRIX_BUCKET){
						bucketsofcpu_i->AddObject(new Bucket(1,idxm+i,n,idxn,values+n*i,mode));
					}
					else{
						bucketsofcpu_i->AddObject(new Bucket(1,idxm+i,values+i,mode));
					}
				}
			}

		};
		/*}}}*/
		void SetLocalMatrixValues(double* local_matrix,int lower_row,int global_N){ /*{{{*/

			int i,j;
			for(i=0;i<m;i++){
				for(j=0;j<n;j++){
					*(local_matrix+global_N*(idxm[i]-lower_row)+idxn[j])=*(values+n*i+j);
				}
			}
			
		};
		/*}}}*/
		void SetLocalVectorValues(double* local_vector,int lower_row){ /*{{{*/

			int i;
			for(i=0;i<m;i++){
				local_vector[idxm[i]-lower_row]=values[i];
			}
		};
		/*}}}*/
		int BucketType(void){ /*{{{*/

			return type;
		};
		/*}}}*/
#ifdef _HAVE_MPI_
			void Isend(int receiver_rank,MPI_Request* requests,int* pcount,MPI_Comm comm){ /*{{{*/
			int count=0;
			int int_mode;

			/*Recover pointer: */
			count=*pcount;

			/*Send all the information required: */
			MPI_Isend(&type,1,MPI_INT,receiver_rank,2,comm,requests+count); count++;
			MPI_Isend(&m,1,MPI_INT,receiver_rank,3,comm,requests+count); count++;
			if(m){ MPI_Isend(idxm,m,MPI_INT,receiver_rank,4,comm,requests+count); count++; }
			if(type==MATRIX_BUCKET){
				MPI_Isend(&n,1,MPI_INT,receiver_rank,5,comm,requests+count); count++;
				if(n){ MPI_Isend(idxn,n,MPI_INT,receiver_rank,6,comm,requests+count); count++; }
				if(m*n){ MPI_Isend(values,m*n,MPI_DOUBLE,receiver_rank,7,comm,requests+count); count++; }
			}
			else{
				if(m){ MPI_Isend(values,m,MPI_DOUBLE,receiver_rank,7,comm,requests+count); count++; }
			}
			int_mode=(int)mode;
			MPI_Isend(&int_mode,1,MPI_INT,receiver_rank,8,comm,requests+count); count++;

			/*Allocate pointers: */
			*pcount=count;

		} /*}}}*/
		void Recv(int sender_rank, MPI_Comm comm){ /*{{{*/

			MPI_Status status;
			int int_mode;

			MPI_Recv(&type,1, MPI_INT,sender_rank,2, comm, &status);
			MPI_Recv(&m,1, MPI_INT,sender_rank,3, comm, &status);
			if(m){
				idxm=new int[m];
				MPI_Recv(idxm,m, MPI_INT,sender_rank,4, comm, &status);
			}
			if(type==MATRIX_BUCKET){
				MPI_Recv(&n,1, MPI_INT,sender_rank,5, comm, &status);
				if(n){
					idxn=new int[n];
					MPI_Recv(idxn,n, MPI_INT,sender_rank,6, comm, &status);
				}
				if(m*n){
					values=new doubletype[m*n];
					MPI_Recv(values,m*n, MPI_DOUBLE,sender_rank,7, comm, &status);
				}
			}
			else{
				if(m){
					values=new doubletype[m];
					MPI_Recv(values,m, MPI_DOUBLE,sender_rank,7, comm, &status);
				}
			} 
			MPI_Recv(&int_mode,1, MPI_INT,sender_rank,8, comm, &status);
			mode=(InsMode)int_mode;

		} /*}}}*/
#endif
};

#endif  /* _BUCKET_H */
