%HYDROLOGYDC class definition
%
%   Usage:
%      hydrologydc=hydrologydc();

classdef hydrologydc
	properties (SetAccess=public) 
		spcsediment_head         = NaN;
		sediment_compressibility = 0;
		sediment_porosity        = 0;
		sediment_thickness       = 0;
		sediment_transmitivity   = 0;
		water_compressibility    = 0;
		
		spcepl_head         = NaN;
		epl_compressibility = 0;
		epl_porosity        = 0;
		epl_thickness       = 0;
		epl_transmitivity   = 0;
		isefficientlayer    = 0;
		
  end
	methods
		function obj = hydrologydc(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				otherwise
					error('constructor not supported');
	    end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{
			
		%Parameters from de Fleurian 2013
			obj.sediment_compressibility = 1.0e-08;
			obj.sediment_porosity        = .4;
			obj.sediment_thickness       = 20.0;
			obj.sediment_transmitivity   = 8.0e-04;
			obj.water_compressibility    = 5.04e-10;
			
			obj.epl_compressibility = 1.0e-08;
			obj.epl_porosity        = 0.4;
			obj.epl_thickness       = 1.0;
			obj.epl_transmitivity   = 8.0e-02;
			obj.isefficientlayer    = 1;
			
		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{
			
		%Early return
			if ~ismember(HydrologyDCInefficientAnalysisEnum(),analyses) & ~ismember(HydrologyDCEfficientAnalysisEnum(),analyses),
				return;
			end
			
			md = checkfield(md,'hydrology.spcsediment_head','forcing',1);
			md = checkfield(md,'hydrology.sediment_compressibility','>',0,'numel',1);
			md = checkfield(md,'hydrology.sediment_porosity','>',0,'numel',1);
			md = checkfield(md,'hydrology.sediment_thickness','>',0,'numel',1);
			md = checkfield(md,'hydrology.sediment_transmitivity','>',0,'numel',1);
			md = checkfield(md,'hydrology.water_compressibility','>',0,'numel',1);

			if obj.isefficientlayer==1,
				
				md = checkfield(md,'hydrology.spcepl_head','forcing',1);
				md = checkfield(md,'hydrology.epl_compressibility','>',0,'numel',1);
				md = checkfield(md,'hydrology.epl_porosity','>',0,'numel',1);
				md = checkfield(md,'hydrology.epl_thickness','>',0,'numel',1);
				md = checkfield(md,'hydrology.epl_transmitivity','>',0,'numel',1);
				md = checkfield(md,'hydrology.isefficientlayer','numel',[1],'values',[0 1]);
	    end
		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   hydrology Dual Porous Continuum Equivalent parameters:'));
			disp(sprintf('   - for the sediment layer'));
			
			fielddisplay(obj,'spcsediment_head','sediment water head constraints (NaN means no constraint) [m above MSL]');
			fielddisplay(obj,'sediment_compressibility','sediment compressibility [Pa^-1]');
			fielddisplay(obj,'sediment_porosity','sediment [dimensionless]');
			fielddisplay(obj,'sediment_thickness','sediment thickness [m]');
			fielddisplay(obj,'sediment_transmitivity','sediment transmitivity [m^2/s]');
			fielddisplay(obj,'water_compressibility','water compressibility [Pa^-1]');

			if obj.isefficientlayer==1,
				disp(sprintf('   - for the epl layer'));
				
				fielddisplay(obj,'spcepl_head','epl water head constraints (NaN means no constraint) [m above MSL]');
				fielddisplay(obj,'epl_compressibility','epl compressibility [Pa^-1]');
				fielddisplay(obj,'epl_porosity','epl [dimensionless]');
				fielddisplay(obj,'epl_thickness','epl thickness [m]');
				fielddisplay(obj,'epl_transmitivity','epl transmitivity [m^2/s]');
	    end
	 
		end % }}}
		function marshall(obj,fid) % {{{ 
			WriteData(fid,'enum',HydrologyEnum(),'data',HydrologydcEnum(),'format','Integer');
			WriteData(fid,'object',obj,'fieldname','spcsediment_head','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'fieldname','sediment_compressibility','format','Double');			
			WriteData(fid,'object',obj,'fieldname','sediment_porosity','format','Double');			
			WriteData(fid,'object',obj,'fieldname','sediment_thickness','format','Double');
			WriteData(fid,'object',obj,'fieldname','sediment_transmitivity','format','Double');			
			WriteData(fid,'object',obj,'fieldname','water_compressibility','format','Double');
			
			WriteData(fid,'object',obj,'fieldname','spcepl_head','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'fieldname','epl_compressibility','format','Double');			
			WriteData(fid,'object',obj,'fieldname','epl_porosity','format','Double');			
			WriteData(fid,'object',obj,'fieldname','epl_thickness','format','Double');
			WriteData(fid,'object',obj,'fieldname','epl_transmitivity','format','Double');
			WriteData(fid,'object',obj,'fieldname','isefficientlayer','format','Boolean');
			
		end % }}}
  end
end
