/*!\file GiaDeflectionCorex
 * \brief: GIA solution from Erik Ivins. 
 * Compute deflection wi from a single disk of radius re, load history hes for 
 * numtimes time steps. 
 */

#include "./GiaDeflectionCorex.h"

#include "../../classes/classes.h"
#include "../../shared/shared.h"
#include "../../include/include.h"
#include "../../toolkits/toolkits.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../InputUpdateFromConstantx/InputUpdateFromConstantx.h"

#define Nrsl 1
#define N3G 1
#define Nafter 1
#define Ntime 2
#define Ntimp 3

/*External blocks: {{{*/
struct blockp{
	double pset[7];
};
      
struct blockt{
	double time[Ntimp];
	double bi[Ntime];
	double dmi[Ntime];
};

struct blocko{
	double rhoi;
	double hload[Ntime];
};

struct blocky{
	double zhload[Ntime];
};
      
struct blockn{
	int irate; 
};

struct blockrad{
	double distrad; 
};

struct blocks{
	double aswokm; 
	double asrpos; 
	double swok; 
};

extern "C" { 
	int distme_( int* pidisk,int* piedge);
	int what0_( int* pidisk,int* piedge);
	extern struct blockp blockp_;
	extern struct blockt blockt_;
	extern struct blocko blocko_;
	extern struct blocky blocky_;
	extern struct blockn blockn_;
	extern struct blockrad blockrad_;
	extern struct blocks blocks_;
}

/*}}}*/
      
void GiaDeflectionCorex( IssmDouble* pwi, GiaDeflectionCoreArgs* arguments){

	/*output: */
	IssmDouble wi=1;
	
	/*inputs: */
	int iedge=1; //c iedge ......... = 1 square-edged, = 2 elliptical disc x-section  (see naruse.f)
	int idisk=1; // disk #
	int irate=0; // =0 fetch w solution (m) only; =1 dw/dt (mm/yr) only 

	/*intermediary: */

	/*inputs: {{{*/
	IssmDouble ri; //radial distance from center of disk to vertex  i
	IssmDouble re; //radius of disk
	IssmDouble current_he; //thickness at current time
	IssmDouble* hes; //loading history (in ice thickness)
	IssmDouble* times; //loading history times
	int numtimes; //loading history length
	IssmDouble currenttime;

	/*gia material parameters: */
	IssmDouble lithosphere_shear_modulus;
	IssmDouble lithosphere_density;
	IssmDouble mantle_shear_modulus;
	IssmDouble mantle_viscosity;
	IssmDouble mantle_density;
	IssmDouble lithosphere_thickness;

	/*ice properties: */
	IssmDouble rho_ice;

	/*some debug info: */
	int disk_id;

/*}}}*/
      
	/*Recover material parameters and loading history: see GiaDeflectionCoreArgs for more details {{{*/
	ri=arguments->ri;
	re=arguments->re;
	current_he=arguments->current_he;
	hes=arguments->hes;
	times=arguments->times;
	numtimes=arguments->numtimes;
	currenttime=arguments->currenttime;
	lithosphere_shear_modulus=arguments->lithosphere_shear_modulus;
	lithosphere_density=arguments->lithosphere_density;
	mantle_shear_modulus=arguments->mantle_shear_modulus;
	mantle_viscosity=arguments->mantle_viscosity;
	mantle_density=arguments->mantle_density;
	lithosphere_thickness=arguments->lithosphere_thickness;
	rho_ice=arguments->rho_ice; 
	disk_id=arguments->idisk;
	/*}}}*/

	// Now, let's set pset from the data that we got in input to GiaDeflectionCorex: 

	blockp_.pset[0]=lithosphere_thickness;
	blockp_.pset[1]=mantle_viscosity;
	blockp_.pset[2]=lithosphere_shear_modulus;
	blockp_.pset[3]=mantle_shear_modulus;
	blockp_.pset[4]=lithosphere_density;
	blockp_.pset[5]=mantle_density;
	blockp_.pset[6]=re;
//	blocko_.rhoi=rho_ice; // use this for non-benchmark experiments 
//	blockp_.pset[6]=800000.0; // for testing benchmark, should be in meters 
	blocko_.rhoi=1000.0;   // use this for benchmark not rho_ice = 917 kg m^-3
      
	/*loading history: */
	blocky_.zhload[0]=current_he;
	blocky_.zhload[1]=current_he;
//	blocky_.zhload[0]=3000; // for testing benchmark 
//	blocky_.zhload[1]=3000; // for testing benchmark  

	/*times: */
	blockt_.time[0]=1.e-4;    // in kyr
	blockt_.time[1]=2500e+0;  // in kyr
	blockt_.time[2]=2400e+0;  // control this for benchmark experiments

	/*irate: */
	blockn_.irate=irate; 

	/*radial distance of i-th element: */
	blockrad_.distrad=ri/1000.0; // in km
//	blockrad_.distrad=500.0; // for testing benchmark 

	/*Call distme driver: */
	distme_(&idisk,&iedge); 
	
	/*Call what0 driver: */
	what0_(&idisk,&iedge); 

	/*this is the solution: */ 
	wi = blocks_.aswokm;

	/*allocate output pointer: */
	*pwi=wi;

//	printf("wi: %g deflection: %g \n",wi,blocks_.aswokm); 

}
