/*!\file:  IssmMpiVec.h
 * \brief implementation of parallel dense ISSM vector. Internally, the parallel dense vector is 
 * split in rows across each cpu. Each vector (representing a subset of rows) on each cpu is fully 
 * dense, and is represented by a linear buffer of type doubletype. 
 * This object needs to answer the API defined by the virtual functions in IssmAbsVec, 
 * and the contructors required by IssmVec (see IssmVec.h)
 */ 

#ifndef _ISSM_MPI_VEC_H_
#define _ISSM_MPI_VEC_H_

/*Headers:*/
/*{{{*/
#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../../shared/Exceptions/exceptions.h"
#include "../../shared/MemOps/xMemCpy.h"
#include "../../shared/Alloc/alloc.h"
#include "../../include/macros.h"
#ifdef _HAVE_MPI_
#include "../mpi/mpiincludes.h"
#endif
#include <math.h>

/*}}}*/

/*We need to template this class, in case we want to create vectors that hold IssmDouble* vector or IssmPDouble* vector. 
  Such vectors would be useful for use without or with the matlab or python interface (which do not care for IssmDouble types, 
  but only rely on IssmPDouble types)*/
template <class doubletype> class IssmAbsVec;

template <class doubletype> 
class IssmMpiVec:public IssmAbsVec<doubletype>{

	public:

		int M; //global size
		int m; //local number of rows
		doubletype* vector;  /*here, doubletype is either IssmDouble or IssmPDouble*/
		DataSet*    buckets;  /*here, we store buckets of values that we will Assemble into a global vector.*/

		/*IssmMpiVec constructors, destructors*/
		/*FUNCTION IssmMpiVec(){{{*/
		IssmMpiVec(){

			this->M=0;
			this->m=0;
			this->vector=NULL;
			this->buckets=new DataSet();
		}
		/*}}}*/
		/*FUNCTION IssmMpiVec(int M){{{*/
		IssmMpiVec(int Min){
			this->Init(Min,false);
		}
		/*}}}*/
		/*FUNCTION IssmMpiVec(int M,bool fromlocalsize){{{*/
		IssmMpiVec(int Min, bool fromlocalsize){
			this->Init(Min,fromlocalsize);
		}
		/*}}}*/
		/*FUNCTION IssmMpiVec(doubletype* serial_vec,int M){{{*/
		IssmMpiVec(doubletype* buffer,int Min){

			this->Init(Min,false);

			if(this->M){
				this->vector=xNew<doubletype>(this->m);
				xMemCpy<doubletype>(this->vector,buffer,this->m);
			}
		}
		/*}}}*/
		/*FUNCTION IssmMpiVec::Init(int Min,bool fromlocalsize){{{*/
		void Init(int Min,bool fromlocalsize){

			this->buckets=new DataSet();
			
			if(fromlocalsize){
				this->m=Min;
				this->M=DetermineGlobalSize(this->m,IssmComm::GetComm());
			}
			else{
				this->M=Min;
				this->m=DetermineLocalSize(this->M,IssmComm::GetComm());
			}
			
			/*Initialize pointer: */
			this->vector=NULL;

			/*Allocate: */
			if (m)this->vector=xNewZeroInit<doubletype>(this->m);
		}
		/*}}}*/
		/*FUNCTION ~IssmMpiVec(){{{*/
		~IssmMpiVec(){
			xDelete<doubletype>(this->vector);
			this->M=0;
			this->n=0;
			delete buckets;
		}
		/*}}}*/

		/*IssmMpiVec specific routines*/
		/*FUNCTION Echo{{{*/
		void Echo(void){
			
			int i,j;

			/*Do a synchronized dump across all the rows: */
			for(i=0;i<IssmComm::GetSize();i++){
				if (IssmComm::GetRank()==i){
					printf("cpu %i #rows: %i\n",i,this->m);
					for (j=0;j<this->m;j++){
						printf("row %i %g",j,this->vector[j]);
					}
				}
				MPI_Barrier(IssmComm::GetComm());
			}
		}
		/*}}}*/
		/*FUNCTION Assemble{{{*/
		void Assemble(void){
			printf("not imlemented yet!");
			exit(1);
		}
		/*}}}*/
		/*FUNCTION SetValues{{{*/
		void SetValues(int ssize, int* list, doubletype* values, InsMode mode){
			
			/*we need to store all the values we collect here in order to Assemble later. 
			 * Indeed, the values we are collecting here most of the time will not belong 
			 * to us, but to another part of the vector on another cpu: */
			_assert_(buckets);

			buckets->AddObject(new Bucket<doubletype>(ssize, list, values, mode));

		}
		/*}}}*/
		/*FUNCTION SetValue{{{*/
		void SetValue(int dof, doubletype value, InsMode mode){

			/*we need to store the value we collect here in order to Assemble later. 
			 * Indeed, the value we are collecting here most of the time will not belong 
			 * to us, but to another part of the vector on another cpu: */
			_assert_(buckets);

			buckets->AddObject(new Bucket<doubletype>(1,&dof,&value, mode));
		}
		/*}}}*/
		/*FUNCTION GetValue{{{*/
		void GetValue(doubletype* pvalue,int dof){
			_error_("Get value on a MpiVec vector not implemented yet!");
		}
		/*}}}*/
		/*FUNCTION GetSize{{{*/
		void GetSize(int* pM){

			*pM=this->M;

		}
		/*}}}*/
		/*FUNCTION GetLocalSize{{{*/
		void GetLocalSize(int* pM){

			*pM=this->m;

		}
		/*}}}*/
		/*FUNCTION Duplicate{{{*/
		IssmMpiVec<doubletype>* Duplicate(void){

			return new IssmMpiVec<doubletype>(this->vector,this->M);

		}
		/*}}}*/
		/*FUNCTION Set{{{*/
		void Set(doubletype value){

			int i;
			for(i=0;i<this->m;i++)this->vector[i]=value;

		}
		/*}}}*/
		/*FUNCTION AXPY{{{*/
		void AXPY(IssmAbsVec<doubletype>* Xin, doubletype a){
			printf("AXPY not implemented yet!");
			exit(1);
		}
		/*}}}*/
		/*FUNCTION AYPX{{{*/
		void AYPX(IssmAbsVec<doubletype>* Xin, doubletype a){
			printf("AYPX not implemented yet!");
			exit(1);
		}
		/*}}}*/
		/*FUNCTION ToMPISerial{{{*/
		doubletype* ToMPISerial(void){
			
			printf("IssmMpiVec ToMPISerial not implemented yet!");
			exit(1);

		}
		/*}}}*/
		/*FUNCTION Copy{{{*/
		void Copy(IssmAbsVec<doubletype>* toin){

			int i;

			/*Assume toin is of the correct type, and downcast: */
			IssmMpiVec* to=NULL;

			to=(IssmMpiVec<doubletype>*)toin;

			to->M=this->M;
			for(i=0;i<this->m;i++)to->vector[i]=this->vector[i];

		}
		/*}}}*/
		/*FUNCTION Norm{{{*/
		doubletype Norm(NormMode mode){

			printf("IssmMpiVec Norm not implemented yet!");
			exit(1);
		}
		/*}}}*/
		/*FUNCTION Scale{{{*/
		void Scale(doubletype scale_factor){

			int i;
			for(i=0;i<this->M;i++)this->vector[i]=scale_factor*this->vector[i];

		}
		/*}}}*/
		/*FUNCTION Dot{{{*/
		doubletype Dot(IssmAbsVec<doubletype>* inputin){

			int i;
			doubletype local_dot=0;
			doubletype dot=0;

			/*Assume inputin is of the correct type, and downcast: */
			IssmMpiVec* input=NULL;

			input=(IssmMpiVec<doubletype>*)inputin;

			for(i=0;i<this->m;i++)local_dot+=this->vector[i]*input->vector[i];

			#ifdef _HAVE_MPI_
			/*MPI_SUM all the dots across the cluster: */
			MPI_Reduce(&local_dot, &dot, 1, MPI_DOUBLE, MPI_SUM, 0, IssmComm::GetComm());
			MPI_Bcast(&dot,1,MPI_DOUBLE,0,IssmComm::GetComm());
			#endif

			return dot;
		}
		/*}}}*/
		/*FUNCTION PointwiseDivide{{{*/
		void PointwiseDivide(IssmAbsVec<doubletype>* xin,IssmAbsVec<doubletype>* yin){

			int i;
			
			/*Assume xin and yin are of the correct type, and downcast: */
			IssmMpiVec* x=NULL;
			IssmMpiVec* y=NULL;

			x=(IssmMpiVec<doubletype>*)xin;
			y=(IssmMpiVec<doubletype>*)yin;


			/*pointwise w=x/y where this->vector is w: */
			for(i=0;i<this->m;i++)this->vector[i]=x->vector[i]/y->vector[i];
		}
		/*}}}*/
};
#endif //#ifndef _ISSM_MPI_VEC_H_
