/*!\file:  IssmMpiDenseMat.h
 * \brief implementation of parallel dense ISSM matrix. Internally, the parallel dense matrix is 
 * split in rows across each cpu. Each matrix (representing a subset of rows) on each cpu is fully 
 * dense, and is represented by a linear buffer of type doubletype. 
 * This object needs to answer the API defined by the virtual functions in IssmAbsMat, 
 * and the contructors required by IssmMat (see IssmMat.h)
 */ 

#ifndef _ISSM_MPI_DENSE_MAT_H_
#define _ISSM_MPI_DENSE_MAT_H_

/*Headers:*/
/*{{{*/
#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../../shared/Exceptions/exceptions.h"
#include "../../shared/MemOps/xMemCpy.h"
#include "../../shared/Alloc/alloc.h"
#include "../../include/macros.h"
#include "../../Container/DataSet.h"
#include "../../classes/IssmComm.h"
#include "../../classes/objects/Bucket.h"
#include <math.h>

/*}}}*/

/*We need to template this class, in case we want to create Matrices that hold
  IssmDouble* matrix or IssmPDouble* matrix. 
  Such matrices would be useful for use without or with the matlab or python
  interface (which do not care for IssmDouble types, but only rely on
  IssmPDouble types)*/
template <class doubletype> class IssmAbsMat;

template <class doubletype> 
class IssmMpiDenseMat:public IssmAbsMat<doubletype>{

	public:

		int M,N;  //global size
		int m;    //local number of rows
		doubletype* matrix;  /*here, doubletype is either IssmDouble or IssmPDouble*/
		DataSet*    buckets;  /*here, we store buckets of values that we will Assemble into a global matrix.*/

		/*IssmMpiDenseMat constructors, destructors*/
		/*FUNCTION IssmMpiDenseMat(){{{*/
		IssmMpiDenseMat(){
			this->M=0;
			this->N=0;
			this->m=0;
			this->matrix=NULL;
			this->buckets=new DataSet();
		}
		/*}}}*/
		/*FUNCTION IssmMpiDenseMat(int M,int N){{{*/
		IssmMpiDenseMat(int Min,int Nin){
			this->Init(Min,Nin);
		}
		/*}}}*/
		/*FUNCTION IssmMpiDenseMat(int M,int N, doubletype sparsity){{{*/
		IssmMpiDenseMat(int pM,int pN, doubletype sparsity){
			/*no sparsity involved here, we are fully dense, so just use the previous constructor: */
			this->Init(pM,pN);
		}
		/*}}}*/
		/*FUNCTION IssmMpiDenseMat(int m,int n,int M,int N,int* d_nnz,int* o_nnz){{{*/
		IssmMpiDenseMat(int m,int n,int pM,int pN,int* d_nnz,int* o_nnz){
			/*not needed, we are fully dense!: */
			this->Init(pM,pN);
		}
		/*}}}*/
		/*FUNCTION IssmMpiDenseMat(doubletype* serial_mat,int M,int N,doubletype sparsity){{{*/
		IssmMpiDenseMat(doubletype* serial_mat,int Min,int Nin,doubletype sparsity){
			
			/*Here, we assume that the serial_mat is local to the local cpu, and that it has 
			 * the correct size (m rows by N colums), n determined by DetermineLocalSize: */
			this->buckets=new DataSet();
			this->M=Min;
			this->N=Nin;
			this->m=DetermineLocalSize(this->M,IssmComm::GetComm());

			this->matrix=NULL;
			if(m*N){
				this->matrix=xNewZeroInit<doubletype>(m*N);
				xMemCpy<doubletype>(this->matrix,serial_mat,m*N);
			}
		}
		/*}}}*/
		/*FUNCTION IssmMpiDenseMat(int M,int N, int connectivity, int numberofdofspernode){{{*/
		IssmMpiDenseMat(int pM,int pN, int connectivity,int numberofdofspernode){
			/*not needed, we are fully dense!: */
			this->Init(pM,pN);
		}
		/*}}}*/
		/*FUNCTION ~IssmMpiDenseMat(){{{*/
		~IssmMpiDenseMat(){
			xDelete<doubletype>(this->matrix);
			M=0;
			N=0;
			m=0;
			delete this->buckets;
		}
		/*}}}*/
		/*FUNCTION IssmMpiDenseMat::Init(int Min,int Nin){{{*/
		void Init(int Min,int Nin){

			this->buckets=new DataSet();
			
			this->M=Min;
			this->N=Nin;
			
			/*Figure out local number of rows: */
			this->m=DetermineLocalSize(this->M,IssmComm::GetComm());
			
			/*Initialize pointer: */
			this->matrix=NULL;

			/*Allocate: */
			if (m*N)this->matrix=xNewZeroInit<doubletype>(this->m*N);
		}
		/*}}}*/

		/*IssmMpiDenseMat specific routines */
		/*FUNCTION Echo{{{*/
		void Echo(void){

			int my_rank;
			int i,j,k;

			/*Do a synchronized dump across all the rows: */
			my_rank=IssmComm::GetRank();
			for(i=0;i<IssmComm::GetSize();i++){
				if (my_rank==i){
					printf("cpu %i #rows: %i\n",i,this->m);
					for (j=0;j<this->m;j++){
						printf("row %i ",j);
						for (k=0;k<this->N;k++){
							printf("%g ",this->matrix[j*this->N+k]);
						}
					}
				}
				MPI_Barrier(IssmComm::GetComm());
			}

		}
		/*}}}*/
		/*FUNCTION Assemble{{{*/
		void Assemble(void){
			_error_("not supported yet!");
		}
		/*}}}*/
		/*FUNCTION Norm{{{*/
		doubletype Norm(NormMode mode){
			_error_("not supported yet!");
		}
		/*}}}*/
		/*FUNCTION GetSize{{{*/
		void GetSize(int* pM,int* pN){
			_error_("not supported yet!");
		}
		/*}}}*/
		/*FUNCTION GetLocalSize{{{*/
		void GetLocalSize(int* pM,int* pN){
			_error_("not supported yet!");
		}
		/*}}}*/
		/*FUNCTION MatMult{{{*/
		void MatMult(IssmAbsVec<doubletype>* Xin,IssmAbsVec<doubletype>* AXin){
			_error_("not supported yet!");
		}
		/*}}}*/
		/*FUNCTION Duplicate{{{*/
		IssmMpiDenseMat<doubletype>* Duplicate(void){
			_error_("not supported yet!");
		}
		/*}}}*/
		/*FUNCTION ToSerial{{{*/
		doubletype* ToSerial(void){
			_error_("not supported yet!");
		}
		/*}}}*/
		/*FUNCTION SetValues{{{*/
		void SetValues(int min,int* idxm,int nin,int* idxn,doubletype* values,InsMode mode){

			/*we need to store all the values we collect here in order to Assemble later. 
			 * Indeed, the values we are collecting here most of the time will not belong 
			 * to us, but to another part of the matrix on another cpu: */
			_assert_(buckets);

			buckets->AddObject(new Bucket<doubletype>(min,idxm,nin,idxn,values,mode));

		}
		/*}}}*/
		/*FUNCTION Convert{{{*/
		void Convert(MatrixType type){
			/*do nothing: */
		}
		/*}}}*/		

};

#endif //#ifndef _ISSM_MPI_DENSE_MAT_H_
