/*!\file:  IssmMpiVec.h
 * \brief implementation of parallel dense ISSM vector. Internally, the parallel dense vector is 
 * split in rows across each cpu. Each vector (representing a subset of rows) on each cpu is fully 
 * dense, and is represented by a linear buffer of type doubletype. 
 * This object needs to answer the API defined by the virtual functions in IssmAbsVec, 
 * and the contructors required by IssmVec (see IssmVec.h)
 */ 

#ifndef _ISSM_MPI_VEC_H_
#define _ISSM_MPI_VEC_H_

/*Headers:*/
/*{{{*/
#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../../shared/Exceptions/exceptions.h"
#include "../../shared/MemOps/xMemCpy.h"
#include "../../shared/Alloc/alloc.h"
#include "../../include/macros.h"
#include <math.h>

/*}}}*/

/*We need to template this class, in case we want to create vectors that hold IssmDouble* vector or IssmPDouble* vector. 
  Such vectors would be useful for use without or with the matlab or python interface (which do not care for IssmDouble types, 
  but only rely on IssmPDouble types)*/
template <class doubletype> class IssmAbsVec;

template <class doubletype> 
class IssmMpiVec:public IssmAbsVec<doubletype>{

	public:

		doubletype* vector;
		int M;

		/*IssmMpiVec constructors, destructors*/
		/*FUNCTION IssmMpiVec(){{{*/
		IssmMpiVec(){

			this->M=0;
			this->vector=NULL;
		}
		/*}}}*/
		/*FUNCTION IssmMpiVec(int M){{{*/
		IssmMpiVec(int pM){

			this->M=pM;
			this->vector=NULL;
			if(this->M) this->vector=xNewZeroInit<doubletype>(pM);
		}
		/*}}}*/
		/*FUNCTION IssmMpiVec(int m,int M){{{*/
		IssmMpiVec(int pm,int pM){

			this->M=pM;
			this->vector=NULL;
			if(this->M) this->vector=xNewZeroInit<doubletype>(pM);
		}
		/*}}}*/
		/*FUNCTION IssmMpiVec(int M,bool fromlocalsize){{{*/
		IssmMpiVec(int pM,bool fromlocalsize){

			this->M=pM;
			this->vector=NULL;
			if(this->M) this->vector=xNewZeroInit<doubletype>(pM);
		}
		/*}}}*/
		/*FUNCTION IssmMpiVec(doubletype* serial_vec,int M){{{*/
		IssmMpiVec(doubletype* buffer,int pM){

			this->M=pM;
			this->vector=NULL;
			if(this->M){
				this->vector=xNew<doubletype>(pM);
				xMemCpy<doubletype>(this->vector,buffer,pM);
			}
		}
		/*}}}*/
		/*FUNCTION ~IssmMpiVec(){{{*/
		~IssmMpiVec(){
			xDelete<doubletype>(this->vector);
			M=0;
		}
		/*}}}*/

		/*IssmMpiVec specific routines*/
		/*FUNCTION Echo{{{*/
		void Echo(void){

			int i;
			_printLine_("IssmMpiVec size " << this->M);
			for(i=0;i<M;i++){
				_printString_(vector[i] << "\n ");
			}
		}
		/*}}}*/
		/*FUNCTION Assemble{{{*/
		void Assemble(void){

			/*do nothing*/

		}
		/*}}}*/
		/*FUNCTION SetValues{{{*/
		void SetValues(int ssize, int* list, doubletype* values, InsMode mode){

			int i;
			switch(mode){
				case ADD_VAL:
					for(i=0;i<ssize;i++) this->vector[list[i]]+=values[i];
					break;
				case INS_VAL:
					for(i=0;i<ssize;i++) this->vector[list[i]]=values[i];
					break;
				default:
					_error_("unknown insert mode!");
					break;
			}

		}
		/*}}}*/
		/*FUNCTION SetValue{{{*/
		void SetValue(int dof, doubletype value, InsMode mode){

			switch(mode){
				case ADD_VAL:
					this->vector[dof]+=value;
					break;
				case INS_VAL:
					this->vector[dof]=value;
					break;
				default:
					_error_("unknown insert mode!");
					break;
			}
		}
		/*}}}*/
		/*FUNCTION GetValue{{{*/
		void GetValue(doubletype* pvalue,int dof){

			*pvalue=this->vector[dof];

		}
		/*}}}*/
		/*FUNCTION GetSize{{{*/
		void GetSize(int* pM){

			*pM=this->M;

		}
		/*}}}*/
		/*FUNCTION GetLocalSize{{{*/
		void GetLocalSize(int* pM){

			*pM=this->M;

		}
		/*}}}*/
		/*FUNCTION Duplicate{{{*/
		IssmMpiVec<doubletype>* Duplicate(void){

			return new IssmMpiVec<doubletype>(this->vector,this->M);

		}
		/*}}}*/
		/*FUNCTION Set{{{*/
		void Set(doubletype value){

			int i;
			for(i=0;i<this->M;i++)this->vector[i]=value;

		}
		/*}}}*/
		/*FUNCTION AXPY{{{*/
		void AXPY(IssmAbsVec<doubletype>* Xin, doubletype a){

			int i;
			
			/*Assume X is of the correct type, and downcast: */
			IssmMpiVec* X=NULL;

			X=(IssmMpiVec<doubletype>*)Xin;


			/*y=a*x+y where this->vector is y*/
			for(i=0;i<this->M;i++)this->vector[i]=a*X->vector[i]+this->vector[i];

		}
		/*}}}*/
		/*FUNCTION AYPX{{{*/
		void AYPX(IssmAbsVec<doubletype>* Xin, doubletype a){

			int i;

			/*Assume X is of the correct type, and downcast: */
			IssmMpiVec* X=NULL;

			X=(IssmMpiVec<doubletype>*)Xin;

			/*y=x+a*y where this->vector is y*/
			for(i=0;i<this->M;i++)this->vector[i]=X->vector[i]+a*this->vector[i];

		}
		/*}}}*/
		/*FUNCTION ToMPISerial{{{*/
		doubletype* ToMPISerial(void){

			doubletype* buffer=NULL;

			if(this->M){
				buffer=xNew<doubletype>(this->M);
				xMemCpy<doubletype>(buffer,this->vector,this->M);
			}
			return buffer;

		}
		/*}}}*/
		/*FUNCTION Copy{{{*/
		void Copy(IssmAbsVec<doubletype>* toin){

			int i;

			/*Assume toin is of the correct type, and downcast: */
			IssmMpiVec* to=NULL;

			to=(IssmMpiVec<doubletype>*)toin;

			to->M=this->M;
			for(i=0;i<this->M;i++)to->vector[i]=this->vector[i];

		}
		/*}}}*/
		/*FUNCTION Norm{{{*/
		doubletype Norm(NormMode mode){

			doubletype norm;
			int i;

			switch(mode){
				case NORM_INF:
					norm=0; for(i=0;i<this->M;i++)norm=max(norm,fabs(this->vector[i]));
					return norm;
					break;
				case NORM_TWO:
					norm=0; 
					for(i=0;i<this->M;i++)norm+=pow(this->vector[i],2);
					return sqrt(norm);
					break;
				default:
					_error_("unknown norm !");
					break;
			}
		}
		/*}}}*/
		/*FUNCTION Scale{{{*/
		void Scale(doubletype scale_factor){

			int i;
			for(i=0;i<this->M;i++)this->vector[i]=scale_factor*this->vector[i];

		}
		/*}}}*/
		/*FU		/*FUNCTION Dot{{{*/
		doubletype Dot(IssmAbsVec<doubletype>* inputin){

			int i;

			/*Assume inputin is of the correct type, and downcast: */
			IssmMpiVec* input=NULL;

			input=(IssmMpiVec<doubletype>*)inputin;


			doubletype dot=0;
			for(i=0;i<this->M;i++)dot+=this->vector[i]*input->vector[i];
			return dot;

		}
		/*}}}*/
		/*FUNCTION PointwiseDivide{{{*/
		void PointwiseDivide(IssmAbsVec<doubletype>* xin,IssmAbsVec<doubletype>* yin){

			int i;
			
			/*Assume xin and yin are of the correct type, and downcast: */
			IssmMpiVec* x=NULL;
			IssmMpiVec* y=NULL;

			x=(IssmMpiVec<doubletype>*)xin;
			y=(IssmMpiVec<doubletype>*)yin;


			/*pointwise w=x/y where this->vector is w: */
			for(i=0;i<this->M;i++)this->vector[i]=x->vector[i]/y->vector[i];
		}
		/*}}}*/
};
#endif //#ifndef _ISSM_MPI_VEC_H_
