%TIMESTEPPING Class definition
%
%   Usage:
%      timestepping=timestepping();

classdef timestepping
	properties (SetAccess=public) 
		start_time      = 0.;
		final_time      = 0.;
		time_step       = 0.;
		time_adapt      = 0;
		cfl_coefficient = 0.;
	end
	methods
		function obj = timestepping(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%time between 2 time steps
			obj.time_step=1./2.;

			%final time
			obj.final_time=10.*obj.time_step;

			%time adaptation? 
			obj.time_adapt=0;
			obj.cfl_coefficient=0.5;
		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{

			md = checkfield(md,'timestepping.start_time','numel',[1],'NaN',1);
			md = checkfield(md,'timestepping.final_time','numel',[1],'NaN',1);
			md = checkfield(md,'timestepping.time_step','numel',[1],'>=',0,'NaN',1);
			md = checkfield(md,'timestepping.time_adapt','numel',[1],'values',[0 1]);
			md = checkfield(md,'timestepping.cfl_coefficient','numel',[1],'>',0,'<=',1);
			if obj.final_time-obj.start_time<0,
				md = checkmessage(md,'timestepping.final_time should be larger than timestepping.start_time');
			end 
		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   timestepping parameters:'));

			fielddisplay(obj,'start_time','simulation starting time [yr]');
			fielddisplay(obj,'final_time','final time to stop the simulation [yr]');
			fielddisplay(obj,'time_step','length of time steps [yr]');
			fielddisplay(obj,'time_adapt','use cfl condition to define time step ? (0 or 1) ');
			fielddisplay(obj,'cfl_coefficient','coefficient applied to cfl condition');

		end % }}}
		function marshall(obj,fid) % {{{
			WriteData(fid,'object',obj,'fieldname','start_time','format','Double');
			WriteData(fid,'object',obj,'fieldname','final_time','format','Double');
			WriteData(fid,'object',obj,'fieldname','time_step','format','Double');
			WriteData(fid,'object',obj,'fieldname','time_adapt','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','cfl_coefficient','format','Double');
		end % }}}
	end
end
