%PROGNOSTIC class definition
%
%   Usage:
%      prognostic=prognostic();

classdef prognostic
	properties (SetAccess=public) 
		 spcthickness           = NaN;
		 min_thickness          = 0;
		 hydrostatic_adjustment = 0;
		 stabilization          = 0;
		 vertex_pairing         = NaN;
		 penalty_factor         = 0;
		 requested_outputs      = NaN;
	end
	methods
		function obj = prognostic(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%Type of stabilization to use 0:nothing 1:artificial_diffusivity 3:Discontinuous Galerkin
			obj.stabilization=1;

			%Factor applied to compute the penalties kappa=max(stiffness matrix)*10^penalty_factor
			obj.penalty_factor=3;

			%Minimum ice thickness that can be used
			obj.min_thickness=1;

			%Hydrostatic adjustment
			obj.hydrostatic_adjustment='Absolute';
		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{

			%Early return
			if ~ismember(PrognosticAnalysisEnum(),analyses) |  (solution==TransientSolutionEnum() & md.transient.isprognostic==0), return; end

			md = checkfield(md,'prognostic.spcthickness','forcing',1);
			md = checkfield(md,'prognostic.hydrostatic_adjustment','values',{'Absolute' 'Incremental'});
			md = checkfield(md,'prognostic.stabilization','values',[0 1 2 3]);
			md = checkfield(md,'prognostic.min_thickness','>',0);
			if ~isempty(md.prognostic.requested_outputs),
				md = checkfield(md,'prognostic.requested_outputs','size',[NaN 1]);
			end

		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   Prognostic solution parameters:'));
			fielddisplay(obj,'spcthickness','thickness constraints (NaN means no constraint) [m]');
			fielddisplay(obj,'min_thickness','minimum ice thickness allowed [m]');
			fielddisplay(obj,'hydrostatic_adjustment','adjustment of ice shelves surface and bed elevations: ''Incremental'' or ''Absolute'' ');
			fielddisplay(obj,'stabilization','0: no, 1: artificial_diffusivity, 2: streamline upwinding, 3: discontinuous Galerkin');

			disp(sprintf('\n      %s','Penalty options:'));
			fielddisplay(obj,'penalty_factor','offset used by penalties: penalty = Kmax*10^offset');
			fielddisplay(obj,'vertex_pairing','pairs of vertices that are penalized');
			fielddisplay(obj,'requested_outputs','additional outputs requested');

		end % }}}
		function marshall(obj,fid) % {{{
			WriteData(fid,'object',obj,'fieldname','spcthickness','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'fieldname','min_thickness','format','Double');
			WriteData(fid,'data',StringToEnum(obj.hydrostatic_adjustment),'format','Integer','enum',PrognosticHydrostaticAdjustmentEnum());
			WriteData(fid,'object',obj,'fieldname','stabilization','format','Integer');
			WriteData(fid,'object',obj,'fieldname','vertex_pairing','format','DoubleMat','mattype',3);
			WriteData(fid,'object',obj,'fieldname','penalty_factor','format','Double');
			WriteData(fid,'object',obj,'fieldname','requested_outputs','format','DoubleMat','mattype',3);
		end % }}}
	end
end
