function manualcb(zmin,zmax,cmap,varargin)
%MANUALCB - custom colorbar
%
%   Usage:
%      manualcb(min,max,colormap,options)
%
%   Available options:
%      - 'fontsize'    : default is 12
%      - 'smallbars'   : bars next to each tick (default is false)
%      - 'position'    : colorbar position in normalized units
%      - 'orientation' : 'vertical' (default) or 'horizontal'
%      - 'title'       : colorbar title

%check inputs
if nargin<3,
	help manualcb
	error('bad usage');
end
if zmin>zmax,
	error('zmin should be smaller than zmax');
end

%process options
options = pairoptions(varargin{:});
fontsize  = getfieldvalue(options,'fontsize',12);
smallbars = getfieldvalue(options,'smallbars',false);

%Colorbar position
if ~exist(options,'position'),
	position = plotboxpos;
	xstart   = position(1)+position(3)+0.01;
	ystart   = position(2);
	width    = .02;
	height   = position(4);
else
	position = getfieldvalue(options,'position');
	xstart = position(1);
	ystart = position(2);
	width  = position(3);
	height = position(4);
end
axes('Units','normalized','Position',[xstart ystart width height],'XTickLabel','','YTickLabel','','Visible','on');
xlim([0 1]);
ylim([0 1]);

%Prepare tickes
deltaz = dtick(zmax-zmin);
ztick  = (deltaz*ceil(zmin/deltaz)):deltaz:zmax;
ytick  = (ztick-zmin)/(zmax-zmin);

%Discolorbar position in normalized unitsplay colorbar
hold on
if strcmpi(getfieldvalue(options,'orientation','vertical'),'vertical'),
	image_rgb = ind2rgb(repmat(uint16(1:length(cmap))',1,2),cmap);
else
	image_rgb = ind2rgb(repmat(uint16(1:length(cmap))',1,2)',cmap);
end
imagesc([0 1],[0 1],image_rgb);
patch([0,0,1,1],[0,1,1,0],'k','FaceColor','none','Clipping','off')

%Add ticks
if strcmpi(getfieldvalue(options,'orientation','vertical'),'vertical'),
	text(1.5+zeros(size(ztick)),ytick,num2str(ztick'),'HorizontalAlignment','left','VerticalAlignment','middle','FontSize',fontsize)
	if smallbars,
		for i=1:numel(ztick)
			patch([0.8 1.0],[ytick(i) ytick(i)],'k')
			patch([0.0 0.2],[ytick(i) ytick(i)],'k')
		end
	end
else
	text(ytick,-0.5+zeros(size(ztick)),num2str(ztick'),'HorizontalAlignment','center','VerticalAlignment','top','FontSize',fontsize)
	if smallbars,
		for i=1:numel(ztick)
			patch([ytick(i) ytick(i)],[0.8 1.0],[ytick(i) ytick(i)],'k')
			patch([ytick(i) ytick(i)],[0.0 0.2],[ytick(i) ytick(i)],'k')

		end
	end
end

if exist(options,'title'),
	title(getfieldvalue(options,'title'),'FontSize',fontsize);
end
	
function delta = dtick(range)
%Tick intervals
m = 10^floor(log10(range));
p = ceil(range/m);
if p <= 1,     delta = .1*m;
elseif p == 2, delta = .2*m;
elseif p <= 5, delta = .5*m;
else           delta = m;
end
