/*\file Shp2Exp.c
 *\brief: shp to exp file conversion mex module.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "./Shp2Exp.h"

void Shp2ExpUsage(void){/*{{{*/
	_pprintLine_("Shp2Exp - shp to exp file conversion module:");
	_pprintLine_("");
	_pprintLine_("   This module converts a file from shp to exp format.");
	_pprintLine_("");
	_pprintLine_("   Usage:");
	_pprintLine_("      [ret]=Shp2Exp(filshp,filexp,sgn,'param name',param,...);");
	_pprintLine_("");
	_pprintLine_("      filshp      file name of shp file to be read (char, extension optional)");
	_pprintLine_("      filexp      file name of exp file to be written (char)");
	_pprintLine_("      sgn         sign for hemisphere (double, +1 (north); -1 (south); or 0 (no translation))");
	_pprintLine_("");
	_pprintLine_("      central_meridian     central meridian (double, optional, but must specify with sp)");
	_pprintLine_("      standard_parallel    standard parallel (double, optional, but must specify with cm)");
	_pprintLine_("");
	_pprintLine_("      ret         return code (non-zero for warning)");
	_pprintLine_("");
	_pprintLine_("   Examples:");
	_pprintLine_("      [ret]=Shp2Exp('file.shp','file.exp', 0);");
	_pprintLine_("      [ret]=Shp2Exp('file.shp','file.exp', 1,'central_meridian',45,'standard_parallel',70);");
	_pprintLine_("      [ret]=Shp2Exp('file.shp','file.exp',-1,'central_meridian', 0,'standard_parallel',71);");
	_pprintLine_("");
}/*}}}*/
WRAPPER(Shp2Exp){

	int i,verbose=1;

	/*input: */
	char    *filshp=NULL,*filexp=NULL;
	int     sgn;
	Options* options=NULL;
	double   cm=0.,sp=0.;

	/* output: */
	int     iret=0;

	#ifndef _HAVE_SHAPELIB_ //only works if shapelib library has been compiled in.
	_error_("Shapelib not available! Cannot carry out shp file translation!");
	#endif

	/*Boot module: */
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	if (nlhs > NLHS) {
		Shp2ExpUsage(); _error_("Shp2Exp usage error");
	}
	if (nrhs < NRHS) {
		Shp2ExpUsage(); _error_("Shp2Exp usage error");
	}

	/*Input datasets: */
	FetchData(&filshp,SHP_IN);
	FetchData(&filexp,EXP_IN);
	FetchData(&sgn,SGN_IN);
	FetchData(&options,NRHS,nrhs,prhs);

	/*  defaults are in Xy2lldef, so don't duplicate them here, and only use user values if both have been specified  */
	if (options->GetOption("central_meridian") || options->GetOption("standard_parallel")) {
		options->Get(&cm,"central_meridian");
		if (verbose) _printLine_("  cm=" << cm);
		options->Get(&sp,"standard_parallel");
		if (verbose) _printLine_("  sp=" << sp);
	}

	/*some checks*/
	if (sgn < -1 || sgn > +1) _error_("Hemisphere sgn=" << sgn << " must be +1 (north), -1 (south), or 0 (no translation).");
	if (fabs(cm)      > 180.) _error_("Central meridian cm=" << cm << " must be between -180 (west) and +180 (east) degrees.");
	if (sp < 0. || sp >  90.) _error_("Standard parallel sp=" << sp << " must be between 0 and 90 degrees (in specified hemisphere).");

	/* Run core computations: */
	if (options->GetOption("central_meridian") && options->GetOption("standard_parallel"))
		iret=Shp2Expx(filshp,filexp,sgn,cm,sp);
	else
		iret=Shp2Expx(filshp,filexp,sgn);

	/*Write data: */
	WriteData(RET_OUT,iret);

	/*Clean-up*/
	delete options;
	xDelete<char>(filexp);
	xDelete<char>(filshp);

	/*end module: */
	MODULEEND();
}
