%INVERSION class definition
%
%   Usage:
%      inversion=inversion();

classdef inversion
	properties (SetAccess=public) 
		iscontrol                   = 0
		tao                         = 0
		incomplete_adjoint          = 0
		control_parameters          = NaN
		nsteps                      = 0
		maxiter_per_step            = NaN
		cost_functions              = NaN
		cost_functions_coefficients = NaN
		gradient_scaling            = NaN
		cost_function_threshold     = 0
		min_parameters              = NaN
		max_parameters              = NaN
		step_threshold              = NaN
		gradient_only               = 0
		vx_obs                      = NaN
		vy_obs                      = NaN
		vz_obs                      = NaN
		vel_obs                     = NaN
		thickness_obs               = NaN
	end
	methods
		function obj = inversion(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%default is incomplete adjoint for now
			obj.incomplete_adjoint=1;

			%parameter to be inferred by control methods (only
			%drag and B are supported yet)
			obj.control_parameters={'FrictionCoefficient'};

			%number of steps in the control methods
			obj.nsteps=20;

			%maximum number of iteration in the optimization algorithm for
			%each step
			obj.maxiter_per_step=20*ones(obj.nsteps,1);

			%the inversed parameter is updated as follows:
			%new_par=old_par + gradient_scaling(n)*C*gradient with C in [0 1];
			%usually the gradient_scaling must be of the order of magnitude of the 
			%inversed parameter (10^8 for B, 50 for drag) and can be decreased
			%after the first iterations
			obj.gradient_scaling=50*ones(obj.nsteps,1);

			%several responses can be used:
			obj.cost_functions=101*ones(obj.nsteps,1);

			%step_threshold is used to speed up control method. When
			%misfit(1)/misfit(0) < obj.step_threshold, we go directly to
			%the next step
			obj.step_threshold=.7*ones(obj.nsteps,1); %30 per cent decrement.

			%stop control solution at the gradient computation and return it? 
			obj.gradient_only=0;

			%cost_function_threshold is a criteria to stop the control methods.
			%if J[n]-J[n-1]/J[n] < criteria, the control run stops
			%NaN if not applied
			obj.cost_function_threshold=NaN; %not activated

		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{

			%Early return
			if ~obj.iscontrol, return; end

			num_controls=numel(md.inversion.control_parameters);
			num_costfunc=size(md.inversion.cost_functions,2);

			md = checkfield(md,'inversion.iscontrol','values',[0 1]);
			md = checkfield(md,'inversion.tao','values',[0 1]);
			md = checkfield(md,'inversion.incomplete_adjoint','values',[0 1]);
			md = checkfield(md,'inversion.control_parameters','cell',1,'values',{'BalancethicknessThickeningRate' 'FrictionCoefficient' 'MaterialsRheologyBbar' 'MaterialsRheologyZbar' 'Vx' 'Vy'});
			md = checkfield(md,'inversion.nsteps','numel',1,'>=',1);
			md = checkfield(md,'inversion.maxiter_per_step','size',[md.inversion.nsteps 1],'>=',0);
			md = checkfield(md,'inversion.step_threshold','size',[md.inversion.nsteps 1]);
			md = checkfield(md,'inversion.cost_functions','size',[md.inversion.nsteps num_costfunc],'values',[101:105 201 501:505]);
			md = checkfield(md,'inversion.cost_functions_coefficients','size',[md.mesh.numberofvertices num_costfunc],'>=',0);
			md = checkfield(md,'inversion.gradient_only','values',[0 1]);
			md = checkfield(md,'inversion.gradient_scaling','size',[md.inversion.nsteps num_controls]);
			md = checkfield(md,'inversion.min_parameters','size',[md.mesh.numberofvertices num_controls]);
			md = checkfield(md,'inversion.max_parameters','size',[md.mesh.numberofvertices num_controls]);

			if solution==BalancethicknessSolutionEnum()
				md = checkfield(md,'inversion.thickness_obs','size',[md.mesh.numberofvertices 1],'NaN',1);
			else
				md = checkfield(md,'inversion.vx_obs','size',[md.mesh.numberofvertices 1],'NaN',1);
				md = checkfield(md,'inversion.vy_obs','size',[md.mesh.numberofvertices 1],'NaN',1);
			end
		end % }}}
		function disp(obj) % {{{
			disp(sprintf('      inversion parameters:'));
			fielddisplay(obj,'iscontrol','is inversion activated?');
			fielddisplay(obj,'incomplete_adjoint','do we assume linear viscosity?');
			fielddisplay(obj,'control_parameters','parameter where inverse control is carried out; ex: {''FrictionCoefficient''}, or {''MaterialsRheologyBbar''}');
			fielddisplay(obj,'nsteps','number of optimization searches');
			fielddisplay(obj,'cost_functions','indicate the type of response for each optimization step');
			fielddisplay(obj,'cost_functions_coefficients','cost_functions_coefficients applied to the misfit of each vertex and for each control_parameter');
			fielddisplay(obj,'cost_function_threshold','misfit convergence criterion. Default is 1%, NaN if not applied');
			fielddisplay(obj,'maxiter_per_step','maximum iterations during each optimization step');
			fielddisplay(obj,'gradient_scaling','scaling factor on gradient direction during optimization, for each optimization step');
			fielddisplay(obj,'step_threshold','decrease threshold for misfit, default is 30%');
			fielddisplay(obj,'min_parameters','absolute minimum acceptable value of the inversed parameter on each vertex');
			fielddisplay(obj,'max_parameters','absolute maximum acceptable value of the inversed parameter on each vertex');
			fielddisplay(obj,'gradient_only','stop control method solution at gradient');
			fielddisplay(obj,'vx_obs','observed velocity x component [m/a]');
			fielddisplay(obj,'vy_obs','observed velocity y component [m/a]');
			fielddisplay(obj,'vel_obs','observed velocity magnitude [m/a]');
			fielddisplay(obj,'thickness_obs','observed thickness [m]');
			disp('Available cost functions:');
			disp('   101: SurfaceAbsVelMisfit');
			disp('   102: SurfaceRelVelMisfit');
			disp('   103: SurfaceLogVelMisfit');
			disp('   104: SurfaceLogVxVyMisfit');
			disp('   105: SurfaceAverageVelMisfit');
			disp('   201: ThicknessAbsMisfit');
			disp('   501: DragCoefficientAbsGradient');
			disp('   502: RheologyBbarAbsGradient');
			disp('   503: ThicknessAbsGradient');
		end % }}}
		function marshall(obj,fid) % {{{

			WriteData(fid,'object',obj,'fieldname','iscontrol','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','tao','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','incomplete_adjoint','format','Boolean');
			if ~obj.iscontrol, return; end
			WriteData(fid,'object',obj,'fieldname','nsteps','format','Integer');
			WriteData(fid,'object',obj,'fieldname','maxiter_per_step','format','DoubleMat','mattype',3);
			WriteData(fid,'object',obj,'fieldname','cost_functions_coefficients','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'fieldname','gradient_scaling','format','DoubleMat','mattype',3);
			WriteData(fid,'object',obj,'fieldname','cost_function_threshold','format','Double');
			WriteData(fid,'object',obj,'fieldname','min_parameters','format','DoubleMat','mattype',3);
			WriteData(fid,'object',obj,'fieldname','max_parameters','format','DoubleMat','mattype',3);
			WriteData(fid,'object',obj,'fieldname','step_threshold','format','DoubleMat','mattype',3);
			WriteData(fid,'object',obj,'fieldname','gradient_only','format','Boolean');
			WriteData(fid,'object',obj,'fieldname','vx_obs','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'fieldname','vy_obs','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'fieldname','vz_obs','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'fieldname','thickness_obs','format','DoubleMat','mattype',1);

			%process control parameters
			num_control_parameters=numel(obj.control_parameters);
			data=zeros(1,num_control_parameters);
			for i=1:num_control_parameters,
				data(i)=StringToEnum(obj.control_parameters{i});
			end
			WriteData(fid,'data',data,'enum',InversionControlParametersEnum(),'format','DoubleMat','mattype',3);
			WriteData(fid,'data',num_control_parameters,'enum',InversionNumControlParametersEnum(),'format','Integer');

			%process cost functions
			num_cost_functions=size(obj.cost_functions,2);
			data=obj.cost_functions;
			pos=find(data==101); data(pos)=SurfaceAbsVelMisfitEnum();
			pos=find(data==102); data(pos)=SurfaceRelVelMisfitEnum();
			pos=find(data==103); data(pos)=SurfaceLogVelMisfitEnum();
			pos=find(data==104); data(pos)=SurfaceLogVxVyMisfitEnum();
			pos=find(data==105); data(pos)=SurfaceAverageVelMisfitEnum();
			pos=find(data==201); data(pos)=ThicknessAbsMisfitEnum();
			pos=find(data==501); data(pos)=DragCoefficientAbsGradientEnum();
			pos=find(data==502); data(pos)=RheologyBbarAbsGradientEnum();
			pos=find(data==503); data(pos)=ThicknessAbsGradientEnum();
			pos=find(data==504); data(pos)=ThicknessAlongGradientEnum();
			pos=find(data==505); data(pos)=ThicknessAcrossGradientEnum();
			WriteData(fid,'data',data,'enum',InversionCostFunctionsEnum(),'format','DoubleMat','mattype',3);
			WriteData(fid,'data',num_cost_functions,'enum',InversionNumCostFunctionsEnum(),'format','Integer');
		end % }}}
	end
end
