function varargout=runme(varargin)
%RUNME - test deck for ISSM nightly runs
%
%   In a test deck directory (tests/Vertification/NightlyRun for example)
%   The following command will launch all the existing tests:
%   >> runme
%   To run the tests 101 and 102:
%   >> runme('id',[101 102])
%   etc...
%
%   Available options:
%      'id'            followed by the list of ids requested
%      'exclude'       ids to be excluded from the test
%      'benchmark'     'nightly' (nightly run/ daily run)
%                      'ismip'  : validation of ismip-hom tests
%                      'eismint': validation of eismint tests
%                      'thermal': validation of thermal tests
%                      'mesh'   : validation of mesh tests
%                      'adolc'   : validation of adolc tests
%                      ...
%      'procedure'     'check' : run the test (default)
%                      'update': update the archive
%                      'model' : prepare the model but no test is run
%
%   Usage:
%      md=runme(varargin);
%
%   Examples:
%      runme;
%      runme('exclude',101);
%      md=runme('id',102,'procedure','model');

%Get ISSM_DIR variable
ISSM_DIR=issmdir();

%Check inputs
% {{{
if nargout>1
	help runme
	error('runme error message: bad usage');
end

%recover options
options=pairoptions(varargin{:});
% }}}

%Process options
%GET benchmark {{{
benchmark=getfieldvalue(options,'benchmark','nightly');
if ~ismember(benchmark,{'all','nightly','ismip','eismint','thermal','mesh','validation','tranforcing','adolc'})
	disp('runme warning: benchmark not supported, defaulting to test ''nightly''')
	benchmark='nightly';
end
% }}}
%GET procedure {{{
procedure=getfieldvalue(options,'procedure','check');
if ~ismember(procedure,{'check','update'})
	disp('runme warning: procedure not supported, defaulting to test ''check''')
	procedure='check';
end
% }}}
%GET output {{{
output=getfieldvalue(options,'output','none');
if ~ismember(output,{'nightly','daily','none'})
	disp('runme warning: output not supported, defaulting to test ''none''')
	output='none';
end
% }}}
%GET RANK and NUMPROCS for multithreaded runs  {{{
rank=getfieldvalue(options,'rank',1);
numprocs=getfieldvalue(options,'numprocs',1);
if (numprocs<rank), numprocs=1; end
% }}}
%GET ids  {{{
flist=dir;%use dir, as it seems to act OS independent
list_ids=[];
for i=1:numel(flist),
	if ( strncmp(flist(i).name,'test',4) &...                         %File name must start with 'test'
			strncmp(fliplr(flist(i).name),fliplr('.m'),2)&...           %File name must end by '.m'
			~strcmp(flist(i).name,'test.m'))                            %File name must be different than 'test.m'
		list_ids(end+1)=eval(flist(i).name(5:end-2));                  %Keep test id only (skip 'test' and '.m')
	end
end
[i1,i2]=parallelrange(rank,numprocs,length(list_ids));               %Get tests for this cpu only
list_ids=list_ids(i1:i2);

test_ids=getfieldvalue(options,'id',list_ids);
test_ids=intersect(test_ids,list_ids);
% }}}
%GET exclude {{{
exclude_ids=getfieldvalue(options,'exclude',[]);
exclude_ids=[exclude_ids];
pos=find(ismember(test_ids,exclude_ids));
test_ids(pos)=[];
% }}}
%Process Ids according to benchmarks{{{
if strcmpi(benchmark,'nightly'),
	test_ids=intersect(test_ids,[1:999]);
elseif strcmpi(benchmark,'ismip'),
	test_ids=intersect(test_ids,[1101:1199]);
elseif strcmpi(benchmark,'eismint'),
	test_ids=intersect(test_ids,[1201:1299]);
elseif strcmpi(benchmark,'thermal'),
	test_ids=intersect(test_ids,[1301:1399]);
elseif strcmpi(benchmark,'mesh'),
	test_ids=intersect(test_ids,[1401:1499]);
elseif strcmpi(benchmark,'adolc'),
	test_ids=intersect(test_ids,[3001:3020]);
elseif strcmpi(benchmark,'validation'),
	test_ids=intersect(test_ids,[1001:1999]);
elseif strcmpi(benchmark,'tranforcing'),
	test_ids=intersect(test_ids,[1501:1502]);
end
% }}}

%Loop over tests and launch sequence
root=pwd;
for id=test_ids,
	try,

		%Execute test
		eval(['cd ' root ]);
		id_string=IdToName(id);
		eval(['test' num2str(id)]);

		%UPDATE ARCHIVE?
		archive_name=['Archive' num2str(id) ];
		if strcmpi(procedure,'update'),
			delete(['../Archives/' archive_name '.nc'])
			for k=1:length(field_names),
				field=field_values{k};
				% matlab writes the dimensions reversed and matrices transposed into netcdf, so compensate for that
				nccreate(['../Archives/' archive_name '.nc'],[archive_name '_field' num2str(k)],...
				         'Dimensions',{[archive_name '_field' num2str(k) '_2'] size(field,2) [archive_name '_field' num2str(k) '_1'] size(field,1)},...
				         'Format','classic');
				ncwrite(['../Archives/' archive_name '.nc'],[archive_name '_field' num2str(k)],transpose(field));
			end
			disp(sprintf(['File ./../Archives/' archive_name '.nc saved\n']));

		%ELSE: CHECK TEST
		else,
			for k=1:length(field_names),

				try,
					%Get field and tolerance
					field=field_values{k};
					fieldname=field_names{k};
					tolerance=field_tolerances{k};

					%compare to archive
					% matlab reads the dimensions reversed and matrices transposed from netcdf, so compensate for that
					archive=transpose(ncread(['../Archives/' archive_name '.nc'],[archive_name '_field' num2str(k)]));
					error_diff=full(max(abs(archive(:)-field(:)))/(max(abs(archive))+eps));

					%disp test result
					if (error_diff>tolerance);
						disp(sprintf(['ERROR   difference: %-7.2g > %7.2g test id: %i test name: %s field: %s'],...
							error_diff,tolerance,id,id_string,fieldname));
					else
						disp(sprintf(['SUCCESS difference: %-7.2g < %7.2g test id: %i test name: %s field: %s'],...
							error_diff,tolerance,id,id_string,fieldname));
					end

				catch me2

					%something went wrong, print failure message:
					directory=strsplit(pwd,'/');
					message=getReport(me2);
					if strcmpi(output,'nightly')
						fid=fopen([ISSM_DIR '/nightlylog/matlaberror.log'], 'at');
						fprintf(fid,'%s',message);
						fprintf(fid,'\n------------------------------------------------------------------\n');
						fclose(fid);
						disp(sprintf(['FAILURE difference: N/A test id: %i test name: %s field: %s'],id,id_string,fieldname));
					elseif strcmpi(output,'daily');
						fid=fopen([ISSM_DIR '/dailylog/matlaberror.log'], 'at');
						fprintf(fid,'%s',message);
						fprintf(fid,'\n------------------------------------------------------------------\n');
						fclose(fid);
						disp(sprintf(['FAILURE difference: N/A test id: %i test name: %s field: %s'],id,id_string,fieldname));
					else
						disp(sprintf(['FAILURE difference: N/A test id: %i test name: %s field: %s'],id,id_string,fieldname));
						rethrow(me2);
					end
				end
			end
		end

	catch me,

		%something went wrong, print failure message:
		directory=strsplit(pwd,'/');
		message=getReport(me);
		if strcmpi(output,'nightly')
			fid=fopen([ISSM_DIR '/nightlylog/matlaberror.log'], 'at');
			fprintf(fid,'%s',message);
			fprintf(fid,'\n------------------------------------------------------------------\n');
			fclose(fid);
			disp(sprintf(['FAILURE difference: N/A test id: %i test name: %s field: %s'],id,id_string,'N/A'));
		elseif strcmpi(output,'daily');
			fid=fopen([ISSM_DIR '/dailylog/matlaberror.log'], 'at');
			fprintf(fid,'%s',message);
			fprintf(fid,'\n------------------------------------------------------------------\n');
			fclose(fid);
			disp(sprintf(['FAILURE difference: N/A test id: %i test name: %s field: %s'],id,id_string,'N/A'));
		else
			disp(sprintf(['FAILURE difference: N/A test id: %i test name: %s field: %s'],id,id_string,'N/A'));
			rethrow(me);
		end
	end
end

%output md if requested
if nargout==1
	varargout{1}=md;
end
