/*!\file:  TriMeshProcessRifts.cpp
 * \brief split a mesh where a rift (or fault) is present
 */ 

#include "./TriMeshProcessRifts.h"

void TriMeshProcessRiftsUsage(void){/*{{{*/
	_printLine_("");
	_printLine_("   usage: [index2,x2,y2,segments2,segmentmarkers2,rifts2]=TriMeshProcessrifts(index1,x1,y1,segments1,segmentmarkers1) ");
	_printLine_("      where: (index1,x1,y1,segments1,segmentmarkers1) is an initial triangulation.");
	_printLine_("      index2,x2,y2,segments2,segmentmarkers2,rifts2 is the resulting triangulation where rifts have been processed.");
}/*}}}*/
WRAPPER(TriMeshProcessRifts){

	/* returned quantities: */
	RiftStruct *riftstruct = NULL;

	/* input: */
	int     nel,nods;
	double *index          = NULL;
	double *x              = NULL;
	double *y              = NULL;
	double *segments       = NULL;
	double *segmentmarkers = NULL;
	int     num_seg;

	/*Boot module*/
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	CHECKARGUMENTS(NLHS,NRHS,&TriMeshProcessRiftsUsage);

	/*Fetch data */
	FetchData(&index,&nel,NULL,INDEXIN);
	FetchData(&x,&nods,XIN);
	FetchData(&y,NULL,YIN);
	FetchData(&segments,&num_seg,NULL,SEGMENTSIN);
	FetchData(&segmentmarkers,NULL,SEGMENTMARKERSIN);

	/*call x layer*/
	TriMeshProcessRiftsx(&index,&nel,&x,&y,&nods,&segments,&segmentmarkers,&num_seg,
				&riftstruct);

	/*Output : */
	WriteData(INDEXOUT,index,nel,3);
	WriteData(XOUT,x,nods,1);
	WriteData(YOUT,y,nods,1);
	WriteData(SEGMENTSOUT,segments,num_seg,3);
	WriteData(SEGMENTMARKERSOUT,segmentmarkers,num_seg,1);

	mxArray    *pmxa_array    = NULL;
	mxArray    *pmxa_array2   = NULL;
	mxArray    *pmxa_array3   = NULL;
	const char *fnames[10];
	mwSize      ndim          = 2;
	mwSize      dimensions[2] = {1,1};
	double     *pair          = NULL;

	/*Create a structure rifts where if i is a rift number, we have the following fields rifts(i).segments and rifts(i).numsegs: */
	fnames[0] = "numsegs";
	fnames[1] = "segments";
	fnames[2] = "pairs";
	fnames[3] = "tips";
	fnames[4] = "penaltypairs";
	fnames[5] = "fill";
	fnames[6] = "friction";
	fnames[7] = "fraction";
	fnames[8] = "fractionincrement";
	fnames[9] = "state";

	dimensions[0]=riftstruct->numrifts;
	pmxa_array=mxCreateStructArray(ndim,dimensions,10,fnames);

	for(int i=0;i<riftstruct->numrifts;i++){

		/*Segments: */
		WriteData(&pmxa_array3,riftstruct->riftssegments[i],riftstruct->riftsnumsegments[i],3);
		mxSetField(pmxa_array,i,"segments",pmxa_array3);
		mxSetField(pmxa_array,i,"numsegs",mxCreateDoubleScalar((double)riftstruct->riftsnumsegments[i]));

		/*Element pairs: */
		WriteData(&pmxa_array3,riftstruct->riftspairs[i],riftstruct->riftsnumpairs[i],2);
		mxSetField(pmxa_array,i,"pairs",pmxa_array3);

		/*Tips: */
		pmxa_array2= mxCreateDoubleMatrix(0,0,mxREAL);
		mxSetM(pmxa_array2,1);
		pair=(double*)mxMalloc(2*sizeof(double));
		pair[0]=*(riftstruct->riftstips+2*i+0);
		pair[1]=*(riftstruct->riftstips+2*i+1);
		mxSetN(pmxa_array2,2);
		mxSetPr(pmxa_array2,pair);
		mxSetField(pmxa_array,i,"tips",pmxa_array2);

		/*Penalty pairs: */
		WriteData(&pmxa_array3,riftstruct->riftspenaltypairs[i],riftstruct->riftsnumpenaltypairs[i],7);
		mxSetField(pmxa_array,i,"penaltypairs",pmxa_array3);

		/*Friction fraction, fractionincrement  and fill: */
		mxSetField(pmxa_array,i,"friction",mxCreateDoubleScalar(0));
		mxSetField(pmxa_array,i,"fill",mxCreateDoubleScalar(IceEnum)); //default is ice
		mxSetField(pmxa_array,i,"fraction",mxCreateDoubleScalar(0)); //default is ice
		mxSetField(pmxa_array,i,"fractionincrement",mxCreateDoubleScalar(0.1)); 

		/*State: */
		pmxa_array2= mxCreateDoubleMatrix(0,0,mxREAL);
		mxSetM(pmxa_array2,1);
		mxSetN(pmxa_array2,riftstruct->riftsnumpenaltypairs[i]);
		mxSetPr(pmxa_array2,riftstruct->state[i]);
		mexCallMATLAB( 1, &pmxa_array3, 1, &pmxa_array2, "transpose");

		mxSetField(pmxa_array,i,"state",pmxa_array3);
	}

	plhs[5]=pmxa_array;

	/*end module: */
	MODULEEND();
}
