/*!\file:  issm.cpp
 * \brief: ISSM main parallel program
 */ 

#include "../issm.h"
#include "../include/globals.h"
	
void ProfilerEcho(Profiler* profiler);
void ProfilerEnd(Profiler* profiler, Results* results, Parameters* parameters);

int main(int argc,char **argv){

	/*FemModel: */
	FemModel *femmodel = NULL;

	/*Print starting banner: {{{*/
	_pprintLine_("");
	_pprintLine_("Ice Sheet System Model (" << PACKAGE_NAME << ") version " << PACKAGE_VERSION);
	_pprintLine_("(website: " << PACKAGE_URL << " contact: " << PACKAGE_BUGREPORT << ")");
	_pprintLine_(""); 
	/*}}}*/

	/*Initialize exception trapping: */
	ExceptionTrapBegin();

	/*Initialize environment (MPI, PETSC, MUMPS, etc ...)*/
	EnvironmentInit(argc,argv);
		
	/*Initialize femmodel from arguments provided command line: */
	femmodel=new FemModel(argc,argv);

	/*Solve: */
	femmodel->Solve();

	/*Some profiling: */
	ProfilerEnd(femmodel->profiler,femmodel->results,femmodel->parameters);

	_pprintLine_("write results to disk:");
	OutputResultsx(femmodel->elements, femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,femmodel->results);
	
	/*Profiling at the end: */
	femmodel->profiler->Tag(Finish); 
	ProfilerEcho(femmodel->profiler);
	
	/*Wrap up: */
	delete femmodel;

	/*Finalize environment:*/
	EnvironmentFinalize();

	/*Finalize exception trapping: */
	ExceptionTrapEnd();

	/*Return unix success: */
	return 0; 
}
	
void ProfilerEcho(Profiler* profiler){ /*{{{*/

	_pprintLine_("");
	_pprintLine_("   "<<setw(40)<<left<<"FemModel initialization elapsed time:"<<profiler->DeltaTime(StartInit,FinishInit));
	_pprintLine_("   "<<setw(40)<<left<<"Core solution elapsed time:"<<profiler->DeltaTime(StartCore,FinishCore));
	_pprintLine_("");
	_pprintLine_("   Total elapsed time:"
		<<profiler->DeltaTimeModHour(Start,Finish)<<" hrs "
		<<profiler->DeltaTimeModMin(Start,Finish)<<" min "
		<<profiler->DeltaTimeModSec(Start,Finish)<<" sec"
		);
	_pprintLine_("");

} /*}}}*/
void ProfilerEnd(Profiler* profiler, Results* results, Parameters* parameters){ /*{{{*/

	bool profiling = false;
	
	IssmDouble solution_time;
	IssmDouble solution_flops;
	IssmDouble solution_memory;

	parameters->FindParam(&profiling,DebugProfilingEnum); 

	if(profiling){

		solution_time=profiler->DeltaTime(StartCore,FinishCore);
		solution_flops=profiler->DeltaFlops(StartCore,FinishCore);
		solution_memory=profiler->Memory(FinishCore);

		_pprintLine_("Solution elapsed time  : " << solution_time << "  Seconds");
		_pprintLine_("Solution elapsed flops : " << solution_flops << "  Flops");
		_pprintLine_("Solution memory used   : " << solution_memory << "  Bytes");

		/*Add to results: */
		results->AddObject(new GenericExternalResult<IssmDouble>(results->Size()+1, ProfilingSolutionTimeEnum, solution_time, 1, 0));
		results->AddObject(new GenericExternalResult<IssmDouble>(results->Size()+1, ProfilingCurrentMemEnum, solution_memory, 1, 0));
		results->AddObject(new GenericExternalResult<IssmDouble>(results->Size()+1, ProfilingCurrentFlopsEnum, solution_flops, 1, 0));
	}

} /*}}}*/
