/*!\file:  issm.cpp
 * \brief: ISSM main parallel program
 */ 

#include "../issm.h"
#include "../include/globals.h"
	
void ProfilerEcho(Profiler* profiler);
void ProfilerEnd(Profiler* profiler, Results* results, Parameters* parameters);

int main(int argc,char **argv){

	/*I/O: */
	FILE *output_fid       = NULL;
	FILE *petscoptionsfid  = NULL;
	bool  waitonlock       = false;

	/*FemModel: */
	FemModel *femmodel = NULL;

	/*configuration: */
	void (*solutioncore)(FemModel*)=NULL; //core solution function pointer
	int* analyses=NULL;
	int  numanalyses;
	int  solution_type;

	/*File names*/
	char *lockfilename   = NULL;
	char *binfilename    = NULL;
	char *outbinfilename = NULL;
	char *petscfilename  = NULL;
	char *rootpath       = NULL;

	/*profiling*/   
	Profiler* profiler=NULL;

	/*Initialize exception trapping: */
	ExceptionTrapBegin();

	/*Initialize environment (MPI, PETSC, MUMPS, etc ...)*/
	EnvironmentInit(argc,argv);
	
	/*Start profiler: */
	profiler=new Profiler(); 
	profiler->Tag(Start);
	
	/*First process inputs*/
	_pprintLine_("");
	_pprintLine_("Ice Sheet System Model (" << PACKAGE_NAME << ") version " << PACKAGE_VERSION);
	_pprintLine_("(website: " << PACKAGE_URL << " contact: " << PACKAGE_BUGREPORT << ")");
	_pprintLine_("");
	ProcessArguments(&solution_type,&binfilename,&outbinfilename,&petscfilename,&lockfilename,&rootpath,argc,argv);

	/*out of solution_type, figure out types of analyses needed in the femmodel: */
	AnalysisConfiguration(&analyses,&numanalyses,solution_type);

	/*Create femmodel, using input file: */
	profiler->Tag(StartInit);
	femmodel=new FemModel(rootpath,binfilename,outbinfilename,solution_type,analyses,numanalyses);
	
	/*get type of solution we are going to run: */
	CorePointerFromSolutionEnum(&solutioncore,femmodel->parameters,solution_type);

	/*Open output file once for all*/
	output_fid=pfopen(outbinfilename,"wb");
	femmodel->parameters->SetParam(output_fid,OutputFilePointerEnum);

	/*add petsc options to parameters: */
	petscoptionsfid=pfopen(petscfilename,"r");
	ParsePetscOptionsx(femmodel->parameters,petscoptionsfid);
	pfclose(petscoptionsfid,petscfilename);

	profiler->Tag(FinishInit);
	
	/*call cores: */
	_pprintLine_("call computational core:");
	profiler->Tag(StartCore); solutioncore(femmodel); profiler->Tag(FinishCore); 
	profiler->Tag(StartAdCore); ad_core(femmodel); profiler->Tag(FinishAdCore); 
	
	ProfilerEnd(profiler,femmodel->results,femmodel->parameters);

	_pprintLine_("write results to disk:");
	OutputResultsx(femmodel->elements, femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,femmodel->results);
	
	/*Close output and petsc options file and write lock file if requested*/
	pfclose(output_fid,lockfilename);
	femmodel->parameters->FindParam(&waitonlock,SettingsWaitonlockEnum);
	if (waitonlock>0){
		_pprintLine_("write lock file:");
		WriteLockFile(lockfilename);
	}

	/*Free resources */
	xDelete<int>(analyses);
	xDelete<char>(lockfilename);
	xDelete<char>(binfilename);
	xDelete<char>(outbinfilename);
	xDelete<char>(petscfilename);
	xDelete<char>(rootpath);
	delete femmodel;

	/*Profiling at the end: */
	profiler->Tag(Finish); 
	ProfilerEcho(profiler);

	/*Finalize environment:*/
	EnvironmentFinalize();

	/*Finalize exception trapping: */
	ExceptionTrapEnd();

	/*Return unix success: */
	return 0; 
}
	
void ProfilerEcho(Profiler* profiler){ /*{{{*/

	_pprintLine_("");
	_pprintLine_("   "<<setw(40)<<left<<"FemModel initialization elapsed time:"<<profiler->DeltaTime(StartInit,FinishInit));
	_pprintLine_("   "<<setw(40)<<left<<"Core solution elapsed time:"<<profiler->DeltaTime(StartCore,FinishCore));
	_pprintLine_("");
	_pprintLine_("   Total elapsed time:"
		<<profiler->DeltaTimeModHour(Start,Finish)<<" hrs "
		<<profiler->DeltaTimeModMin(Start,Finish)<<" min "
		<<profiler->DeltaTimeModSec(Start,Finish)<<" sec"
		);
	_pprintLine_("");

} /*}}}*/
void ProfilerEnd(Profiler* profiler, Results* results, Parameters* parameters){ /*{{{*/

	bool profiling = false;
	
	IssmDouble solution_time;
	IssmDouble solution_flops;
	IssmDouble solution_memory;

	parameters->FindParam(&profiling,DebugProfilingEnum); 

	if(profiling){

		solution_time=profiler->DeltaTime(StartCore,FinishCore);
		solution_flops=profiler->DeltaFlops(StartCore,FinishCore);
		solution_memory=profiler->Memory(FinishCore);

		_pprintLine_("Solution elapsed time  : " << solution_time << "  Seconds");
		_pprintLine_("Solution elapsed flops : " << solution_flops << "  Flops");
		_pprintLine_("Solution memory used   : " << solution_memory << "  Bytes");

		/*Add to results: */
		results->AddObject(new GenericExternalResult<IssmDouble>(results->Size()+1, ProfilingSolutionTimeEnum, solution_time, 1, 0));
		results->AddObject(new GenericExternalResult<IssmDouble>(results->Size()+1, ProfilingCurrentMemEnum, solution_memory, 1, 0));
		results->AddObject(new GenericExternalResult<IssmDouble>(results->Size()+1, ProfilingCurrentFlopsEnum, solution_flops, 1, 0));
	}

} /*}}}*/
