/*!\file:  issm.cpp
 * \brief: ISSM main parallel program
 */ 

#include "../issm.h"
#include "../include/globals.h"
	
void ProfilerEcho(Profiler* profiler);

int main(int argc,char **argv){

	/*I/O: */
	FILE *output_fid       = NULL;
	FILE *petscoptionsfid  = NULL;
	bool  waitonlock       = false;
	bool  dakota_analysis,control_analysis,tao_analysis;

	/*AD: */
	bool autodiff=false;
	size_t   tape_stats[11];

	/*FemModel: */
	FemModel *femmodel = NULL;

	/*configuration: */
	void (*solutioncore)(FemModel*)=NULL; //core solution function pointer
	int* analyses=NULL;
	int  numanalyses;
	int  solution_type;

	/*File names*/
	char *lockfilename   = NULL;
	char *binfilename    = NULL;
	char *outbinfilename = NULL;
	char *petscfilename  = NULL;
	char *rootpath       = NULL;

	/*profiling*/   
	Profiler* profiler=NULL;
	bool profiling = false;
	IssmPDouble Time_start, Flops_start;
	IssmPDouble Solution_time, Memory_use, Current_flops;

	/*Initialize exception trapping: */
	ExceptionTrapBegin();

	/*Initialize environment (MPI, PETSC, MUMPS, etc ...)*/
	EnvironmentInit(argc,argv);
	
	/*Some profiling: */
	profiler=new Profiler(); profiler->Tag(Start);
	
	/*First process inputs*/
	_pprintLine_("");
	_pprintLine_("Ice Sheet System Model (" << PACKAGE_NAME << ") version " << PACKAGE_VERSION);
	_pprintLine_("(website: " << PACKAGE_URL << " contact: " << PACKAGE_BUGREPORT << ")");
	_pprintLine_("");
	ProcessArguments(&solution_type,&binfilename,&outbinfilename,&petscfilename,&lockfilename,&rootpath,argc,argv);

	/*out of solution_type, figure out types of analyses needed in the femmodel: */
	AnalysisConfiguration(&analyses,&numanalyses,solution_type);

	/*Create femmodel, using input file: */
	profiler->Tag(StartInit);
	
	femmodel=new FemModel(rootpath,binfilename,outbinfilename,solution_type,analyses,numanalyses);
	
	/*get type of solution we are going to run: */
	CorePointerFromSolutionEnum(&solutioncore,femmodel->parameters,solution_type);

	/*Open output file once for all*/
	output_fid=pfopen(outbinfilename,"wb");
	femmodel->parameters->SetParam(output_fid,OutputFilePointerEnum);

	/*add petsc options to parameters: */
	petscoptionsfid=pfopen(petscfilename,"r");
	ParsePetscOptionsx(femmodel->parameters,petscoptionsfid);
	pfclose(petscoptionsfid,petscfilename);

	/*get parameters: */
	femmodel->parameters->FindParam(&waitonlock,SettingsWaitonlockEnum);
	femmodel->parameters->FindParam(&dakota_analysis,QmuIsdakotaEnum);
	femmodel->parameters->FindParam(&control_analysis,InversionIscontrolEnum);
	femmodel->parameters->FindParam(&tao_analysis,InversionTaoEnum);
	femmodel->parameters->FindParam(&profiling,DebugProfilingEnum); 
	femmodel->parameters->FindParam(&autodiff,AutodiffIsautodiffEnum);

	/*Profiling: */
	profiler->Tag(FinishInit);
	
	/*If running AD, then initialize a placeholder with which to work: */
	#ifdef _HAVE_ADOLC_
	GenericParam<Adolc_edf> *theAdolcEDF_p=new GenericParam<Adolc_edf>(AdolcParamEnum);
	theAdolcEDF_p->GetParameterValue().myEDF_for_solverx_p=reg_ext_fct(EDF_for_solverx);
	// to save some space:
	// we know we won't use adolc inside of  the solver:
	theAdolcEDF_p->GetParameterValue().myEDF_for_solverx_p->nestedAdolc=false;
	// the solution vector is just allocated and doesn't have a meaningfull prior value
	theAdolcEDF_p->GetParameterValue().myEDF_for_solverx_p->dp_y_priorRequired=false;
	 // the solver wrapper makes sure the matrix and the right hand side don't change
	theAdolcEDF_p->GetParameterValue().myEDF_for_solverx_p->dp_x_changes=false;
	femmodel->parameters->AddObject(theAdolcEDF_p);
	#else
	if(autodiff) _error_("ISSM was not compiled with ADOLC support, cannot carry out autodiff analysis!");
	#endif

	_pprintLine_("call computational core:");
	profiler->Tag(StartCore);
	
	if(profiling)ProfilingStart(&Time_start,&Flops_start);

	if(dakota_analysis){
		#ifdef _HAVE_DAKOTA_
		Dakotax(femmodel);
		#else
		_error_("ISSM was not compiled with dakota support, cannot carry out dakota analysis!");
		#endif
	}
	else if(control_analysis){
		#ifdef _HAVE_CONTROL_
		if(tao_analysis)
		 controltao_core(femmodel);
		else
		 control_core(femmodel);
		#else
		_error_("ISSM was not compiled with control support, cannot carry out dakota analysis!");
		#endif
	}
	else{
		solutioncore(femmodel);
	}

	if(profiling){
		ProfilingEnd(&Solution_time,&Memory_use,&Current_flops,Time_start,Flops_start);
		femmodel->results->AddObject(new GenericExternalResult<double>(femmodel->results->Size()+1, ProfilingSolutionTimeEnum, Solution_time, 1, 0));
		femmodel->results->AddObject(new GenericExternalResult<double>(femmodel->results->Size()+1, ProfilingCurrentMemEnum, Memory_use, 1, 0));
		femmodel->results->AddObject(new GenericExternalResult<double>(femmodel->results->Size()+1, ProfilingCurrentFlopsEnum, Current_flops, 1, 0));
	}

	#ifdef _HAVE_ADOLC_
	if(autodiff){
		trace_off();
		AutodiffDriversx(femmodel->elements, femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,femmodel->results);
	}
	#endif

	profiler->Tag(FinishCore);

	
	_pprintLine_("write results to disk:");
	OutputResultsx(femmodel->elements, femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,femmodel->results);
	
	/*If running AD, close our tape, print statistics: {{{*/
	#ifdef _HAVE_ADOLC_
	if(autodiff){
		tapestats(1,tape_stats); //reading of tape statistics
		if(VerboseAutodiff()){
			_pprintLine_("   ADOLC statistics: ");
			_pprintLine_("   "<<setw(45)<<left<<"Number of independents: " <<tape_stats[0]);
			_pprintLine_("   "<<setw(45)<<left<<"Number of dependents: " <<tape_stats[1]);
			_pprintLine_("   "<<setw(45)<<left<<"Maximal number of live active variables: " <<tape_stats[2]);
			_pprintLine_("   "<<setw(45)<<left<<"Size of value stack (number of overwrites): " <<tape_stats[3]);
			_pprintLine_("   "<<setw(45)<<left<<"Buffer size (a multiple of eight): " <<tape_stats[4]);
			_pprintLine_("   "<<setw(45)<<left<<"Total number of operations recorded: " <<tape_stats[5]);
		}
	}
	#endif  /*}}}*/

	/*Close output and petsc options file and write lock file if requested*/
	pfclose(output_fid,lockfilename);
	if (waitonlock>0){
		_pprintLine_("write lock file:");
		WriteLockFile(lockfilename);
	}

	/*Free resources */
	xDelete<int>(analyses);
	xDelete<char>(lockfilename);
	xDelete<char>(binfilename);
	xDelete<char>(outbinfilename);
	xDelete<char>(petscfilename);
	xDelete<char>(rootpath);
	delete femmodel;

	/*Get finish time and close*/
	profiler->Tag(Finish); ProfilerEcho(profiler);

	/*Finalize environment:*/
	EnvironmentFinalize();

	/*Finalize exception trapping: */
	ExceptionTrapEnd();

	/*Return unix success: */
	return 0; 
}
	


void ProfilerEcho(Profiler* profiler){

	_pprintLine_("");
	_pprintLine_("   "<<setw(40)<<left<<"FemModel initialization elapsed time:"<<profiler->Delta(StartInit,FinishInit));
	_pprintLine_("   "<<setw(40)<<left<<"Core solution elapsed time:"<<profiler->Delta(StartCore,FinishCore));
	_pprintLine_("");
	_pprintLine_("   Total elapsed time:"
		<<profiler->DeltaModHour(Start,Finish)<<" hrs "
		<<profiler->DeltaModMin(Start,Finish)<<" min "
		<<profiler->DeltaModSec(Start,Finish)<<" sec"
		);
	_pprintLine_("");

}

