/*!\file Profiler.c
 * \brief: implementation of the Profiler object
 */

/*Include files: {{{*/
#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "./Profiler.h"
/*}}}*/

/*Profiler constructors and destructors:*/
/*FUNCTION Profiler::Profiler() default constructor {{{*/
Profiler::Profiler(){
		 this->timetags=new Parameters();
}
/*}}}*/
/*FUNCTION Profiler::~Profiler(){{{*/
Profiler::~Profiler(){
	delete timetags;
}
/*}}}*/

/*Object virtual functions definitions:*/
/*FUNCTION Profiler::Echo{{{*/
void Profiler::Echo(void){

	_printLine_("Profiler:");
	_printLine_("   time tags: ");
	this->timetags->Echo();

}
/*}}}*/
/*FUNCTION Profiler::DeepEcho{{{*/
void Profiler::DeepEcho(void){

	_printLine_("Profiler:");
	_printLine_("   time tags: ");
	this->timetags->DeepEcho();

}
/*}}}*/
/*FUNCTION Profiler::Id{{{*/
int    Profiler::Id(void){ return -1; }
/*}}}*/
/*FUNCTION Profiler::ObjectEnum{{{*/
int Profiler::ObjectEnum(void){

	return ProfilerEnum;

}
/*}}}*/

/*Profiler routines:*/
/*FUNCTION Profiler::Tag {{{*/
void  Profiler::Tag(int tagenum,bool dontmpisync){

	double time;

	/*If mpisync requested, make sure all the cpus are at the same point 
	 *in the execution: */
	if(!dontmpisync){
		#ifdef _HAVE_MPI_
		MPI_Barrier(MPI_COMM_WORLD); 
		#endif
	}

	/*Now capture time: */
	#ifdef _HAVE_MPI_
	time=MPI_Wtime();
	#else
	time=(IssmPDouble)clock();
	#endif

	/*Plug into this->timetags: */
	this->timetags->AddObject(new DoubleParam(tagenum,time));

}
/*}}}*/
/*FUNCTION Profiler::Delta {{{*/
double  Profiler::Delta(int inittag, int finaltag){

	double init, final;
	this->timetags->FindParam(&init,inittag);
	this->timetags->FindParam(&final,finaltag);

	#ifdef _HAVE_MPI_
	return final-init;
	#else
	return (final-init)/CLOCKS_PER_SEC;
	#endif
}
/*}}}*/
/*FUNCTION Profiler::DeltaModHour {{{*/
int Profiler::DeltaModHour(int inittag, int finishtag){

	double init, finish;
	this->timetags->FindParam(&init,inittag);
	this->timetags->FindParam(&finish,finishtag);

	#ifdef _HAVE_MPI_
	return int((finish-init)/3600);
	#else
	return int((finish-init)/CLOCKS_PER_SEC/3600);
	#endif

}
/*}}}*/
/*FUNCTION Profiler::DeltaModMin {{{*/
int Profiler::DeltaModMin(int inittag, int finishtag){

	double init, finish;
	this->timetags->FindParam(&init,inittag);
	this->timetags->FindParam(&finish,finishtag);

	#ifdef _HAVE_MPI_
	return int(int(finish-init)%3600/60);
	#else
	return int(int(finish-init)/CLOCKS_PER_SEC%3600/60);
	#endif
}
/*}}}*/
/*FUNCTION Profiler::DeltaModSec {{{*/
int Profiler::DeltaModSec(int inittag, int finishtag){

	double init, finish;
	this->timetags->FindParam(&init,inittag);
	this->timetags->FindParam(&finish,finishtag);

	#ifdef _HAVE_MPI_
	return int(finish-init)%60;
	#else
	return int(finish-init)/CLOCKS_PER_SEC%60;
	#endif
}
/*}}}*/
