! $Id: ESMF_LogErrUTest.F90,v 1.54.2.1 2010/02/05 19:59:10 svasquez Exp $
!
! Earth System Modeling Framework
! Copyright 2002-2010, University Corporation for Atmospheric Research,
! Massachusetts Institute of Technology, Geophysical Fluid Dynamics
! Laboratory, University of Michigan, National Centers for Environmental
! Prediction, Los Alamos National Laboratory, Argonne National Laboratory,
! NASA Goddard Space Flight Center.
! Licensed under the University of Illinois-NCSA License.
!
!==============================================================================
!
      program ESMF_LogErrUTest

!------------------------------------------------------------------------------
 
#include "ESMF_Macros.inc"
#include "ESMF.h"

!==============================================================================
!BOP
! !PROGRAM: ESMF_LogErrTest - One line general statement about this test
!
! !DESCRIPTION:
!
! The code in this file drives F90 LogErr unit tests.
! The companion file ESMF\_LogErr.F90 contains the definitions for the
! LogErr methods.
!
!-----------------------------------------------------------------------------
! !USES:
      use ESMF_TestMod     ! test methods
      use ESMF_Mod

      implicit none

!------------------------------------------------------------------------------
! The following line turns the CVS identifier string into a printable variable.
      character(*), parameter :: version = &
      '$Id: ESMF_LogErrUTest.F90,v 1.54.2.1 2010/02/05 19:59:10 svasquez Exp $'
!------------------------------------------------------------------------------

      ! cumulative result: count failures; no failures equals "all pass"
      integer :: result = 0

      ! individual test result code
      integer :: rc

      ! individual test failure message
      character(ESMF_MAXSTR) :: failMsg
      character(ESMF_MAXSTR) :: name


      !LOCAL VARIABLES:
      type(ESMF_Log) :: log1, log5, log7
      type(ESMF_LogType) :: logtype

#ifdef ESMF_TESTEXHAUSTIVE
      type(ESMF_VM):: vm
      character(ESMF_MAXSTR) :: pet_num
      real :: r1
      logical :: is_error
      character(ESMF_MAXSTR) :: msg_type
      character(80) :: filename
      integer :: num_pets, my_pet, input_status, ran_num, rc2, k, i
      integer :: datetime_commbuf(8)
      integer, pointer :: rndseed(:)
      type(ESMF_Log) :: log2, log4, log6
      character (5) :: random_chars
      character (9) :: msg_string, random_string
      character :: random_char
      integer :: my_v(8), log_v(8)
      character(8) :: todays_date
      character(10) :: todays_time
      character(4) :: my_pet_char
      character(1) :: pet_char
      type(ESMF_TimeInterval) :: one_sec, zero, time_diff
      type(ESMF_Time) :: my_time, log_time
#endif



!------------------------------------------------------------------------------
! The unit tests are divided into Sanity and Exhaustive. The Sanity tests are
! always run. When the environment variable, EXHAUSTIVE, is set to ON then 
! the EXHAUSTIVE and sanity tests both run. If the EXHAUSTIVE variable is set
! to OFF, then only the sanity unit tests.
! Special strings (Non-exhaustive and exhaustive) have been
! added to allow a script to count the number and types of unit tests.
!------------------------------------------------------------------------------
      print *, "Starting LogErr Tests"

      call ESMF_TestStart(ESMF_SRCLINE, rc=rc)
      call ESMF_CalendarSetDefault(ESMF_CAL_GREGORIAN, rc)

      !------------------------------------------------------------------------
      !NEX_UTest
      ! Test Log Open
      write(failMsg, *) "Did not return ESMF_SUCCESS"
      call ESMF_LogOpen(log1, "Log_Test_File", rc=rc)
      write(name, *) "Open Log Test"
      call ESMF_Test((rc.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc = ", rc

      !------------------------------------------------------------------------
      !NEX_UTest
      ! Test Log Open
      logtype = ESMF_LOG_SINGLE
      write(failMsg, *) "Did not return ESMF_SUCCESS"
      call ESMF_LogOpen(log5, "Single_Log_File", logtype=logtype,  rc=rc)
      write(name, *) "Open ESMF_LOG_SINGLE Log Test"
      call ESMF_Test((rc.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc = ", rc


      !------------------------------------------------------------------------
      !NEX_UTest
      ! Test Log Write
      write(failMsg, *) "Did not return ESMF_SUCCESS"
      call ESMF_LogWrite(log=log5, msg="Log Single Msg",msgtype=ESMF_LOG_INFO, &
                         rc=rc)
      write(name, *) "Write to Single Log Test"
      call ESMF_Test((rc.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc = ", rc

      !------------------------------------------------------------------------
      !NEX_UTest
      ! Test Log Open
      logtype = ESMF_LOG_NONE
      write(failMsg, *) "Did not return ESMF_FAILURE"
      call ESMF_LogOpen(log5, "None_Log_File", logtype=logtype,  rc=rc)
      write(name, *) "Open ESMF_LOG_NONE Log of opened file Test"
      call ESMF_Test((rc.eq.ESMF_FAILURE), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc = ", rc

      !------------------------------------------------------------------------
      !NEX_UTest
      ! Test Log Open
      logtype = ESMF_LOG_NONE
      write(failMsg, *) "Did not return ESMF_SUCCESS"
      call ESMF_LogOpen(log7, "None_Log_File", logtype=logtype,  rc=rc)
      write(name, *) "Open ESMF_LOG_NONE Log Test"
      call ESMF_Test((rc.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc = ", rc

      !------------------------------------------------------------------------
      !NEX_UTest
      ! Test Log Write
      write(failMsg, *) "Did not return ESMF_SUCCESS"
      call ESMF_LogWrite(log=log1, msg="Log Write 2",msgtype=ESMF_LOG_INFO, &
                         rc=rc)
      write(name, *) "Use of separate log Test"
      call ESMF_Test((rc.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc = ", rc

      !------------------------------------------------------------------------
      !NEX_UTest
      ! Test Log Close
      write(failMsg, *) "Did not return ESMF_SUCCESS"
      call ESMF_LogClose(log1, rc)
      write(name, *) "Close Log Test"
      call ESMF_Test((rc.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc = ", rc


#ifdef ESMF_TESTEXHAUSTIVE


      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Log Close of never pened file
      write(failMsg, *) "Did not return ESMF_SUCCESS"
      call ESMF_LogClose(log4, rc)
      write(name, *) "Close Log File of never opened file Test"
      call ESMF_Test((rc.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc = ", rc

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test LogFlush of an unopened log
      write(failMsg, *) "Did not return ESMF_FAILURE"
      write(name, *) " LogFlush of unopened log Test"
      call ESMF_LogFlush(log6, rc=rc)
      call ESMF_Test((rc.eq.ESMF_FAILURE), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc = ", rc

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Log Write
      write(failMsg, *) "Did not return ESMF_SUCCESS"
      call ESMF_LogWrite(msg="Log Write 2",msgtype=ESMF_LOG_INFO,rc=rc)
      write(name, *) "Use of default log Test"
      call ESMF_Test((rc.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc = ", rc

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Error Msg Found Error
      write(failMsg, *) "Did not return FALSE"
      is_error=ESMF_LogMsgFoundError(ESMF_FAILURE,"hello",rcToReturn=rc2)
      write(name, *) "Error Msg Found Error Test"
      call ESMF_Test((is_error), name, failMsg, result, ESMF_SRCLINE)

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Value of rcToReturn
      write(failMsg, *) "Did not return ESMF_FAILURE"
      write(name, *) " Verify rcToReturn Value Test"
      call ESMF_Test((rc2.eq.ESMF_FAILURE), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc2 = ", rc2

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Error Msg Found Error
      write(failMsg, *) "Did not return ESMF_SUCCESS"
      is_error=ESMF_LogMsgFoundError(ESMF_SUCCESS,"hello",rcToReturn=rc2)
      write(name, *) "Error Msg Found Error Test"
      call ESMF_Test((.NOT.is_error), name, failMsg, result, ESMF_SRCLINE)

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Log Found Alloc Error
      write(failMsg, *) "Did not return .TRUE."
      is_error=ESMF_LogFoundAllocError(ESMF_FAILURE,rcToReturn=rc2)
      write(name, *) "Log Found Alloc Error Test"
      call ESMF_Test((is_error), name, failMsg, result, ESMF_SRCLINE)

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Value of rcToReturn
      write(failMsg, *) "Did not return ESMF_RC_MEM_ALLOCATE"
      write(name, *) " Verify rcToReturn Value Test"
      call ESMF_Test((rc2.eq.ESMF_RC_MEM_ALLOCATE), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc2 = ", rc2

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Error Msg Found Alloc Error
      write(failMsg, *) "Did not return .TRUE."
      is_error=ESMF_LogMsgFoundAllocError(ESMF_FAILURE,"hello",rcToReturn=rc2)
      write(name, *) "Error Msg Found Alloc Error Test"
      call ESMF_Test((is_error), name, failMsg, result, ESMF_SRCLINE)

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Value of rcToReturn
      write(failMsg, *) "Did not return ESMF_RC_MEM_ALLOCATE"
      write(name, *) " Verify rcToReturn Value Test"
      call ESMF_Test((rc2.eq.ESMF_RC_MEM_ALLOCATE), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc2 = ", rc2

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Error Msg Found Error
      write(failMsg, *) "Did not return .FALSE."
      is_error=ESMF_LogMsgFoundAllocError(ESMF_SUCCESS,"hello",rcToReturn=rc2)
      write(name, *) "Error Msg Found Error Test"
      call ESMF_Test((.NOT.is_error), name, failMsg, result, ESMF_SRCLINE)

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Log Found Alloc Error
      write(failMsg, *) "Did not return ESMF_FAILURE"
      is_error=ESMF_LogFoundAllocError(ESMF_FAILURE,rcToReturn=rc2)
      write(name, *) "Log Found Alloc Error Test"
      call ESMF_Test((is_error), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc2 = ", rc2

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Value of rcToReturn
      write(failMsg, *) "Did not return ESMF_RC_MEM_ALLOCATE"
      write(name, *) " Verify rcToReturn Value Test"
      call ESMF_Test((rc2.eq.ESMF_RC_MEM_ALLOCATE), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc2 = ", rc2

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Log Found Alloc Error
      write(failMsg, *) "Did not return ESMF_SUCCESS"
      is_error=ESMF_LogFoundAllocError(ESMF_SUCCESS,rcToReturn=rc2)
      write(name, *) "Log Found Alloc Error Test"
      call ESMF_Test((.NOT.is_error), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc = ", rc
      !------------------------------------------------------------------------

      !EX_UTest
      ! Test Value of rcToReturn
      write(failMsg, *) "Did not return ESMF_SUCCESS"
      write(name, *) " Verify rcToReturn Value Test"
      call ESMF_Test((rc2.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc2 = ", rc2

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Log Found Dealloc Error
      write(failMsg, *) "Did not return .TRUE."
      is_error=ESMF_LogFoundDeallocError(ESMF_FAILURE,rcToReturn=rc2)
      write(name, *) "Log Found Dealloc Error Test"
      call ESMF_Test((is_error), name, failMsg, result, ESMF_SRCLINE)

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Value of rcToReturn
      write(failMsg, *) "Did not return ESMF_RC_MEM_DEALLOCATE"
      write(name, *) " Verify rcToReturn Value Test"
      call ESMF_Test((rc2.eq.ESMF_RC_MEM_DEALLOCATE), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc2 = ", rc2

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Error Msg Found Dealloc Error
      write(failMsg, *) "Did not return .TRUE."
      is_error=ESMF_LogMsgFoundDeallocError(ESMF_FAILURE,"hello",rcToReturn=rc2)
      write(name, *) "Error Msg Found Dealloc Error Test"
      call ESMF_Test((is_error), name, failMsg, result, ESMF_SRCLINE)

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Value of rcToReturn
      write(failMsg, *) "Did not return ESMF_RC_MEM_DEALLOCATE"
      write(name, *) " Verify rcToReturn Value Test"
      call ESMF_Test((rc2.eq.ESMF_RC_MEM_DEALLOCATE), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc2 = ", rc2

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Error Msg Found Error
      write(failMsg, *) "Did not return .FALSE."
      is_error=ESMF_LogMsgFoundDeallocError(ESMF_SUCCESS,"hello",rcToReturn=rc2)
      write(name, *) "Error Msg Found Error Test"
      call ESMF_Test((.NOT.is_error), name, failMsg, result, ESMF_SRCLINE)

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Log Found Dealloc Error
      write(failMsg, *) "Did not return ESMF_FAILURE"
      is_error=ESMF_LogFoundDeallocError(ESMF_FAILURE,rcToReturn=rc2)
      write(name, *) "Log Found Dealloc Error Test"
      call ESMF_Test((is_error), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc2 = ", rc2

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Value of rcToReturn
      write(failMsg, *) "Did not return ESMF_RC_MEM_DEALLOCATE"
      write(name, *) " Verify rcToReturn Value Test"
      call ESMF_Test((rc2.eq.ESMF_RC_MEM_DEALLOCATE), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc2 = ", rc2

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Log Found Dealloc Error
      write(failMsg, *) "Did not return ESMF_SUCCESS"
      is_error=ESMF_LogFoundDeallocError(ESMF_SUCCESS,rcToReturn=rc2)
      write(name, *) "Log Found Dealloc Error Test"
      call ESMF_Test((.NOT.is_error), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc = ", rc
      !------------------------------------------------------------------------

      !EX_UTest
      ! Test Value of rcToReturn
      write(failMsg, *) "Did not return ESMF_SUCCESS"
      write(name, *) " Verify rcToReturn Value Test"
      call ESMF_Test((rc2.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc2 = ", rc2

      !-----------------------------------------------------------------------
      !EX_UTest
      ! Test Log Found Error
      write(failMsg, *) "Did not return ESMF_SUCCESS"
      is_error=ESMF_LogFoundError(ESMF_SUCCESS,rcToReturn=rc2)
      write(name, *) "Log Found Error Test"
      call ESMF_Test((.NOT.is_error), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc = ", rc
      !------------------------------------------------------------------------


      !EX_UTest
      ! Test Value of rcToReturn
      write(failMsg, *) "Did not return ESMF_SUCCESS"
      write(name, *) " Verify rcToReturn Value Test"
      call ESMF_Test((rc2.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc2 = ", rc2

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Log Found Error
      write(failMsg, *) "Did not return ESMF_FAILURE"
      is_error=ESMF_LogFoundError(ESMF_FAILURE,rcToReturn=rc2)
      write(name, *) "Log Found Error Test"
      call ESMF_Test((is_error), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc = ", rc

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Value of rcToReturn
      write(failMsg, *) "Did not return ESMF_FAILURE"
      write(name, *) " Verify rcToReturn Value Test"
      call ESMF_Test((rc2.eq.ESMF_FAILURE), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc2 = ", rc2

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test LogFlush of new unopened log
      write(failMsg, *) "Did not return ESMF_FAILURE"
      write(name, *) " LogFlush of unopened log Test"
      call ESMF_LogFlush(log2, rc=rc)
      call ESMF_Test((rc.eq.ESMF_FAILURE), name, failMsg, result, ESMF_SRCLINE)

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Log Write without opening log file
      write(failMsg, *) "Returned ESMF_SUCCESS"
      call ESMF_LogWrite(log=log2, msg="Log Write One",msgtype=ESMF_LOG_INFO,rc=rc)
      write(name, *) "Write without opening log file Test"
      call ESMF_Test((rc.ne.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc = ", rc

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Log Open
      write(failMsg, *) "Did not return ESMF_SUCCESS"
      call ESMF_LogOpen(log2, "Log_Test_File_2", rc=rc)
      write(name, *) "Open Log Test"
      call ESMF_Test((rc.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc = ", rc

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Log reopenOpen 
      write(failMsg, *) "Did not return ESMF_FAILURE"
      call ESMF_LogOpen(log2, "Log_Test_File_3", rc=rc)
      write(name, *) "Open Already Open Log Test"
      call ESMF_Test((rc.eq.ESMF_FAILURE), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc = ", rc

      !------------------------------------------------------------------------
      print *, "Starting a no-op loop to wait before testing time and date"
      ! Get the local PET number
      call ESMF_VMGetGlobal(vm, rc=rc)
      call ESMF_VMGet(vm, localPet=my_pet, petCount=num_pets, rc=rc)

      call date_and_time(date=todays_date, time=todays_time, values=my_v)

      ! Each PET needs to have a time which is at least 1 second later
      ! than its lower-numbered neighbor.

      do, i=1, num_pets-1
        if (my_pet == i-1) then
          datetime_commbuf = my_v
          call ESMF_VMSend (vm,  &
            datetime_commbuf, count=size (datetime_commbuf), dst=i)
        else if (my_pet == i) then
          call ESMF_VMRecv (vm,  &
            datetime_commbuf, count=size (datetime_commbuf), src=i-1)
          do
            call date_and_time (values=my_v, date=todays_date, time=todays_time)
            if (datetime_commbuf(7) /= my_v(7)) exit
          end do
          datetime_commbuf = my_v
        end if
        call ESMF_VMBarrier (vm)
      end do

      print *, "Ending the no-op loop"

      ! Generate a random string using clock as seed and write it to log file
      call random_seed()
      call random_seed(size=k)
      print *, "size of random seed = ", k
      allocate(rndseed(k))
      call date_and_time(values=my_v)
      do i=1,k
        rndseed(i)=i*(my_v(6)+my_v(7))
      end do
      print *, "generated a random seed based on current time = " , rndseed
      call random_seed(put=rndseed(1:k))
      deallocate(rndseed)
      do i=1, 5
      	call random_number(r1)
      	ran_num = int(26.0*r1) + 65
      	random_char  = achar(ran_num)
	random_chars(i:i) = random_char
      end do
      print *, "Random string is ", random_chars

      ! Convert PET to character
      pet_char  = achar(my_pet + 48)
      ! Append to "PET"
      my_pet_char = "PET" // pet_char
      random_string = my_pet_char // random_chars

      !EX_UTest
      ! Test Log Write 
      write(failMsg, *) "Did not return ESMF_SUCCESS"
      ! Get date and time to compare later in the test
      call date_and_time(values=my_v)
      call ESMF_LogWrite(log=log2, msg=random_string,msgtype=ESMF_LOG_INFO,rc=rc)
      write(name, *) "Write to log file Test"
      call ESMF_Test((rc.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc = ", rc

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test LogFlush of log
      write(failMsg, *) "Did not return ESMF_SUCCESS"
      write(name, *) " LogFlush Test"
      call ESMF_LogFlush(log2, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)

      !------------------------------------------------------------------------
      !EX_UTest
      ! Test Log Close
      write(failMsg, *) "Did not return ESMF_SUCCESS"
      call ESMF_LogClose(log2, rc)
      write(name, *) "Close Log Test"
      call ESMF_Test((rc.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)
      print *, " rc = ", rc

      !------------------------------------------------------------------------
      ! Verify that the file can be opened with Fortran IO
      !EX_UTest
      write(name, *) "Open Log file with Fortran IO Test"
      write(failMsg, *) "open() returned failure"
      filename = my_pet_char // ".Log_Test_File_2"
      open (unit=1, file = filename, action = "read", &
            form = "formatted", iostat = input_status)
      call ESMF_Test((input_status.eq.0), name, failMsg, result, ESMF_SRCLINE)
      print *, " filename = ", filename
      print *, " input_status = ", input_status
      if (input_status.ne.0) goto 100 ! cannot continue test without open file

      !------------------------------------------------------------------------
      ! Verify that the correct string was written to the file
      !EX_UTest
      rc = ESMF_FAILURE
      do
          read (1, *, iostat = input_status) todays_date, todays_time, &
                                             msg_type, Pet_num, msg_string
          if (input_status < 0) then
              exit
          endif 
          if (msg_string.eq.random_string) then
              rc = ESMF_SUCCESS
              exit
          endif
      end do
      write(failMsg, *) "Did not return ESMF_SUCCESS"
      write(name, *) "Verify random string Test"
      call ESMF_Test((rc.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)
      print *, "rc = ", rc
      print *, "msg_string is ", msg_string

      !------------------------------------------------------------------------
      !EX_UTest
      ! Verify correct date and time was written to log file (within 1 second)
      write(failMsg, *) "Date and/or Time in file is wrong"
      write(name, *) "Verify Date and Time in Log File Test"
      read(todays_date, 10) log_v(1), log_v(2), log_v(3)
   10 format(i4,2i2)
      read(todays_time, 20) log_v(5), log_v(6), log_v(7), log_v(8)
   20 format(3i2,1x,i3)
      call ESMF_TimeSet(my_time, yy=my_v(1), mm=my_v(2), dd=my_v(3), &
                        h=my_v(5), m=my_v(6), s=my_v(7), ms=my_v(8), rc=rc)
      call ESMF_TimeSet(log_time, yy=log_v(1), mm=log_v(2), dd=log_v(3), &
                        h=log_v(5), m=log_v(6), s=log_v(7), ms=log_v(8), rc=rc)
      time_diff = log_time - my_time
      call ESMF_TimeIntervalSet(zero, s=0, rc=rc)
      call ESMF_TimeIntervalSet(one_sec, s=1, rc=rc)
      call ESMF_Test((time_diff.ge.zero .and. time_diff.le.one_sec), &
                      name, failMsg, result, ESMF_SRCLINE)
      print *, " my_time is "
      call ESMF_TimePrint(my_time, "string", rc)
      print *, " log_time is "
      call ESMF_TimePrint(log_time, "string", rc)

      !------------------------------------------------------------------------
      !EX_UTest
      ! Verify correct message type was written to log file
      write(failMsg, *) "Message type in file is wrong"
      write(name, *) "Verify Message Type in Log File Test"
      call ESMF_Test((msg_type.eq."INFO"), name, failMsg, result, ESMF_SRCLINE)

      !------------------------------------------------------------------------
      !EX_UTest
      ! Verify correct PET was written into log file
      write(failMsg, *) "PET number in file is wrong"
      write(name, *) "Verify PET number in Log File Test"
      call ESMF_Test((lge(my_pet_char,pet_num).and.lle(my_pet_char,pet_num)), &
        name, failMsg, result, ESMF_SRCLINE)
      print *, " My PET char is ", my_pet_char, "and I read: ", pet_num
      
      !------------------------------------------------------------------------
      !EX_UTest
      ! Test LogFlush of new unopened log
      write(failMsg, *) "Did not return ESMF_FAILURE"
      write(name, *) " LogFlush of unopened log Test"
      call ESMF_LogFlush(log2, rc=rc)
      call ESMF_Test((rc.eq.ESMF_FAILURE), name, failMsg, result, ESMF_SRCLINE)

#endif
100   continue
      call ESMF_TestEnd(result, ESMF_SRCLINE)


      end program ESMF_LogErrUTest
