/*!\file AutodiffDriversx
 * \brief: compute outputs from the AD mode,  using our dependents and independents, and drivers available in Adolc.
 */
#include <set>
#include "../../modules/modules.h"
#include "../../shared/shared.h"
#include "../../include/include.h"
#include "../../toolkits/toolkits.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/Numerics/adolc_edf.h"


void AutodiffDriversx(Elements* elements,Nodes* nodes,Vertices* vertices,Loads* loads,Materials* materials,Parameters* parameters,Results* results){


	/*diverse: */
	int         i;
	int         dummy;

	bool        isautodiff         = false;
	int         num_dependents;
	int         num_independents;
	char*       driver=NULL;

	/*state variables: */
	IssmDouble *axp                = NULL;
	double     *xp                 = NULL;

	/*AD mode on?: */
	parameters->FindParam(&isautodiff,AutodiffIsautodiffEnum);

	if(isautodiff){

		#ifdef _HAVE_ADOLC_

			if(VerboseAutodiff())_pprintLine_("   start AD driver");

			/*preliminary checks: */
			parameters->FindParam(&num_dependents,AutodiffNumDependentsEnum);
			parameters->FindParam(&num_independents,AutodiffNumIndependentsEnum);
			if(!(num_dependents*num_independents)) return;
			
			/*retrieve state variable: */
			parameters->FindParam(&axp,&dummy,AutodiffXpEnum);

			/* driver argument */
			xp=xNew<double>(num_independents);
			for(i=0;i<num_independents;i++){
				xp[i]=reCast<double,IssmDouble>(axp[i]);
			}

			/*get the EDF pointer:*/
			ext_diff_fct *anEDF_for_solverx_p=dynamic_cast<GenericParam<Adolc_edf> * >(parameters->FindParamObject(AdolcParamEnum))->GetParameterValue().myEDF_for_solverx_p;

			/*Branch according to AD driver: */
			parameters->FindParam(&driver,AutodiffDriverEnum);

			/* these are always needed regardless of the interpreter */
                        anEDF_for_solverx_p->dp_x=xNew<double>(anEDF_for_solverx_p->max_n);
                        anEDF_for_solverx_p->dp_y=xNew<double>(anEDF_for_solverx_p->max_m);

			if (strcmp(driver,"fos_forward")==0){
				
				int     anIndepIndex;
				double *tangentDir=NULL;
				double *jacTimesTangentDir=NULL;
				double *theOutput=NULL;
				
				/*retrieve direction index: */
				parameters->FindParam(&anIndepIndex,AutodiffFosForwardIndexEnum);
				
				if (anIndepIndex<0 || anIndepIndex>=num_independents) _error_("index value for AutodiffFosForwardIndexEnum should be in [0,num_independents-1]");

				tangentDir=xNewZeroInit<double>(num_independents);
				tangentDir[anIndepIndex]=1.0;
			
				jacTimesTangentDir=xNew<double>(num_dependents);
				theOutput=xNew<double>(num_dependents);
				
				/*set the forward method function pointer: */
				anEDF_for_solverx_p->fos_forward=EDF_fos_forward_for_solverx;

                                /*allocate the space for the parameters to invoke the EDF fos_forward:*/
				anEDF_for_solverx_p->dp_X=xNew<double>(anEDF_for_solverx_p->max_n);
				anEDF_for_solverx_p->dp_Y=xNew<double>(anEDF_for_solverx_p->max_m);

				/*call driver: */
				fos_forward(1,num_dependents,num_independents, 0, xp, tangentDir, theOutput, jacTimesTangentDir );

				/*add to results*/
				results->AddObject(new GenericExternalResult<IssmPDouble*>(results->Size()+1,AutodiffJacobianEnum,jacTimesTangentDir,num_dependents,1,1,0.0));
				
				/*free resources :*/
				xDelete(theOutput);
				xDelete(jacTimesTangentDir);
				xDelete(tangentDir);
			}
			else if ((strcmp(driver,"fov_forward")==0) || (strcmp(driver,"fov_forward_all")==0)){
				
				int* indepIndices=NULL;
				int tangentDirNum;
				int dummy;
				double **jacTimesSeed=NULL;
				double **seed=NULL;
				double *theOutput=NULL;
				std::set<unsigned int> anIndexSet;

					
				/*retrieve directions:*/
				if (strcmp(driver,"fov_forward_all")==0){
					tangentDirNum=num_independents;
					indepIndices=xNewZeroInit<int>(tangentDirNum);
					for(i=0;i<num_independents;i++)indepIndices[i]=1;
				}
				else{
					parameters->FindParam(&indepIndices,&tangentDirNum,&dummy,AutodiffFovForwardIndicesEnum);
				}

				/*Some checks: */
				if (tangentDirNum<1 || tangentDirNum>num_independents) _error_("tangentDirNum should be in [1,num_independents]");

				/* full Jacobian or Jacobian projection:*/
				jacTimesSeed=xNew<double>(num_dependents,tangentDirNum);
				
				/*set the forward method function pointers: */
				anEDF_for_solverx_p->fov_forward=EDF_fov_forward_for_solverx;
				// anEDF_for_solverx_p->fov_reverse=EDF_fov_reverse_for_solverx;

				/*allocate the space for the parameters to invoke EDF fov_forward:*/
				anEDF_for_solverx_p->dpp_X=xNew<double>(anEDF_for_solverx_p->max_n, tangentDirNum);
				anEDF_for_solverx_p->dpp_Y=xNew<double>(anEDF_for_solverx_p->max_m, tangentDirNum);

				/*seed matrix: */
				seed=xNewZeroInit<double>(num_independents,tangentDirNum);

				/*collect indices in a set to prevent accidental duplicates as long as we don't do compression:*/
				for (int i=0; i<tangentDirNum; ++i) {
					/* make sure the index is in range*/
					if (indepIndices[i]>num_independents) {
						_error_("indepIndices values must be in [0,num_independents-1]");
					}
					if (anIndexSet.find(indepIndices[i])!=anIndexSet.end()) {
						_error_("duplicate indepIndices values are not allowed until we implement Jacobian decompression");
					}
					anIndexSet.insert(indepIndices[i]);
					/* now populate the seed matrix from the set of independent indices;
					 * simple setup with a single 1.0 per column and at most a single 1.0 per row*/
					seed[indepIndices[i]][i]=1.0;
				}

				/*allocate output: */
				theOutput=xNew<double>(num_dependents);

				/*call driver: */
				fov_forward(1,num_dependents,num_independents, tangentDirNum, xp, seed, theOutput, jacTimesSeed );
				/*Free resources: */
				xDelete(theOutput);
				xDelete(indepIndices);
				xDelete(seed);

				/*add to results: */
				results->AddObject(new GenericExternalResult<IssmPDouble*>(results->Size()+1,AutodiffJacobianEnum,*jacTimesSeed,num_dependents*tangentDirNum,1,1,0.0));

				/*Free resources: */
				xDelete(jacTimesSeed);
				xDelete(indepIndices);
			}
			else if (strcmp(driver,"fos_reverse")==0) {

				int     aDepIndex=0;
				double *aWeightVector=NULL;
				double *weightVectorTimesJac=NULL;

				/*retrieve direction index: */
				parameters->FindParam(&aDepIndex,AutodiffFosReverseIndexEnum);

				if (aDepIndex<0 || aDepIndex>=num_dependents) _error_("index value for AutodiffFosReverseIndexEnum should be in [0,num_dependents-1]");

				aWeightVector=xNewZeroInit<double>(num_dependents);
				aWeightVector[aDepIndex]=1.0;

				weightVectorTimesJac=xNew<double>(num_independents);

				/*set the forward method function pointer: */
				anEDF_for_solverx_p->fos_reverse=EDF_fos_reverse_for_solverx;

				/*allocate the space for the parameters to invoke the EDF fos_reverse :*/
				anEDF_for_solverx_p->dp_U=xNew<double>(anEDF_for_solverx_p->max_m);
				anEDF_for_solverx_p->dp_Z=xNew<double>(anEDF_for_solverx_p->max_n);

				/*call driver: */
				fos_reverse(1,num_dependents,num_independents, aWeightVector, weightVectorTimesJac );

				/*add to results*/
				results->AddObject(new GenericExternalResult<IssmPDouble*>(results->Size()+1,AutodiffJacobianEnum,weightVectorTimesJac,num_independents,1,1,0.0));

				/*free resources :*/
				xDelete(weightVectorTimesJac);
				xDelete(aWeightVector);
			}
			else if (strcmp(driver,"fov_reverse")==0) {
				/*allocate the space for the parameters to invoke the EDF fos_reverse :*/
				anEDF_for_solverx_p->dpp_U=xNew<double>(num_dependents,anEDF_for_solverx_p->max_m);
				anEDF_for_solverx_p->dpp_Z=xNew<double>(num_dependents,anEDF_for_solverx_p->max_n);
				_error_("driver: " << driver << " not yet supported!");
			}
			else _error_("driver: " << driver << " not yet supported!");

			/* delete the allocated space for the parameters:*/
			xDelete(anEDF_for_solverx_p->dp_x);
			xDelete(anEDF_for_solverx_p->dp_X);
			xDelete(anEDF_for_solverx_p->dpp_X);
			xDelete(anEDF_for_solverx_p->dp_y);
			xDelete(anEDF_for_solverx_p->dp_Y);
			xDelete(anEDF_for_solverx_p->dpp_Y);
			xDelete(anEDF_for_solverx_p->dp_U);
			xDelete(anEDF_for_solverx_p->dpp_U);
			xDelete(anEDF_for_solverx_p->dp_Z);
			xDelete(anEDF_for_solverx_p->dpp_Z);

			/*Free resources: */
			xDelete(xp);
			xDelete(axp); 
			
			if(VerboseAutodiff())_pprintLine_("   end AD driver");

		#else
			_error_("Should not be requesting AD drivers when an AD library is not available!");
		#endif
	}
}
