function md=YamsCall(md,field,hmin,hmax,gradation,epsilon),
%YAMSCALL - call yams
%
%   build a metric using the Hessian of the given field
%   call Yams and the output mesh is plugged onto the model
%   -hmin = minimum edge length (m)
%   -hmax = maximum edge length (m)
%   -gradation = maximum edge length gradation between 2 elements
%   -epsilon = average error on each element (m/yr)
%
%   Usage:
%      md=YamsCall(md,field,hmin,hmax,gradation,epsilon);
%
%   Example:
%      md=YamsCall(md,md.vel_obs,1500,10^8,1.3,0.9);

global ISSM_DIR

%2d geometric parameter (do not change)
scale=2/9; 

%Compute Hessian
t1=clock; fprintf('%s','      computing Hessian...');
hessian=ComputeHessian(md.elements,md.x,md.y,field,'node');
t2=clock;fprintf('%s\n',[' done (' num2str(etime(t2,t1)) ' seconds)']);

%Compute metric
t1=clock; fprintf('%s','      computing metric...');
if length(md.gridonwater)==md.numberofgrids,
	pos=find(md.gridonwater);
else
	pos=[];
end
metric=ComputeMetric(hessian,scale,epsilon,hmin,hmax,pos);
t2=clock;fprintf('%s\n',[' done (' num2str(etime(t2,t1)) ' seconds)']);

%write files
t1=clock; fprintf('%s','      writing initial mesh files...');
save -ascii carre0.met  metric

fid=fopen('carre0.mesh','w');

%initialiation
fprintf(fid,'\n%s\n%i\n','MeshVersionFormatted',1);

%dimension
fprintf(fid,'\n%s\n%i\n','Dimension',2);

%Vertices
fprintf(fid,'\n%s\n%i\n\n','Vertices',md.numberofgrids);
fprintf(fid,'%8g %8g %i\n',[md.x md.y zeros(md.numberofgrids,1)]');

%Triangles
fprintf(fid,'\n\n%s\n%i\n\n','Triangles',md.numberofelements);
fprintf(fid,'%i %i %i %i\n',[md.elements zeros(md.numberofelements,1)]');

%close
fclose(fid);
t2=clock;fprintf('%s\n',[' done (' num2str(etime(t2,t1)) ' seconds)']);

%call yams
fprintf('%s\n','      call Yams...');
system(['yams2-linux -O 1 -v -0 -ecp -hgrad ' num2str(gradation)  ' carre0 carre1']);

%plug new mesh
t1=clock; fprintf('\n%s','      reading final mesh files...');
Tria=load('carre1.tria');
Coor=load('carre1.coor');
md.x=Coor(:,1);
md.y=Coor(:,2);
md.z=zeros(size(Coor,1),1);
md.elements=Tria;

md.numberofgrids=size(Coor,1);
md.numberofelements=size(Tria,1);
t2=clock;fprintf('%s\n\n',[' done (' num2str(etime(t2,t1)) ' seconds)']);

%clean up:
system('rm carre0.mesh carre0.met carre1.tria carre1.coor carre1.meshb');
